// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPArticleStockFieldSet.view.StockGrid', {

    extend: 'Ext.grid.Panel',
    alias: 'widget.viison_pickware_erp_article_stock_field_set-stock_grid',
    cls: 'viison_pickware_erp_article_stock_field_set-stock_grid viison-common--grid has--no-vertical-header-dividers has--cell-size-large',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_article_stock_field_set/main',

    style: {
        borderTop: 'none',
        borderBottom: 'none',
    },

    viewConfig: {
        stripeRows: false,
        trackOver: false,
        selectedItemCls: '',
        selectedCellCls: '',
        getRowClass: function (record) {
            if (record.get('id') === 'reservedStock') {
                return 'is--last-before-sum-row';
            }

            if (record.get('id') === 'availableStock') {
                return 'is--sum-row';
            }

            return '';
        },
    },

    selType: 'cellmodel',

    /**
     * @Override
     */
    afterRender: function () {
        this.callParent(arguments);

        // Cancel cell editing, when the content of the window containing the grid
        // is scrolled. This is necessary, since the cell editor will not scroll
        // along with the content of the window due to the placement strategy of
        // the cell editing plugin, which uses absolute positioning relative to
        // the browser's top left corner.
        var cellEditing = this.getPlugin('cellediting');
        this.up('window').getEl().on('mousewheel', cellEditing.cancelEdit, cellEditing);
    },

    /**
     * @Override
     */
    initComponent: function () {
        this.columns = this.getColumns();
        this.plugins = this.getPlugins();
        this.store = this.createStore();

        this.callParent(arguments);
    },

    /**
     * Creates the grid columns
     *
     * @return [array] grid columns
     */
    getColumns: function () {
        return [
            {
                header: this.getViisonSnippet('title'),
                dataIndex: 'label',
                flex: 1,
                sortable: false,
                renderer: function (value, metaData, record) {
                    if (!record.get('enabled')) {
                        metaData.tdCls = 'is--disabled';

                        return value;
                    }

                    metaData.tdAttr = 'data-qtip="' + record.get('helptext') + '"';

                    return '<span class="has--helptext">' + value + '</span>';
                },
            }, {
                header: '',
                width: 70,
                dataIndex: 'value',
                align: 'right',
                tdCls: 'is--font-size-large',
                sortable: false,
                renderer: function (value, metaData, record) {
                    if (!record.get('enabled')) {
                        metaData.tdCls = 'is--disabled';

                        return '';
                    }

                    // Use custom styled negative sign for reservedStock (if value is in fact negative)
                    if (record.get('id') === 'reservedStock' && value < 0) {
                        return '<span class="is--reserved-stock-sign">-</span>' + Math.abs(value);
                    }

                    return value;
                },
                getEditor: function (record) {
                    // Return false (= no editor) when record is in fact not editable
                    if (!record.get('editingEnabled')) {
                        return false;
                    }

                    return Ext.create('Ext.grid.CellEditor', {
                        onFieldBlur: Ext.emptyFn, // Prevent cell editor from automatically committing changes, when the user clicks outside the input field
                        cancelOnEsc: true,
                        field: {
                            xtype: 'numberfield',
                            cls: 'is--viison-pickware-erp-celledit-large',
                            allowBlank: false,
                            listeners: {
                                scope: this,
                                change: function (field, newValue) {
                                    if (this.articleDetail.get('pickwareStockManagementDisabled')) {
                                        // In this case, neither physical stock nor reserved stock make any sense, so don't update them
                                        return;
                                    }
                                    if (this.articleDetail.phantom) {
                                        // Article creation, hence update the displayed physical stock, because it will be set
                                        // to the available stock when saving it
                                        this.getStore().getById('physicalStock').set('value', newValue);
                                    } else {
                                        this.getStore().getById('reservedStock').set('value', this.articleDetail.get('pickwarePhysicalStockForSale') - newValue);
                                    }
                                },
                            },
                        },
                    });
                }.bind(this),
            }, {
                xtype: 'actioncolumn',
                width: 58,
                align: 'right',
                tdCls: 'is--actioncolumn',
                sortable: false,
                items: [
                    {
                        iconCls: 'sprite-pencil',
                        tooltip: this.getViisonSnippet('stock_grid/available_stock/start_editing/tooltip'),
                        scope: this,
                        handler: function (view, rowIndex) {
                            var record = this.getStore().getAt(rowIndex);

                            if (record.get('editingEnabled')) {
                                this.getPlugin('cellediting').startEdit(record, 1);
                            }
                        },
                        getClass: function (value, meta, record) {
                            if (!record.get('editable')) {
                                return 'x-hidden';
                            }

                            if (!record.get('editingEnabled')) {
                                return 'is--disabled';
                            }

                            return 'is--start-editing-button';
                        },
                    }, {
                        iconCls: 'sprite-tick',
                        tooltip: this.getViisonSnippet('stock_grid/available_stock/confirm_changes/tooltip'),
                        scope: this,
                        handler: function () {
                            this.getPlugin('cellediting').completeEdit();
                        },
                        getClass: function (value, meta, record) {
                            if (record.get('editable')) {
                                return 'x-hidden is--edit-button';
                            }

                            return 'x-hidden';
                        },
                    }, {
                        iconCls: 'sprite-cross',
                        tooltip: this.getViisonSnippet('stock_grid/available_stock/reject_changes/tooltip'),
                        scope: this,
                        handler: function () {
                            this.getPlugin('cellediting').cancelEdit();
                        },
                        getClass: function (value, meta, record) {
                            if (record.get('editable')) {
                                return 'x-hidden is--edit-button';
                            }

                            return 'x-hidden';
                        },
                    },
                ],
            },
        ];
    },

    /**
     * Creates the grid plugins
     *
     * @return [array] grid plugins
     */
    getPlugins: function () {
        return [
            Ext.create('Shopware.apps.ViisonPickwareERPArticleStockFieldSet.ux.grid.plugin.CellEditing', {
                pluginId: 'cellediting',
                clicksToEdit: 2,
                listeners: {
                    scope: this,
                    beforeEdit: function () {
                        if (!this.articleDetail || this.articleDetail.phantom) {
                            // Disable cell editing, when article is not saved yet
                            return false;
                        }

                        this.showEditButtons();

                        return true;
                    },
                    edit: this.onEdit,
                    canceledit: function () {
                        this.hideEditButtons();
                        this.reconfigure(this.createStore());
                    },
                },
            }),
        ];
    },

    /**
     * Saves the available stock value. If the article is stock managed an alert is displayed asking
     * the user to confirm the update.
     */
    onEdit: function () {
        this.hideEditButtons();

        var oldAvailableInstock = this.articleDetail.get('inStock');
        var newAvailableStock = this.getStore().getById('availableStock').get('value');
        // Check whether the value was changed
        if (newAvailableStock === oldAvailableInstock) {
            return;
        }

        var fieldSet = this.up('viison_pickware_erp_article_stock_field_set-stock_field_set');

        if (this.articleDetail.get('pickwareStockManagementDisabled')) {
            // Don't ask for confirmation if article is not stock managed, since there are no stock entries to
            // corrupt in this case.
            this.fireEvent(
                'viisonPickwareERPSaveAvailableStock',
                fieldSet,
                this.articleDetail,
                newAvailableStock,
                oldAvailableInstock
            );
        } else {
            // Ask the user for confirmation before 'saving' the changes
            Ext.Msg.show({
                title: this.getViisonSnippet('alert/available_stock/confirm_editing/title'),
                msg: Ext.String.format(
                    this.getViisonSnippet('alert/available_stock/confirm_editing/message'),
                    newAvailableStock
                ),
                buttons: Ext.Msg.YESNO,
                icon: Ext.Msg.WARNING,
                scope: this,
                callback: function (buttonId) {
                    if (buttonId !== 'yes') {
                        this.reconfigure(this.createStore());

                        return;
                    }

                    this.fireEvent(
                        'viisonPickwareERPSaveAvailableStock',
                        fieldSet,
                        this.articleDetail,
                        newAvailableStock,
                        oldAvailableInstock
                    );
                },
            });
        }
    },

    /**
     * Sets the article detail an reconfigures the grid based on the article data.
     *
     * @param articleDetail
     */
    setArticleDetail: function (articleDetail) {
        this.articleDetail = articleDetail;
        this.reconfigure(this.createStore());
    },

    /**
     * Cancels the editing process of the available stock value.
     */
    cancelEditing: function () {
        this.getPlugin('cellediting').cancelEdit();
    },

    /**
     * Creates the grid store
     *
     * @returns {Ext.data.ArrayStore}
     */
    createStore: function () {
        return Ext.create('Ext.data.Store', {
            idIndex: 0,
            fields: [
                { name: 'id', type: 'string' },
                { name: 'enabled', type: 'boolean' },
                { name: 'editable', type: 'boolean' },
                { name: 'editingEnabled', type: 'boolean' },
                { name: 'label', type: 'string' },
                { name: 'helptext', type: 'string' },
                { name: 'value', type: 'int' },
            ],
            data: [
                {
                    id: 'physicalStock',
                    enabled: this.isActiveMode(),
                    editable: false,
                    editingEnabled: false,
                    label: this.getViisonSnippet('field/physical_stock/label'),
                    helptext: this.getViisonSnippet('field/physical_stock/help_text'),
                    value: this.articleDetail ? this.articleDetail.get('pickwarePhysicalStockForSale') : 0,
                }, {
                    id: 'reservedStock',
                    enabled: this.isActiveMode(),
                    editable: false,
                    editingEnabled: false,
                    label: this.getViisonSnippet('field/reserved_stock/label'),
                    helptext: this.getViisonSnippet('field/reserved_stock/help_text'),
                    value: this.articleDetail ? this.articleDetail.get('pickwareReservedStock') : 0,
                }, {
                    id: 'availableStock',
                    enabled: true,
                    editable: true,
                    editingEnabled: this.articleDetail && !this.articleDetail.phantom,
                    label: this.getViisonSnippet('field/available_stock/label'),
                    helptext: this.getViisonSnippet('field/available_stock/help_text'),
                    value: this.articleDetail ? this.articleDetail.get('inStock') : 0,
                },
            ],
        });
    },

    /**
     * Returns whether or not the grid is in active mode, whereby the grid is said to be in active mode
     * if the article is loaded and stock managed.
     *
     * @returns {boolean}
     */
    isActiveMode: function () {
        return this.articleDetail && !this.articleDetail.get('pickwareStockManagementDisabled');
    },

    /**
     * Displays the edit buttons of the available stock cell.
     */
    showEditButtons: function () {
        var editButtons = Ext.DomQuery.select('.is--edit-button', this.getEl().dom);
        Ext.Array.each(editButtons, function (editButton) {
            Ext.get(editButton).removeCls('x-hidden');
        });
        var startEditingButton = Ext.DomQuery.select('.is--start-editing-button', this.getEl().dom);
        Ext.get(startEditingButton).addCls('x-hidden');
    },

    /**
     * Hides the edit buttons of the available stock cell.
     */
    hideEditButtons: function () {
        var editButtons = Ext.DomQuery.select('.is--edit-button', this.getEl().dom);
        Ext.Array.each(editButtons, function (editButton) {
            Ext.get(editButton).addCls('x-hidden');
        });
        var startEditingButton = Ext.DomQuery.select('.is--start-editing-button', this.getEl().dom);
        Ext.get(startEditingButton).removeCls('x-hidden');
    },

});
