// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPArticleSupplierAssignment.controller.Main', {

    extend: 'Ext.app.Controller',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_article_supplier_assignment/main',

    refs: [
        { ref: 'mainTabPanel', selector: 'article-detail-window tabpanel' },
        { ref: 'cancelButton', selector: 'article-detail-window button[name=cancel-button]' },
        { ref: 'saveButton', selector: 'article-detail-window button[name=save-article-button]' },
        { ref: 'supplierAssignmentPanel', selector: 'viison_pickware_erp_article_supplier_assignment-detail-supplier_assignment' },
        { ref: 'copyAssignmentButton', selector: 'viison_pickware_erp_article_supplier_assignment-detail-supplier_assignment button[action=copyAssignmentToAllVariants]' },
    ],

    /**
     * @Override
     */
    init: function () {
        this.control({
            'viison_pickware_erp_article_supplier_assignment-detail-supplier_assignment': {
                moveItemsFromLeftToRight: this.onMoveSuppliersFromLeftToRight,
                moveItemsFromLeftToRightViaDragDrop: this.onMoveSuppliersFromLeftToRight,
                moveItemsFromRightToLeft: this.onMoveSuppliersFromRightToLeft,
                moveItemsFromRightToLeftViaDragDrop: this.onMoveSuppliersFromRightToLeft,
                supplierRowEdited: this.onSupplierRowEdited,
                leftPanelSearchFieldChanged: this.onSearchFieldChanged,
                rightPanelSearchFieldChanged: this.onSearchFieldChanged,
                copyAssignmentToAllVariants: this.onCopyAssignmentToAllVariants,
                afterrender: this.onAfterRender,
            },
            'viison_pickware_erp_article_supplier_assignment-detail-supplier_assignment viison_common_variant_combo_box-combo_box': {
                variantChanged: this.onVariantChanged,
            },
        });

        // Create the two stores for the assignment
        this.leftStore = Ext.create('Shopware.apps.ViisonPickwareERPArticleSupplierAssignment.store.Supplier', {
            autoLoad: false,
        });
        this.rightStore = Ext.create('Shopware.apps.ViisonPickwareERPArticleSupplierAssignment.store.AssignedSupplier', {
            autoLoad: false,
        });

        this.callParent(arguments);
    },

    /**
     * Creates and returns a new supplier assignment panel.
     *
     * @returns {Shopware.apps.ViisonPickwareERPArticleSupplierAssignment.view.detail.SupplierAssignment}
     */
    createAssignmentTab: function () {
        var tab = Ext.create('Shopware.apps.ViisonPickwareERPArticleSupplierAssignment.view.detail.SupplierAssignment', {
            leftStore: this.leftStore,
            rightStore: this.rightStore,
        });

        this.subApplication.on('ProductModule:storesChanged', this.articleChanged, tab);

        return tab;
    },

    /**
     * This method gets called when the user switches the article of the detail window by clicking on an article in
     * the article list while the split view mode is active.
     *
     * @param Shopware.apps.Article.model.Article article
     */
    articleChanged: function (article) {
        this.down('viison_common_variant_combo_box-combo_box').setArticle(article);
    },

    setArticle: function (article) {
        this.article = article;
    },

    /**
     * Ensures that the data isn't loaded before the grid is actually rendered. This ensures that the empty text message
     * will always be displayed if necessary, what wouldn't be the case if the store is loaded before the grid is
     * rendered.
     */
    onAfterRender: function () {
        setTimeout(function () {
            // Pass the article to the variant selection to load its data
            var variantSelection = this.getSupplierAssignmentPanel().down('viison_common_variant_combo_box-combo_box');
            variantSelection.getStore().on('load', function (store, records) {
                // Workaround a shopware bug where an article with no generated variants returns an empty list instead of
                // the main detail, hence just add the main detail to the store here
                if (records.length !== 0) {
                    return;
                }

                store.loadData([
                    this.article.getMainDetail().first(),
                ], false);
            }, this, { single: true });
            variantSelection.setArticle(this.article);
        }.bind(this), 50);
    },

    /**
     * Sets the ID of the article detail that is being edited to each of
     * the newly assigned records and syncs the changes.
     *
     * @param Shopware.apps.ViisonPickwareERPArticleSupplierAssignment.view.detail.SupplierAssignment panel
     * @param Shopware.apps.ViisonPickwareERPArticleSupplierAssignment.model.Supplier[] records
     */
    onMoveSuppliersFromLeftToRight: function (panel, records) {
        // Update assignments
        var variantSelection = panel.down('viison_common_variant_combo_box-combo_box');
        var selectedArticleDetail = variantSelection.getSelectedVariant();
        Ext.each(records, function (assignedRecord) {
            assignedRecord.set('articleDetailId', ((selectedArticleDetail) ? selectedArticleDetail.get('id') : null));
            // Use the default purchase price as the purchase price of the assignment
            assignedRecord.set(
                'purchasePrice',
                (selectedArticleDetail) ? selectedArticleDetail.get('purchasePrice') : 0
            );

            // Make the supplier default, if no other supplier is assigned
            assignedRecord.set('defaultSupplier', (panel.rightStore.count() === 1));
            assignedRecord.commit();
            assignedRecord.phantom = true;
        }, this);
        this.updateCopyAssignmentButton();

        // Save the changes
        this.syncAssignedSuppliers(this.getViisonSnippet('notification/error/message/add'), function () {
            panel.leftStore.loadPage(panel.leftStore.currentPage);
            panel.rightStore.loadPage(panel.rightStore.currentPage);
        });
    },

    /**
     * Syncs the changes made to the right (assigned) store before reloading both
     * the left (available) and right stores.
     *
     * @param Shopware.apps.ViisonPickwareERPArticleSupplierAssignment.view.detail.SupplierAssignment panel
     */
    onMoveSuppliersFromRightToLeft: function (panel) {
        // Save the changes
        this.updateCopyAssignmentButton();
        this.syncAssignedSuppliers(this.getViisonSnippet('notification/error/message/delete'), function () {
            panel.leftStore.loadPage(panel.leftStore.currentPage);
            panel.rightStore.loadPage(panel.rightStore.currentPage);
        });
    },

    /**
     * Syncs the changes to the server. If the edited record is set to be default supplier
     * all other supplier are updated to not be the default supplier. This ensures that there
     * is only one default supplier at a time.
     *
     * @param Shopware.apps.ViisonPickwareERPArticleSupplierAssignment.view.detail.SupplierAssignment panel
     * @param Shopware.apps.ViisonPickwareERPArticleSupplierAssignment.model.Supplier record
     */
    onSupplierRowEdited: function (panel, record) {
        // Make sure to have only one default supplier
        if (record.get('defaultSupplier')) {
            panel.rightStore.each(function (supplier) {
                if (supplier.get('id') !== record.get('id')) {
                    supplier.set('defaultSupplier', false);
                }
            });
        }

        // Save the changes
        record.setDirty();
        this.syncAssignedSuppliers(this.getViisonSnippet('notification/error/message/update'));
    },

    /**
     * Sets the given query as an extra parameter to the given store's proxy
     * and loads the first page of the store.
     *
     * @param string query
     * @param Ext.data.Store store
     */
    onSearchFieldChanged: function (query, store) {
        store.getProxy().extraParams.query = query;
        store.loadPage(1);
    },

    /**
     * If given an article detail, the respective supplier assignments are loaded,
     * otherwise both stores and their filters are cleared.
     *
     * @param Shopware.apps.Article.model.Detail|null|false articleDetail
     */
    onVariantChanged: function (articleDetail) {
        if (articleDetail) {
            this.loadAssignedSuppliers(articleDetail);
        } else {
            // Clear both stores
            this.leftStore.clearFilter(true);
            this.rightStore.clearFilter(true);
            this.leftStore.removeAll();
            this.rightStore.removeAll();
        }
    },

    /**
     * Asks the user for a confirmation before transferring the assignments of the currently
     * selected article variant to all other variants of the same article.
     *
     * @param Shopware.apps.ViisonPickwareERPArticleSupplierAssignment.view.detail.SupplierAssignment panel
     */
    onCopyAssignmentToAllVariants: function (panel) {
        // Present a confirm alert before copying the assignments
        Ext.Msg.confirm(
            this.getViisonSnippet('alert/title'),
            this.getViisonSnippet('alert/message/copy_assignments'),
            function (buttonId) {
                if (buttonId !== 'yes') {
                    return;
                }

                // Copy all assignments
                var variantSelection = panel.down('viison_common_variant_combo_box-combo_box');
                var selectedArticleDetail = variantSelection.getSelectedVariant();
                this.getSupplierAssignmentPanel().setLoading(true);
                Ext.Ajax.request({
                    url: ViisonCommonApp.assembleBackendUrl(
                        'ViisonPickwareERPArticleSupplierAssignment/transferSupplierArticleDetails'
                    ),
                    params: {
                        articleDetailId: (selectedArticleDetail) ? selectedArticleDetail.get('id') : null,
                    },
                    callback: function (options, success) {
                        this.getSupplierAssignmentPanel().setLoading(false);
                        if (!success) {
                            Shopware.Notification.createGrowlMessage(
                                this.getViisonSnippet('notification/error/title'),
                                this.getViisonSnippet('notification/error/message/copy_assignments'),
                                'ViisonPickwareERPArticleSupplierAssignment'
                            );
                        }
                    },
                    scope: this,
                });
            },
            this
        );
    },

    /**
     * Updates both the articleDetailIdBlacklist filter on the available (left) store
     * as well as the articleDetailId filter on the assigned (right) store using the
     * ID of the given article detail. Finally loads the first page of both stores.
     *
     * @param Shopware.apps.Article.model.Detail articleDetail
     */
    loadAssignedSuppliers: function (articleDetail) {
        // Update the filters on both supplier stores
        this.leftStore.filters.add('articleDetailBlacklist', Ext.create('Ext.util.Filter', {
            property: 'articleDetailIdBlacklist',
            expression: '!=',
            value: articleDetail.get('id'),
        }));
        this.rightStore.filters.add('articleDetailId', Ext.create('Ext.util.Filter', {
            property: 'supplierMapping.articleDetailId',
            value: articleDetail.get('id'),
        }));

        // Reload both stores
        this.leftStore.loadPage(1);
        this.rightStore.loadPage(1, {
            callback: function () {
                this.updateCopyAssignmentButton();
            },
            scope: this,
        });
    },

    /**
     * Enables/disables the 'copy assignment' button based on the currently selected
     * suppliers. That is, if at least one supplier is assigned, the button is enabled,
     * otherwise it is disabled.
     */
    updateCopyAssignmentButton: function () {
        var disabled = this.getSupplierAssignmentPanel().rightStore.count() === 0;
        this.getCopyAssignmentButton().setDisabled(disabled);
    },

    /**
     * Synchronizes the assignment store and, in case of failure, shows a growl notification.
     *
     * @param string errorMessage
     * @param function callback (optional)
     */
    syncAssignedSuppliers: function (errorMessage, callback) {
        // Check the store for any changes
        var numberOfNewRecords = this.rightStore.getNewRecords().length;
        var numberOfModifiedRecords = this.rightStore.getModifiedRecords().length;
        var numberOfRemovedRecords = this.rightStore.getRemovedRecords().length;
        var numChanges = numberOfNewRecords + numberOfModifiedRecords + numberOfRemovedRecords;
        if (numChanges === 0) {
            if (Ext.isFunction(callback)) {
                callback(true);
            }

            return;
        }

        // Sync the store
        this.getSupplierAssignmentPanel().rightPanel.setLoading(true);
        this.rightStore.sync({
            scope: this,
            success: function () {
                this.getSupplierAssignmentPanel().rightPanel.setLoading(false);
                if (Ext.isFunction(callback)) {
                    callback(true);
                }
            },
            failure: function () {
                this.getSupplierAssignmentPanel().rightPanel.setLoading(false);
                // Revert any changes like added, updated or removed records
                this.rightStore.rejectChanges();
                // Display an error notification
                var message = errorMessage || '';
                Shopware.Notification.createGrowlMessage(
                    this.getViisonSnippet('notification/error/title'),
                    message,
                    'ViisonPickwareERPArticleSupplierAssignment'
                );
                if (Ext.isFunction(callback)) {
                    callback(false);
                }
            },
        });
    },

});
