// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

/**
 * Overrides the default readRecords implementation to flatten the data,
 * because the actual supplier data is nested in an 'supplier' object,
 * while the supplier article detail assignment data is toplevel.
 */
Ext.define('Shopware.apps.ViisonPickwareERPArticleSupplierAssignment.reader.AssignedSupplierJson', {

    extend: 'Ext.data.reader.Json',
    type: 'json',
    root: 'data',
    totalProperty: 'total',

    /**
     * @Override
     */
    readRecords: function (data) {
        if (data && data.data) {
            var newData = [];
            Ext.each(data.data, function (item) {
                var newItem = item.supplier;
                newItem.articleDetailId = item.articleDetailId;
                newItem.defaultSupplier = item.defaultSupplier;
                newItem.purchasePrice = item.purchasePrice;
                newItem.packingUnit = item.packingUnit;
                newItem.minimumOrderAmount = item.minimumOrderAmount;
                newItem.supplierArticleNumber = item.supplierArticleNumber;
                newData.push(newItem);
            });
            data.data = newData;
        }

        return this.callParent([data]);
    },

});

/**
 * Overrides the default getRecordData implementation to only send information about
 * the supplier article details assignment, since all supplier information is static.
 */
Ext.define('Shopware.apps.ViisonPickwareERPArticleSupplierAssignment.writer.AssignedSupplierJson', {

    extend: 'Ext.data.writer.Json',
    type: 'json',
    root: 'data',
    allowSingle: false,

    /**
     * @Override
     */
    getRecordData: function (record) {
        return {
            supplierId: record.get('id'),
            articleDetailId: record.get('articleDetailId'),
            defaultSupplier: record.get('defaultSupplier'),
            purchasePrice: record.get('purchasePrice'),
            packingUnit: record.get('packingUnit') || 1,
            minimumOrderAmount: record.get('minimumOrderAmount'),
            supplierArticleNumber: record.get('supplierArticleNumber'),
        };
    },

});

/**
 * Although this store extends the customised version of the Supplier store, it communicates
 * with the ViisonPickwareERPSupplierManagement and reads/writes supplier article detail assignments.
 * However, these assignments are locally handled as Supplier model instances. This allows
 * us to use drag'n'drop from a Supplier grid to one that actually holds supplier article
 * detail assignments. The communication with the different API is possible by using the
 * AssignedSupplierJson readers and writers, which 'translate' between the data format of
 * supplier article detail assignments and (customised) supplier.
 */
Ext.define('Shopware.apps.ViisonPickwareERPArticleSupplierAssignment.store.AssignedSupplier', {

    extend: 'Shopware.apps.ViisonPickwareERPArticleSupplierAssignment.store.Supplier',
    sorters: [
        { property: 'name' },
    ],
    proxy: {
        type: 'ajax',
        batchActions: true,
        api: {
            read: ViisonCommonApp.assembleBackendUrl(
                'ViisonPickwareERPSupplierManagement/getSupplierArticleDetailList'
            ),
            create: ViisonCommonApp.assembleBackendUrl(
                'ViisonPickwareERPSupplierManagement/createSupplierArticleDetails'
            ),
            update: ViisonCommonApp.assembleBackendUrl(
                'ViisonPickwareERPSupplierManagement/updateSupplierArticleDetails'
            ),
            destroy: ViisonCommonApp.assembleBackendUrl(
                'ViisonPickwareERPSupplierManagement/deleteSupplierArticleDetails'
            ),
        },
        reader: Ext.create('Shopware.apps.ViisonPickwareERPArticleSupplierAssignment.reader.AssignedSupplierJson', {}),
        writer: Ext.create('Shopware.apps.ViisonPickwareERPArticleSupplierAssignment.writer.AssignedSupplierJson', {}),
    },

});
