// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPArticleSupplierOrderList.controller.Main', {

    extend: 'Ext.app.Controller',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_article_supplier_order_list/main',

    selectedWarehouseIds: [],
    selectedArticleDetailId: null,

    refs: [
        { ref: 'supplierOrderList', selector: 'viison_pickware_erp_article_supplier_order_list-detail-supplier_order_list' },
    ],

    /**
     * @Override
     */
    init: function () {
        this.control({
            'viison_pickware_erp_article_supplier_order_list-detail-supplier_order_list': {
                afterrender: this.onOrdersListAfterRender,
            },
            'viison_pickware_erp_article_supplier_order_list-detail-supplier_order_list viison_common_variant_combo_box-combo_box': {
                variantChanged: this.onVariantChanged,
            },
            'viison_pickware_erp_article_supplier_order_list-detail-supplier_order_list viison_pickware_erp_supplier_common-warehouse_multiselection-combo-box': {
                selectedWarehouseIdsChanged: this.onSelectedWarehouseIdsChanged,
            },
        });

        // Create supplier orders and target warehouse stores
        this.supplierOrderStore = this.getStore('Shopware.apps.ViisonPickwareERPArticleSupplierOrderList.store.Order');
        this.targetWarehouseStore = Ext.create('Shopware.apps.ViisonPickwareERPWarehouseManagement.store.Warehouse', {
            autoLoad: true,
        });

        this.callParent(arguments);
    },

    /**
     * Creates and returns a new supplier orders list panel.
     *
     * @return Shopware.apps.ViisonPickwareERPArticleSupplierOrderList.view.detail.SupplierOrderList
     */
    createOrderListTab: function () {
        var orderStatusStore = this.getStore('Shopware.apps.ViisonPickwareERPSupplierOrders.store.order.Status');
        var tab = Ext.create('Shopware.apps.ViisonPickwareERPArticleSupplierOrderList.view.detail.SupplierOrderList', {
            store: this.supplierOrderStore,
            orderStatusStore: orderStatusStore,
            totalIncomingStock: 0,
            targetWarehouseStore: this.targetWarehouseStore,
        });

        // Repaint grid view after order status store is loaded
        orderStatusStore.on('load', function () {
            tab.reconfigure(this.supplierOrderStore);
        }, this);

        this.subApplication.on('ProductModule:storesChanged', this.articleChanged, tab);

        // Add listener on the store to update the displayed incoming stock
        this.supplierOrderStore.on('load', function (store) {
            this.totalIncomingStock = store.getProxy().getReader().rawData.totalIncomingStock;
            if (tab.totalIncomingStockLabel) {
                tab.totalIncomingStockLabel.update({
                    totalIncomingStock: this.totalIncomingStock,
                });
            }
        }, this);

        return tab;
    },

    /**
     * Ensures that the data isn't loaded before the grid is actually rendered.
     * This ensures that the empty text message will always be displayed if necessary,
     * what wouldn't be the case if the store is loaded before the grid is rendered.
     */
    onOrdersListAfterRender: function () {
        setTimeout(function () {
            // Pass the article to the variant selection to load its data
            var variantSelection = this.getSupplierOrderList().down('viison_common_variant_combo_box-combo_box');
            variantSelection.setArticle(this.article);
        }.bind(this), 50);
    },

    /**
     * If given an article detail, the respective supplier orders are loaded,
     * otherwise the orders store and its filters are cleared.
     *
     * @param Shopware.apps.Article.model.Detail|null|false articleDetail
     * @param Shopware.apps.ViisonCommonVariantComboBox.view.ComboBox combobox
     */
    onVariantChanged: function (articleDetail, combobox) {
        if (articleDetail) {
            this.selectedArticleDetailId = articleDetail.get('id');
            this.loadSupplierOrders(combobox);
        } else {
            // Clear store
            this.supplierOrderStore.clearFilter(true);
            this.supplierOrderStore.removeAll();
        }
    },

    /**
     * Reload the supplier order store with the given target warehouseId filter.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierCommon.view.WarehouseMultiselectionComboBox combobox
     * @param array warehouseIds
     */
    onSelectedWarehouseIdsChanged: function (combobox, warehouseIds) {
        this.selectedWarehouseIds = warehouseIds;
        this.loadSupplierOrders(combobox);
    },

    /**
     * @param Shopware.apps.Article.model.Article article
     */
    setArticle: function (article) {
        this.article = article;
    },

    /**
     * (Re-)loads the supplier order store based on the currently selected combination of article detail id
     * and warehouse ids.
     *
     * @param {Ext.form.field.ComboBox} combobox
     */
    loadSupplierOrders: function (combobox) {
        var panel = combobox.up('viison_pickware_erp_article_supplier_order_list-detail-supplier_order_list');
        if (panel) {
            panel.setLoading(true);
        }

        // Update the filter and load the result
        if (this.selectedArticleDetailId) {
            this.supplierOrderStore.filters.add('articleDetail', Ext.create('Ext.util.Filter', {
                property: 'items.articleDetailId',
                value: this.selectedArticleDetailId,
            }));
        } else {
            this.supplierOrderStore.filters.removeAtKey('articleDetail');
        }
        if (this.selectedWarehouseIds.length > 0) {
            this.supplierOrderStore.filters.add('selectedWarehouseIds', Ext.create('Ext.util.Filter', {
                property: 'warehouse.id',
                value: this.selectedWarehouseIds,
            }));
        } else {
            this.supplierOrderStore.filters.removeAtKey('selectedWarehouseIds');
        }

        this.supplierOrderStore.loadPage(1, {
            scope: this,
            callback: function (records, operation, success) {
                if (panel) {
                    panel.setLoading(false);
                }
                if (!success) {
                    Shopware.Notification.createGrowlMessage(
                        this.getViisonSnippet('notification/error/title'),
                        this.getViisonSnippet('notification/error/message/load'),
                        'ViisonPickwareERPArticleSupplierOrderList'
                    );
                }
            },
        });
    },

    /**
     * This method gets called when the user switches the article of the detail window by clicking on an article in
     * the article list while the split view mode is active.
     *
     * @param Shopware.apps.Article.model.Article article
     */
    articleChanged: function (article) {
        this.down('viison_common_variant_combo_box-combo_box').setArticle(article);
    },

});
