// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPBarcodeLabelPresets.controller.Editor', {

    extend: 'Ext.app.Controller',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_barcode_label_presets/main',

    refs: [
        { ref: 'mainWindow', selector: 'viison_pickware_erp_barcode_label_presets_main' },
        { ref: 'editorPanel', selector: 'viison_pickware_erp_barcode_label_presets_editor' },
        { ref: 'tabPanel', selector: 'viison_pickware_erp_barcode_label_presets_editor tabpanel' },
        { ref: 'fieldList', selector: 'viison_pickware_erp_barcode_label_presets_editor_field_list' },
        { ref: 'layoutTab', selector: 'viison_pickware_erp_barcode_label_presets_editor_layout_tab' },
        { ref: 'contentTab', selector: 'viison_pickware_erp_barcode_label_presets_editor_content_tab' },
        { ref: 'presetIdField', selector: 'viison_pickware_erp_barcode_label_presets_editor_layout_tab [name=id]' },
        { ref: 'previewField', selector: 'viison_pickware_erp_barcode_label_presets_editor [name=articleNumber]' },
        { ref: 'previewButton', selector: 'viison_pickware_erp_barcode_label_presets_editor [name=previewButton]' },
        { ref: 'templateField', selector: 'viison_pickware_erp_barcode_label_presets_editor_layout_tab [name=templateIdentifier]' },
    ],

    /**
     * @override
     */
    init: function () {
        this.control({
            'viison_pickware_erp_barcode_label_presets_editor [action=resetPreset]': {
                click: this.onResetPreset,
            },
            'viison_pickware_erp_barcode_label_presets_editor [name=articleNumber]': {
                valueselect: this.onPreviewArticleChange,
            },
            'viison_pickware_erp_barcode_label_presets_editor [action=previewPreset]': {
                click: this.onPreviewPreset,
            },
            'viison_pickware_erp_barcode_label_presets_editor [action=savePreset]': {
                click: this.onSavePreset,
            },
            'viison_pickware_erp_barcode_label_presets_editor_layout_tab [name=paperLayoutIdentifier]': {
                change: this.onPaperLayoutChange,
            },
            'viison_pickware_erp_barcode_label_presets_editor_layout_tab [name=templateIdentifier]': {
                beforeselect: this.onTemplateBeforeSelect,
                change: this.onTemplateChange,
            },
            viison_pickware_erp_barcode_label_presets_editor_field_list: {
                copyFieldToClipboard: this.onCopyFieldToClipboard,
            },
        });

        this.callParent(arguments);
    },

    /**
     * Removes all values from the editor.
     */
    clearEditor: function () {
        var editor = this.getEditorPanel();

        editor.clear();
        editor.disable();
    },

    /**
     * Displays a preset's values
     *
     * @param {Object} preset
     */
    setPreset: function (preset) {
        var editor = this.getEditorPanel();

        if (!preset.isFunctionalityInstalled()) {
            editor.hide();
        } else {
            this.getFieldList().bindStore(this.getFieldsStore(preset));
            this.getTemplateField().bindStore(this.getTemplateStore(preset));
            editor.setPreset(preset);
            editor.enable();
            editor.show();
        }
    },

    /**
     * Returns the correct fields store. The function will be overridden by other plugins
     * to display the available fields.
     *
     * @param {Shopware.apps.ViisonPickwareERPBarcodeLabelPresets.model.Preset} preset
     * @returns {Ext.Store}
     */
    getFieldsStore: function (preset) {
        return preset.get('type') === 'article' ? this.getStore('Fields') : null;
    },

    /**
     * Returns the correct templates store. The function will be overriden by other plugins
     * to display the available fields.
     *
     * @param {Shopware.apps.ViisonPickwareERPBarcodeLabelPresets.model.Preset} preset
     * @returns {Ext.Store}
     */
    getTemplateStore: function (preset) { // eslint-disable-line no-unused-vars
        // `preset` variable is use by the overring methods.
        return this.getStore('Templates');
    },

    /**
     * Reset event handler.
     */
    onResetPreset: function () {
        this.resetPreset();
    },

    /**
     * Remove all unsaved changes
     */
    resetPreset: function () {
        var preset = this.getStore('Presets').findRecord(
            'id',
            this.getPresetIdField().getValue()
        );

        if (preset) {
            this.setPreset(preset);
        }
    },

    /**
     * Called once an article is selected in the top bar. Unfortunately not, if the field is cleared.
     */
    onPreviewArticleChange: function () {
        var articleOrdernumber = this.getPreviewField().getHiddenField().getValue();

        if (articleOrdernumber.length > 0) {
            this.getPreviewButton().enable();
        }
    },

    /**
     * Button click handler for preview button.
     */
    onPreviewPreset: function () {
        if (!this.getEditorPanel().isValid()) {
            return;
        }

        this.previewPreset();
    },

    /**
     * Opens a new window to preview the current preset settings.
     */
    previewPreset: function () {
        var articleOrdernumber = this.getPreviewField().getHiddenField().getValue();

        if (articleOrdernumber.length === 0) {
            Ext.MessageBox.alert(
                this.getViisonSnippet('alert/preview_no_article_selected/title'),
                this.getViisonSnippet('alert/preview_no_article_selected/message')
            );

            return;
        }

        var params = this.getEditorPanelValues();
        params.previewIdentifier = articleOrdernumber;

        var url = ViisonCommonApp.assembleBackendUrl(
            '/ViisonPickwareERPBarcodeLabelPresets/renderPreview',
            params
        );
        window.open(url, '_blank');
    },

    /**
     * Returns all values of the editor.
     */
    getEditorPanelValues: function () {
        var preset = null;
        if (this.getPresetIdField().getValue()) {
            preset = this.getStore('Presets').findRecord(
                'id',
                this.getPresetIdField().getValue()
            );
        }

        return this.getEditorPanel().getValues(true, preset);
    },

    /**
     * Save preview button handler.
     */
    onSavePreset: function () {
        if (!this.getEditorPanel().isValid()) {
            return;
        }

        this.savePreset();
    },

    /**
     * Save the preset to the db.
     */
    savePreset: function () {
        var mainWindow = this.getMainWindow();
        var preset = this.getStore('Presets').findRecord(
            'id',
            this.getPresetIdField().getValue()
        );
        var presetValues = this.getEditorPanel().getValues(true, preset);

        mainWindow.setLoading(true);
        Ext.Ajax.request({
            url: ViisonCommonApp.assembleBackendUrl('/ViisonPickwareERPBarcodeLabelPresets/savePreset'),
            params: presetValues,
            scope: this,
            success: function (response) {
                var responseData = JSON.parse(response.responseText);

                var titleSnippet = '';
                var messageSnippet = '';
                if (responseData.success === true) {
                    titleSnippet = this.getViisonSnippet('notifictaion/save_preset/success/title');
                    messageSnippet = this.getViisonSnippet('notifictaion/save_preset/success/message');

                    // To update the id of a newly created preset
                    this.getPresetIdField().setValue(responseData.id);
                } else {
                    titleSnippet = this.getViisonSnippet('notifictaion/save_preset/failure/title');
                    messageSnippet = this.getViisonSnippet('notifictaion/save_preset/failure/message');
                }

                Shopware.Msg.createGrowlMessage(
                    titleSnippet,
                    Ext.String.format(messageSnippet, presetValues.name),
                    'Pickware'
                );

                this.getStore('Presets').load();
                mainWindow.setLoading(false);
            },
        });
    },

    /**
     * Enables and disbales the editing of the paper width and height boxes based on the paper type.
     * Change event handler.
     *
     * @param {Object} combobox
     * @param {string} paperLayoutIdentifier
     */
    onPaperLayoutChange: function (combobox, paperLayoutIdentifier) {
        var editor = this.getEditorPanel();

        if (paperLayoutIdentifier === 'custom') {
            editor.down('[name=paperWidth]').enable();
            editor.down('[name=paperHeight]').enable();
        } else {
            editor.down('[name=paperWidth]').disable();
            editor.down('[name=paperHeight]').disable();
        }
    },

    /**
     * Template select handler.
     * Prompts if the user really wants to change the template and thereby maybe loose data.
     *
     * @param {Object} combobox
     * @param {Object} template
     * @returns {boolean}
     */
    onTemplateBeforeSelect: function (combobox, template) {
        if (template.get('identifier') === combobox.getValue()) {
            return true;
        }

        Ext.MessageBox.confirm(
            this.getViisonSnippet('confirm/template_changed/title'),
            this.getViisonSnippet('confirm/template_changed/message'),
            function (confirmValue) {
                if (confirmValue === 'yes') {
                    combobox.setValue(template.get('identifier'));
                }
            },
            this
        );

        return false;
    },

    /**
     * Adjust editor content once the template changes.
     *
     * @param {Object} combobox
     * @param {string} templateIdentifier
     */
    onTemplateChange: function (combobox, templateIdentifier) {
        if (!templateIdentifier) {
            return;
        }

        var preset = this.getStore('Presets').findRecord(
            'id',
            combobox.up('viison_pickware_erp_barcode_label_presets_editor_layout_tab').down('[name=id]').getValue(),
            false,
            false,
            true
        );

        var template = this.getTemplateStore(preset).findRecord('identifier', templateIdentifier, 0, false, false, true);
        var contentForm = this.getContentTab();

        contentForm.getTemplateImage().setSrc(template.get('imageFile'));
        this.getLayoutTab().getTemplateImage().setSrc(template.get('imageFile'));

        if (preset) {
            this.mapOldBlocksOntoBlocksOfNewTemplate(templateIdentifier, preset.get('type'));
        }
    },

    /**
     * Maps the template blocks of a template onto the template blocks of a new template.
     *
     * @param {string} templateIdentifier
     * @param {string} presetType
     */
    mapOldBlocksOntoBlocksOfNewTemplate: function (templateIdentifier, presetType) {
        var blocks = [];
        var formValues = this.getContentTab().getForm().getValues();
        Object.keys(formValues).forEach(function (fieldName) {
            if (fieldName.substr(0, 6) === 'block-') {
                blocks.push({
                    name: fieldName.substr(6),
                    value: formValues[fieldName],
                });
            }
        });

        // Instantiate a preset model with the backuped blocks data.
        // Remark: There is no easy way to instantiate all associations of a model
        // along with the model itself. Hence we use a json reader to ensure, that
        // the blocks association is setup correctly.
        var presetReader = Ext.create('Ext.data.reader.Json', {
            model: 'Shopware.apps.ViisonPickwareERPBarcodeLabelPresets.model.Preset',
        });
        var presets = presetReader.read([{
            type: presetType,
            templateIdentifier: templateIdentifier,
            blocks: blocks,
        }]);
        var preset = presets.records[0];

        // Rerender the blocks form based on the new template and the backuped
        // blocks data. Please notice, that the blocks definition of the new
        // template may differ from the one of the old template and hence the
        // backuped blocks data may not be fully migrateable to the new template.
        // The backuped blocks data will be matched to the blocks defined by the
        // new template using their block names, meaning that the data of equally
        // named blocks will be kept.
        this.getContentTab().renderBlocks(preset);
    },

    /**
     * Copys values to the clipboard.
     *
     * @param {Object} field
     */
    onCopyFieldToClipboard: function (field) {
        var textarea = document.createElement('textarea');
        textarea.value = field.get('variable');
        document.body.appendChild(textarea);
        textarea.select();

        try {
            document.execCommand('copy');
        } catch (err) {
            Shopware.Msg.createGrowlMessage(
                this.getViisonSnippet('alert/copy_field_top_clipboard/failure/title'),
                this.getViisonSnippet('alert/copy_field_top_clipboard/failure/message'),
                'Pickware'
            );
        }

        document.body.removeChild(textarea);
    },

    /**
     *
     * @returns {boolean}
     */
    hasUnsavedChanges: function () {
        return this.getEditorPanel().hasUnsavedChanges();
    },

    /**
     * Checks whether the preset is the one show in the editor.
     *
     * @param {Object} preset
     * @returns {boolean}
     */
    isCurrentlyEditedPreset: function (preset) {
        var editorPresetValues = this.getEditorPanel().getValues(true);

        return preset.get('id') === parseInt(editorPresetValues.id, 10);
    },

    /**
     * Displays the layout tab.
     */
    goToLayoutTab: function () {
        this.getTabPanel().setActiveTab(this.getLayoutTab());
    },

});
