// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPBarcodeLabelPresets.controller.List', {

    extend: 'Ext.app.Controller',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_barcode_label_presets/main',

    refs: [
        { ref: 'mainWindow', selector: 'viison_pickware_erp_barcode_label_presets_main' },
        { ref: 'presetGridPanel', selector: 'viison_pickware_erp_barcode_label_presets_list' },
    ],

    /**
     * @override
     */
    init: function () {
        this.control({
            viison_pickware_erp_barcode_label_presets_list: {
                beforeselect: this.onBeforeSelect,
                deletePreset: this.onDeletePreset,
                clonePreset: this.onClonePreset,
                createPreset: this.onCreatePreset,
            },
        });

        this.callParent(arguments);
    },

    /**
     * Preset select handler.
     *
     * @param {Object} grid
     * @param {Object} preset
     * @returns {boolean}
     */
    onBeforeSelect: function (grid, preset) {
        return this.getController('Main').performPresetAction(
            function () {
                this.getController('EditorContext').setPreset(preset);
            }.bind(this),
            function () {
                this.getController('EditorContext').setPreset(preset);
                this.getPresetGridPanel().getView().select(preset);
            }.bind(this)
        );
    },

    /**
     * Delete button click handler.
     *
     * @param {Object} preset
     */
    onDeletePreset: function (preset) {
        Ext.MessageBox.confirm(
            this.getViisonSnippet('confirm/delete_preset/title'),
            Ext.String.format(
                this.getViisonSnippet('confirm/delete_preset/message'),
                preset.get('name')
            ),
            function (confirmValue) {
                if (confirmValue === 'yes') {
                    this.deletePreset(preset);
                }
            },
            this
        );
    },

    /**
     * Remove a preset.
     *
     * @param {Object} preset
     */
    deletePreset: function (preset) {
        var mainWindow = this.getMainWindow();

        mainWindow.setLoading(true);
        Ext.Ajax.request({
            url: ViisonCommonApp.assembleBackendUrl('/ViisonPickwareERPBarcodeLabelPresets/deletePreset'),
            params: {
                id: preset.get('id'),
            },
            scope: this,
            success: function (response) {
                var responseData = JSON.parse(response.responseText);
                var titleSnippet = '';
                var messageSnippet = '';
                if (responseData.success === true) {
                    titleSnippet = this.getViisonSnippet('notification/delete_preset/success/title');
                    messageSnippet = this.getViisonSnippet('notification/delete_preset/success/message');

                    this.getStore('Presets').remove(preset);
                    if (this.getController('Editor').isCurrentlyEditedPreset(preset)) {
                        this.getController('Editor').clearEditor();
                    }
                } else {
                    titleSnippet = this.getViisonSnippet('notification/delete_preset/failure/title');
                    messageSnippet = this.getViisonSnippet('notification/delete_preset/failure/message');
                }

                Shopware.Msg.createGrowlMessage(
                    titleSnippet,
                    Ext.String.format(messageSnippet, preset.get('name')),
                    'Pickware'
                );

                mainWindow.setLoading(false);
            },
        });
    },

    /**
     * Duplicate the preset handler.
     *
     * @param {Object} preset
     */
    onClonePreset: function (preset) {
        Ext.MessageBox.prompt(
            this.getViisonSnippet('confirm/clone_preset/title'),
            Ext.String.format(
                this.getViisonSnippet('confirm/clone_preset/message'),
                preset.get('name')
            ),
            function (buttonValue, presetName) {
                if (buttonValue !== 'ok') {
                    return;
                }

                if (Ext.util.Format.trim(presetName) === '') {
                    Shopware.Msg.createGrowlMessage(
                        this.getViisonSnippet('notification/clone_preset/empty_name/title'),
                        this.getViisonSnippet('notification/clone_preset/empty_name/message'),
                        'Pickware'
                    );

                    return;
                }

                this.clonePreset(preset, presetName);
            },
            this,
            false, // multiline
            preset.get('name') + ' ' + this.getViisonSnippet('clone_preset/name/postfix')
        );
    },

    /**
     * Duplicate a preset and give it a new name.
     *
     * @param {Object} preset
     * @param {string} presetName
     */
    clonePreset: function (preset, presetName) {
        var mainWindow = this.getMainWindow();

        mainWindow.setLoading(true);
        Ext.Ajax.request({
            url: ViisonCommonApp.assembleBackendUrl('/ViisonPickwareERPBarcodeLabelPresets/clonePreset'),
            params: {
                id: preset.get('id'),
                name: presetName,
            },
            scope: this,
            success: function (response) {
                var responseData = JSON.parse(response.responseText);
                var titleSnippet = '';
                var messageSnippet = '';
                if (responseData.success === true) {
                    titleSnippet = this.getViisonSnippet('notification/clone_preset/success/title');
                    messageSnippet = this.getViisonSnippet('notification/clone_preset/success/message');

                    // Reload presets store and auto-select cloned preset
                    this.getStore('Presets').load({
                        scope: this,
                        callback: function () {
                            var clonedPreset = this.getStore('Presets').findRecord('id', responseData.id);
                            this.getPresetGridPanel().getSelectionModel().select(clonedPreset);
                        },
                    });
                } else {
                    titleSnippet = this.getViisonSnippet('notification/duplicate_preset/failure/title');
                    messageSnippet = this.getViisonSnippet('notification/duplicate_preset/failure/message');
                }

                Shopware.Msg.createGrowlMessage(
                    titleSnippet,
                    Ext.String.format(messageSnippet, preset.get('name')),
                    'Pickware'
                );

                mainWindow.setLoading(false);
            },
        });
    },

    /**
     * Create preset click handler.
     *
     * @param {string} type
     */
    onCreatePreset: function (type) {
        this.getController('Main').performPresetAction(function () {
            this.createPreset(type || 'article');
            this.getController('Editor').goToLayoutTab();
        }.bind(this));
    },

    /**
     * Creates a new preset.
     *
     * @param {string} type
     */
    createPreset: function (type) {
        var paperLayout = this.getStore('PaperLayouts').first();
        var template = this.getStore('Templates').first();

        var preset = Ext.create('Shopware.apps.ViisonPickwareERPBarcodeLabelPresets.model.Preset', {
            id: -1,
            type: type,
            name: 'Neue Vorlage',
            paperLayoutIdentifier: paperLayout.get('identifier'),
            templateIdentifier: template.get('identifier'),
            isFunctionalityInstalled: true,
        });

        this.getController('Editor').setPreset(preset);
    },

});
