// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPBarcodeLabelPresets.controller.Main', {

    extend: 'Ext.app.Controller',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_barcode_label_presets/main',

    mainWindow: null,
    continueDespiteUnsavedChanges: false,

    /**
     * @override
     */
    init: function () {
        this.control({
            viison_pickware_erp_barcode_label_presets_main: {
                beforeclose: this.onBeforeCloseWindow,
            },
        });

        // Create the main window
        this.mainWindow = this.getView('Main').create({
            mainController: this,
        });
        this.mainWindow.setLoading(true);

        this.preloadStores(
            function () {
                this.mainWindow.setLoading(false);
            }.bind(this)
        );

        this.callParent(arguments);
    },

    /**
     * Prevents a window from being accidentally closed with unsaved changes.
     *
     * @param {Object} window
     * @returns {boolean}
     */
    onBeforeCloseWindow: function (window) {
        var canBeClosed = this.performPresetAction(
            Ext.emptyFn,
            function () {
                this.continueDespiteUnsavedChanges = true;
                window.close();
            }.bind(this)
        );

        if (!canBeClosed) {
            window.show();
        }

        return canBeClosed;
    },

    /**
     * Prompts the user before an action is executed which could wipe the unsaved changes.
     *
     * @param {function} callback
     * @param {function} confirmCallback
     * @returns {boolean}
     */
    performPresetAction: function (callback, confirmCallback) {
        confirmCallback = confirmCallback || callback;

        if (!this.getController('Editor').hasUnsavedChanges()) {
            callback();

            return true;
        }

        if (this.continueDespiteUnsavedChanges) {
            // Prevent endless loop of closing window, firing beforeclose event, detecting unsaved changes,
            // confirming that changes can be dismissed, closing window, firing beforeclose event...
            this.continueDespiteUnsavedChanges = false;

            return true;
        }

        Ext.MessageBox.confirm(
            this.getViisonSnippet('confirm/unsaved_changes/title'),
            this.getViisonSnippet('confirm/unsaved_changes/message'),
            function (confirmValue) {
                if (confirmValue === 'yes') {
                    this.continueDespiteUnsavedChanges = true;
                    confirmCallback();
                }
            },
            this
        );

        return false;
    },

    /**
     * Loads all needed stores when the window is opened.
     *
     * @param {function} callback
     */
    preloadStores: function (callback) {
        var stores = this.collectStoresToBePreloaded();

        var loadStatusChecker = function () {
            var allStoresLoaded = true;
            stores.forEach(function (store) {
                if (store.isLoading()) {
                    allStoresLoaded = false;
                }
            });

            if (allStoresLoaded) {
                callback();
            }
        };

        stores.forEach(function (store) {
            store.load({
                callback: loadStatusChecker,
            });
        });
    },

    /**
     * Returns all stores to be preloaded.
     *
     * @return {Object}
     */
    collectStoresToBePreloaded: function () {
        var preloadStores = [
            this.getStore('BarcodeTypes'),
            this.getStore('Fields'),
            this.getStore('Templates'),
            this.getStore('PaperLayouts'),
            this.getStore('Presets'),
        ];

        ViisonCommonEventBus.fireEvent('viison_pickware_erp_barcode_label_presets_collect_preload_stores', preloadStores);

        return preloadStores;
    },
});
