// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPBarcodeLabelPresets.model.Preset', {

    extend: 'Ext.data.Model',

    fields: [
        { name: 'id', type: 'int' },
        { name: 'type', type: 'text' },
        { name: 'name', type: 'text' },
        { name: 'translatedName', type: 'text' },
        { name: 'paperLayoutIdentifier', type: 'text' },
        { name: 'templateIdentifier', type: 'text' },
        { name: 'paperHeight', type: 'float' },
        { name: 'paperWidth', type: 'float' },
        { name: 'paddingTop', type: 'float' },
        { name: 'paddingRight', type: 'float' },
        { name: 'paddingBottom', type: 'float' },
        { name: 'paddingLeft', type: 'float' },
        { name: 'isFunctionalityInstalled', type: 'boolean' },
        {
            name: 'comment',
            type: 'text',
            convert: function (val) {
                if (val !== null) {
                    return val;
                }

                return '';
            },
        },
        {
            name: 'group',
            type: 'string',
            convert: function (val, record) {
                return (record.get('type') === 'bin_location' || record.get('type') === 'picking_box') ? 'other' : record.get('type');
            },
        },
    ],

    hasMany: {
        model: 'Shopware.apps.ViisonPickwareERPBarcodeLabelPresets.model.PresetBlock',
        associationKey: 'blocks',
        name: 'getBlocks',
    },

    /**
     * Returns if the label is of type article.
     *
     * @returns {boolean}
     */
    isArticleBarcodeLabelPreset: function () {
        return this.get('type') === 'article';
    },

    /**
     * Returns if the content can be edited.
     *
     * @returns {boolean}
     */
    isContentEditable: function () {
        return this.get('type') === 'article';
    },

    /**
     * Checks is the preset can be deleted.
     *
     * @returns {boolean}
     */
    isPresetDeleteable: function () {
        return this.get('type') === 'article';
    },

    /**
     * Checks is the preset can be deleted.
     *
     * @returns {boolean}
     */
    isPresetDuplicatable: function () {
        return this.get('type') === 'article';
    },

    /**
     * Returns if the label's functionality is installed.
     *
     * @returns {boolean}
     */
    isFunctionalityInstalled: function () {
        return this.get('isFunctionalityInstalled');
    },

    /**
     * Returns the name of the field list store for the preset type.
     *
     * @returns {string}
     */
    getFieldsStore: function () {
        return this.subApp.getStore('Fields');
    },

    /**
     * Returns the overlay xtype in case the preset is not installed.
     *
     * @return {string}
     */
    getOverlayXtype: function () {
        if (this.get('type') === 'customer_card') {
            return 'viison_pickware_erp_barcode_label_presets_overlay_available_with_pos';
        }

        if (this.get('type') === 'picking_box') {
            return 'viison_pickware_erp_barcode_label_presets_overlay_available_with_mobile';
        }

        return null;
    },

    /**
     * Check is the given preset has the same values as the current object.
     *
     * @param {Object} presetValues
     * @returns {boolean}
     */
    equals: function (presetValues) {
        switch (this.get('type')) {
            case 'article':
                return this.hasEqualLayoutValues(presetValues) && this.hasEqualContentValues(presetValues);
            case 'customer_card':
                return this.hasEqualLayoutValues(presetValues) && this.hasEqualContentValues(presetValues);
            case 'bin_location':
                return this.hasEqualLayoutValues(presetValues);
            case 'picking_box':
                return this.hasEqualLayoutValues(presetValues);
            default:
                return false;
        }
    },

    /**
     * Check if the layout values are the same.
     *
     * @param {Object} presetValues
     * @returns {boolean}
     */
    hasEqualLayoutValues: function (presetValues) {
        if (this.get('name') !== presetValues.name) {
            return false;
        }

        if (this.get('templateIdentifier') !== presetValues.templateIdentifier) {
            return false;
        }

        if (this.get('paperLayoutIdentifier') !== presetValues.paperLayoutIdentifier) {
            return false;
        }

        if (this.get('paperLayoutIdentifier') === 'custom') {
            if (parseFloat(this.get('paperWidth')) !== parseFloat(presetValues.paperWidth)) {
                return false;
            }
            if (parseFloat(this.get('paperHeight')) !== parseFloat(presetValues.paperHeight)) {
                return false;
            }
        }

        if (parseFloat(this.get('paddingTop')) !== parseFloat(presetValues.paddingTop)) {
            return false;
        }

        if (parseFloat(this.get('paddingRight')) !== parseFloat(presetValues.paddingRight)) {
            return false;
        }

        if (parseFloat(this.get('paddingBottom')) !== parseFloat(presetValues.paddingBottom)) {
            return false;
        }

        if (parseFloat(this.get('paddingLeft')) !== parseFloat(presetValues.paddingLeft)) {
            return false;
        }

        if (Ext.util.Format.trim(this.get('comment')) !== Ext.util.Format.trim(presetValues.comment)) {
            return false;
        }

        return true;
    },

    /**
     * Check if the content values are the same.
     *
     * @param {Object} presetValues
     * @returns {boolean}
     */
    hasEqualContentValues: function (presetValues) {
        var isEqual = true;
        this.getBlocks().each(function (presetBlock) {
            var blockId = 'block-' + presetBlock.get('name');
            if (typeof presetValues[blockId] !== 'undefined') {
                isEqual = isEqual && this.isBlockUnchanged(presetBlock.get('value'), presetValues[blockId]);
            } else {
                isEqual = false;
            }
        }.bind(this));

        return isEqual;
    },

    /**
     * Check if a blocks values was changed by the user.
     *
     * @param {string} expected
     * @param {string} actual
     * @returns {boolean}
     */
    isBlockUnchanged: function (expected, actual) {
        if (!this.wereBlocksPotentiallyShown(actual)) {
            return true;
        }

        // Check if tinymce edited the value to prevent the immediate loading of all images.
        // Tinymce will change the value back, once the image is loaded.
        if (actual.indexOf('/engine/Library/TinyMce/plugins/media_selection/assets/placeholder-image.png') !== -1) {
            return true;
        }

        return expected === actual;
    },

    /**
     * Checks if the blocks might have been shown. This information is useful since blocks which have not been shown
     * might have a different content then blocks which have been shown due to tinymce's image handling.
     *
     * @param {string} value
     * @returns {boolean}
     */
    wereBlocksPotentiallyShown: function (value) {
        return value.indexOf('src="http') === -1;
    },

    /**
     * Returns the block with a certain name
     *
     * @param {string} name
     * @returns {Shopware.apps.ViisonPickwareERPBarcodeLabelPresets.model.PresetBlock}
     */
    getBlockByName: function (name) {
        var block = null;
        this.getBlocks().each(function (presetBlock) {
            if (presetBlock.get('name') === name) {
                block = presetBlock;
            }
        });

        return block;
    },
});
