// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPBarcodeLabelPresets.view.Editor', {

    extend: 'Ext.panel.Panel',
    alias: 'widget.viison_pickware_erp_barcode_label_presets_editor',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_barcode_label_presets/main',

    cls: 'viison_pickware_erp_barcode_label_presets_editor viison-common--tab-panel has--header-without-border',
    layout: {
        type: 'vbox',
        pack: 'start',
        align: 'stretch',
    },
    border: false,

    /**
     * @override
     */
    initComponent: function () {
        this.items = this.createLayout();

        this.callParent(arguments);
    },

    /**
     * Return main layout.
     *
     * @returns {Object[]}
     */
    createLayout: function () {
        return [
            this.createTopToolbar(),
            {
                xtype: 'tabpanel',
                flex: 1,
                layout: 'fit',
                border: false,
                bodyStyle: {
                    border: 'none !important',
                },
                autoScroll: true,
                items: [
                    {
                        xtype: 'viison_pickware_erp_barcode_label_presets_editor_layout_tab',
                        itemId: 'layoutTab',
                    },
                    {
                        xtype: 'viison_pickware_erp_barcode_label_presets_editor_content_tab',
                        itemId: 'contentTab',
                        disabled: true,
                    },
                ],
            },
            this.createBottomToolbar(),
        ];
    },

    /**
     * Add top bar with preset name and preview button.
     *
     * @returns {Object[]}
     */
    createTopToolbar: function () {
        return {
            dock: 'top',
            xtype: 'toolbar',
            itemId: 'toolbar',
            ui: 'shopware-ui',
            layout: {
                type: 'hbox',
                align: 'stretch',
            },
            padding: '7 12 7 12',
            style: {
                borderBottom: '1px solid #CFD8DE !important',
            },
            items: [
                this.createPresetNameForm(),
                { xtype: 'tbfill' },
            ],
        };
    },

    /**
     * Return the name field for the preset.
     *
     * @returns {Object[]}
     */
    createPresetNameForm: function () {
        return {
            xtype: 'form',
            itemId: 'presetNameForm',
            border: false,
            layout: {
                type: 'hbox',
                align: 'middle',
            },
            bodyStyle: {
                backgroundColor: 'transparent !important',
            },
            plugins: [{
                pluginId: 'translation',
                ptype: 'translation',
                translationType: 'viison_pickware_erp_barcode_label_preset_name',
                translationMerge: true,
            }],
            items: [
                {
                    xtype: 'label',
                    cls: 'is--viison-common--label',
                    text: this.getViisonSnippet('layout_tab/label_description_field') + ':',
                    margin: '7 10 0 0',
                }, {
                    xtype: 'textfield',
                    name: 'name',
                    translatable: true,
                    translationLabel: this.getViisonSnippet('layout_tab/label_description_field'),
                    labelWidth: 0, // Due to the inner workings of the translation plugin we need to hide the label this way instead of using "hideLabel"
                    allowBlank: false,
                    width: 300,
                    maskOnDisable: false,
                },
            ],
        };
    },

    /**
     * Add the article selection box and the preview button.
     *
     * @returns {Object[]}
     */
    createPreviewForm: function () {
        return new Ext.form.Panel({
            xtype: 'form',
            border: false,
            name: 'previewForm',
            layout: {
                type: 'hbox',
                align: 'middle',
            },
            bodyStyle: {
                backgroundColor: 'transparent !important',
            },
            items: [
                {
                    xtype: 'label',
                    cls: 'is--viison-common--label',
                    text: this.getViisonSnippet('topbar/article_search_label') + ':',
                    margin: '7 10 0 0',
                },
                {
                    xtype: 'articlesearchfield',
                    itemId: 'previewArticleField',
                    name: 'articleNumber',
                    cls: 'is--viison-common--article-live-search-field',
                    store: Ext.create('Shopware.apps.Base.store.Variant', {}),
                    hideLabel: true,
                    width: 250,
                    maskOnDisable: false,
                    dropDownOffset: [0, 8],
                },
                {
                    xtype: 'button',
                    itemId: 'previewButton',
                    action: 'previewPreset',
                    name: 'previewButton',
                    text: this.getViisonSnippet('topbar/preview_button'),
                    iconCls: 'sprite-globe--arrow',
                    align: 'right',
                    margin: '0 0 0 5',
                },
            ],
        });
    },

    /**
     * Return the editor's toolbar on the bottom.
     * @returns {Object[]}
     */
    createBottomToolbar: function () {
        return {
            xtype: 'toolbar',
            dock: 'bottom',
            ui: 'shopware-ui is--viison-common--button-panel',
            layout: {
                pack: 'end',
            },
            width: '100%',
            items: [
                {
                    xtype: 'button',
                    action: 'resetPreset',
                    text: this.getViisonSnippet('bottombar/reset_button'),
                    align: 'right',
                    cls: 'secondary',
                },
                {
                    xtype: 'button',
                    action: 'savePreset',
                    text: this.getViisonSnippet('bottombar/save_button'),
                    align: 'right',
                    cls: 'primary',
                },
            ],
        };
    },

    /**
     * Extracts all current values from the editor.
     *
     * @param {boolean} includeValuesOfDisabledFields
     * @param {Shopware.apps.ViisonPickwareERPBarcodeLabelPresets.model.Preset} preset
     * @param {Object} preset
     */
    getValues: function (includeValuesOfDisabledFields, preset) {
        var nameField = this.getPresetNameField();

        return Ext.apply(
            this.getContentTabValues(preset),
            this.getLayoutFormValues(includeValuesOfDisabledFields),
            {
                name: nameField.getValue(),
            }
        );
    },

    /**
     * Get layout form values from the editor.
     *
     * @param {boolean} includeValuesOfDisabledFields
     * @returns {Object}
     */
    getLayoutFormValues: function (includeValuesOfDisabledFields) {
        if (typeof includeValuesOfDisabledFields === 'undefined') {
            includeValuesOfDisabledFields = false;
        }

        var layoutTab = this.getLayoutTabPanel();

        var layoutFormValues = {};
        if (includeValuesOfDisabledFields) {
            var disabledAfterValueProcessing = false;
            if (this.isDisabled()) {
                this.enable();
                disabledAfterValueProcessing = true;
            }

            var disabledItems = layoutTab.query('[disabled=true]');
            Ext.each(disabledItems, function (item) {
                item.enable();
            });
            layoutFormValues = layoutTab.getForm().getValues();
            Ext.each(disabledItems, function (item) {
                item.disable();
            });

            if (disabledAfterValueProcessing) {
                this.disable();
            }
        } else {
            layoutFormValues = layoutTab.getForm().getValues();
        }

        return layoutFormValues;
    },

    /**
     * Extract block values from the editor.
     *
     * @param {Shopware.apps.ViisonPickwareERPBarcodeLabelPresets.model.Preset} preset
     * @returns {Object}
     */
    getContentTabValues: function (preset) {
        var contentTab = this.getContentTabPanel();
        var contentValues = contentTab.getForm().getValues();

        if (!preset) {
            return contentValues;
        }

        var blockNames = Object.keys(contentValues);
        blockNames.forEach(function (field) {
            if (contentValues[field].indexOf('/engine/Library/TinyMce/plugins/media_selection/assets/placeholder-image.png') !== -1) {
                contentValues[field] = preset.getBlockByName(field.substr(6)).get('value');
            }
        });

        return contentValues;
    },

    /**
     * Checks if the entered values are valid.
     *
     * @returns {boolean}
     */
    isValid: function () {
        var layoutTab = this.getLayoutTabPanel();
        var contentTab = this.getContentTabPanel();

        return layoutTab.getForm().isValid() && contentTab.getForm().isValid();
    },

    /**
     * Determines whether there are unsaved changes.
     *
     * @returns {boolean}
     */
    hasUnsavedChanges: function () {
        var presetValues = this.getValues(true);

        if (!presetValues.id) {
            return false;
        }

        if (presetValues.id === -1) {
            return true;
        }

        var presetsStore = this.subApp.getStore('Presets');
        var preset = presetsStore.getById(parseInt(presetValues.id, 10));
        if (!preset) {
            return true;
        }

        return !preset.equals(presetValues);
    },

    /**
     * Set the preset, the editor shows.
     *
     * @param {Object} preset
     */
    setPreset: function (preset) {
        this.removePreviewForm();

        this.getLayoutTabPanel().setPreset(preset);
        this.getContentTabPanel().setPreset(preset);

        // Update preset name field
        var presetNameForm = this.getPresetNameForm();
        presetNameForm.getForm().loadRecord(preset);

        if (preset.isContentEditable()) {
            this.enableTopToolbar();
            this.showPreviewForm(preset);

            // Trigger initialization of field translation
            presetNameForm.getForm().fireEvent('recordchange');
        } else {
            this.disableTopToolbar();
            this.down('tabpanel').setActiveTab(this.getLayoutTabPanel());
        }

        this.enable();
    },

    /**
     * Removes the preview form from the UI.
     */
    removePreviewForm: function () {
        this.down('#toolbar').remove(this.down('[name=previewForm]'));
    },

    /**
     * Shows the preview form for article presets.
     *
     * @param {Object} preset
     */
    showPreviewForm: function (preset) {
        if (preset.isArticleBarcodeLabelPreset()) {
            this.removePreviewForm();
            this.down('#toolbar').add(this.createPreviewForm());
        }
    },

    /**
     * Clear the editor.
     */
    clear: function () {
        this.getLayoutTabPanel().getForm().reset();
        this.getContentTabPanel().getForm().reset();
        this.getPresetNameField().setValue('');
    },

    /**
     * @override
     */
    disable: function () {
        this.disableTopToolbar();

        this.callParent(arguments);
    },

    /**
     * Disables the editing of the top bar.
     */
    disableTopToolbar: function () {
        this.getPresetNameField().disable();
    },

    /**
     * Enable the toolbar
     */
    enableTopToolbar: function () {
        this.getPresetNameField().enable();
    },

    getPresetNameField: function () {
        return this.down('textfield[name=name]');
    },

    getPresetNameForm: function () {
        return this.down('#presetNameForm');
    },

    getPreviewButton: function () {
        return this.down('#previewButton');
    },

    getPreviewArticleField: function () {
        return this.down('#previewArticleField');
    },

    getLayoutTabPanel: function () {
        return this.down('#layoutTab');
    },

    getContentTabPanel: function () {
        return this.down('#contentTab');
    },

});
