// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPBarcodeLabelPresets.view.editor.ContentTab', {

    extend: 'Ext.form.Panel',
    alias: 'widget.viison_pickware_erp_barcode_label_presets_editor_content_tab',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_barcode_label_presets/main',

    cls: 'viison_pickware_erp_barcode_label_presets_editor_content_tab',
    autoShow: true,
    autoScroll: true,
    layout: {
        type: 'hbox',
    },
    border: false,

    /**
     * @override
     */
    initComponent: function () {
        this.title = this.getViisonSnippet('content_tab/title');
        this.items = this.createLayout();

        this.callParent(arguments);
    },

    /**
     * Return the main layout.
     *
     * @returns {Object[]}
     */
    createLayout: function () {
        return [
            {
                xtype: 'panel',
                itemId: 'blocksPanel',
                cls: 'is--blocks-panel',
                height: '100%',
                border: false,
                flex: 2,
                layout: {
                    type: 'vbox',
                    align: 'stretch',
                },
                items: [
                    {
                        xtype: 'panel',
                        style: {
                            borderBottom: '1px solid #CFD8DE',
                            backgroundColor: '#f3f5f7',
                        },
                        layout: {
                            type: 'vbox',
                            align: 'center',
                            pack: 'center',
                        },
                        border: false,
                        height: 201,
                        items: {
                            xtype: 'image',
                            itemId: 'templateImage',
                            width: 200,
                            height: 150,
                        },
                    },
                    {
                        xtype: 'fieldset',
                        flex: 1,
                        border: false,
                    },
                ],
            },
            {
                xtype: 'viison_pickware_erp_barcode_label_presets_editor_field_list',
                flex: 1,
            },
        ];
    },

    /**
     * Set the preset for the content tab.
     *
     * @param {Shopware.apps.ViisonPickwareERPBarcodeLabelPresets.model.Preset} preset
     */
    setPreset: function (preset) {
        if (!preset.isContentEditable()) {
            this.disable();

            return;
        }

        this.renderBlocks(preset);
        this.enable();
    },

    /**
     * Show all fields for the template's blocks.
     *
     * @param {Shopware.apps.ViisonPickwareERPBarcodeLabelPresets.model.Preset} preset
     */
    renderBlocks: function (preset) {
        var templatesStore = this.subApp.getController('Editor').getTemplateStore(preset);
        var barcodeTypesStore = this.subApp.getStore('BarcodeTypes');
        var template = templatesStore.findRecord('identifier', preset.get('templateIdentifier'), 0, false, false, true);
        var blocksPanel = this.getBlocksPanel();

        blocksPanel.remove(blocksPanel.query('fieldset')[0]);

        var barcodeFields = [];
        var contentFields = [];
        template.getBlocks().each(function (templateBlock) {
            var presetBlock = preset.getBlocks().findRecord('name', templateBlock.get('name'));
            var blockValue = '';
            if (presetBlock) {
                blockValue = presetBlock.get('value');
            }

            if (templateBlock.get('name').indexOf('barcode') === 0) {
                barcodeFields = barcodeFields.concat(
                    this.createNewBarcodeField(preset, templateBlock, barcodeTypesStore)
                );
            } else {
                contentFields = contentFields.concat(this.createNewContentField(templateBlock.get('name'), blockValue));
            }
        }, this);

        blocksPanel.add(new Ext.form.FieldSet({
            flex: 1,
            autoScroll: true,
            border: false,
            margin: 0,
            padding: 0,
            defaults: {
                xtype: 'textfield',
            },
            items: [
                {
                    xtype: 'container',
                    layout: {
                        type: 'vbox',
                        align: 'stretch',
                    },
                    padding: 25,
                    items: barcodeFields.concat(contentFields),
                },
            ],
        }));
    },

    /**
     * Instantiates a barcode field with its label.
     *
     * @param {Object} preset
     * @param {Object} templateBlock
     * @param {Object} barcodeTypesStore
     * @return {Array}
     */
    createNewBarcodeField: function (preset, templateBlock, barcodeTypesStore) {
        var value = null;
        var barcodeFields = [];

        if (preset.getBlocks().findRecord('name', templateBlock.get('name'))) {
            value = preset.getBlocks().findRecord('name', templateBlock.get('name')).get('value');
        }

        barcodeFields.push({
            xtype: 'label',
            cls: 'is--viison-common--label',
            text: 'Barcode',
            margin: '0 0 7 0',
        });
        barcodeFields.push(new Ext.form.ComboBox({
            hideLabel: true,
            name: 'block-' + templateBlock.get('name'),
            labelAlign: 'top',
            width: '100%',
            queryMode: 'local',
            displayField: 'description',
            valueField: 'variable',
            allowBlank: false,
            value: value || barcodeTypesStore.first(),
            store: barcodeTypesStore,
            margin: '0 0 25 0',
        }));

        return barcodeFields;
    },

    /**
     * Instantiates a content field with its label.
     *
     * @param {string} name
     * @param {string} value
     * @return {Array}
     */
    createNewContentField: function (name, value) {
        var contentFields = [];
        contentFields.push({
            xtype: 'label',
            cls: 'is--viison-common--label is--block',
            text: name,
        });
        contentFields.push(new Shopware.form.field.TinyMCE({
            hideLabel: true,
            name: 'block-' + name,
            labelAlign: 'top',
            width: '100%',
            height: 200,
            margin: '0 0 35 0',
            value: value,
            editor: {
                plugins: 'media_selection,fullscreen',
                skin_variant: 'shopware',
                theme_advanced_buttons1: 'undo,redo,|,bold,italic,underline,|,fontsizeselect,forecolor,|,bullist,numlist,|,justifyleft,justifycenter,justifyright,|,media_selection,code,fullscreen,',
                theme_advanced_buttons2: '',
                theme_advanced_buttons3: '',
                theme_advanced_buttons4: '',
            },
        }));

        return contentFields;
    },

    getBlocksPanel: function () {
        return this.down('#blocksPanel');
    },

    getTemplateImage: function () {
        return this.down('#templateImage');
    },

});
