// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPBarcodeLabelPrinting.controller.Main', {

    extend: 'Ext.app.Controller',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_barcode_label_printing/main',

    mainWindow: null,
    advancedWindow: null,

    /**
     * @Override
     */
    init: function () {
        this.control({
            'viison_pickware_erp_barcode_label_printing button[action=showBarcodeLabels]': {
                click: this.onShowBarcodeLabels,
            },
            'viison_pickware_erp_barcode_label_printing button[action=deleteBarcodeLabels]': {
                click: this.onDeleteBarcodeLabels,
            },
            'viison_pickware_erp_barcode_label_printing button[action=cancel]': {
                click: this.onCancel,
            },
            'viison_pickware_erp_barcode_label_printing-main-list': {
                deleteBarcodeLabel: this.onDeleteBarcodeLabel,
                saveBarcodeLabel: this.onSaveBarcodeLabel,
            },
            viison_pickware_erp_barcode_label_printing: {
                addArticleBarcodeLabel: this.onAddArticleBarcodeLabel,
                addBinLocationBarcodeLabel: this.onAddBinLocationBarcodeLabel,
            },
        });

        // Create the main window
        this.mainWindow = this.getView('Main').create({
            articleLabelStore: this.getStore('ArticleBarcodeLabel').load(),
            binLocationLabelStore: this.getStore('BinLocationBarcodeLabel').load(),
        });

        this.callParent(arguments);
    },

    /**
     * Reads the coordinates from the form and uses them to open a new tab
     * showing a PDF with all generated barcode labels. Additionally an alert is displayed,
     * reminding the user to delete the labels after printing them.
     *
     * @param {Ext.button.Button} button
     */
    onShowBarcodeLabels: function (button) {
        var labelListTab = button.up('viison_pickware_erp_barcode_label_printing [customIdentifier=tabParent]');
        var list = labelListTab.down('viison_pickware_erp_barcode_label_printing-main-list');

        // Check if the form is valid
        if (!labelListTab.getForm().isValid()) {
            return;
        }
        // Check if at least one barcode label exists
        var labelStore = labelListTab.labelStore;
        if (labelStore.count() === 0) {
            Shopware.Notification.createStickyGrowlMessage({
                title: this.getViisonSnippet('notification/no_labels_error/title'),
                text: this.getViisonSnippet('notification/no_labels_error/message'),
            });

            return;
        }

        // ExtJS does not return values on disabled fields using `getForm().getFieldValues()`. We therefore have to
        // query the value separately.
        var values = labelListTab.getForm().getFieldValues();
        values.preset = labelListTab.down('[name=preset]').value;

        // Create the query parameters
        var params = {
            preset: values.preset,
            type: labelListTab.labelStore.labelType,
            params: {},
        };
        if (typeof values.startX !== 'undefined') {
            params.startX = values.startX;
        }
        if (typeof values.startY !== 'undefined') {
            params.startY = values.startY;
        }

        var blackListedFormNames = [
            'inputItem',
            'preset',
            'startX',
            'startY',
            'type',
        ];
        Object.keys(values).forEach(function (key) {
            if (key.indexOf('hidden') === 0) {
                return;
            }

            if (key.indexOf('live') === 0) {
                return;
            }

            if (key.indexOf('combobox') === 0) {
                return;
            }

            if (blackListedFormNames.indexOf(key) > -1) {
                return;
            }

            params.params[key] = values[key];
        });
        params.params = JSON.stringify(params.params);

        // Add a label filter to the parameters
        var filter = this.getDefaultFilters(labelStore);
        var labels = list.getSelectionModel().getSelection();
        if (labels.length > 0) {
            var labelIds = Ext.Array.map(labels, function (label) {
                return label.get('id');
            });
            filter.push({
                property: list.getFilterPropertyName(),
                value: labelIds,
            });
        }
        params.filter = Ext.JSON.encode(filter);

        params.sort = Ext.JSON.encode(this.getSorters(labelStore));

        // Open the PDF in a new window
        window.open(
            ViisonCommonApp.assembleBackendUrl(labelStore.renderBarcodeLabelsUrl, params),
            '_blank'
        );

        // Show an alert reminding the user to remove the (selected) barcode labels
        if (labels.length > 0) {
            this.showSelectedRemovalConfirmationAlert(labelListTab, this.getViisonSnippet('alert/confirm_clearing/after_printing_selection/title'), this.getViisonSnippet('alert/confirm_clearing/after_printing_selection/message'));
        } else {
            this.showDeleteAllConfirmationAlert(labelListTab, this.getViisonSnippet('alert/confirm_clearing/after_printing/title'), this.getViisonSnippet('alert/confirm_clearing/after_printing/message'));
        }
    },

    getDefaultFilters: function (labelStore) {
        if (!labelStore) {
            return [];
        }

        var filters = [];
        labelStore.filters.each(function (filter) {
            filters.push({
                property: filter.property,
                value: filter.value,
                expression: filter.expression,
                operator: filter.operator,
            });
        });

        return filters;
    },

    /**
     * @param {Ext.data.Store} store
     * @returns {object[]}
     */
    getSorters: function (store) {
        if (!store || !store.remoteSort) {
            return [];
        }

        var sorters = [];
        store.sorters.each(function (sorter) {
            sorters.push({
                property: sorter.property,
                direction: sorter.direction,
            });
        });

        return sorters;
    },

    /**
     * Called when the user clicks the delete (selected) label
     *
     * @param {Ext.button.Button} button
     */
    onDeleteBarcodeLabels: function (button) {
        var labelListTab = button.up('viison_pickware_erp_barcode_label_printing [customIdentifier=tabParent]');
        var list = labelListTab.down('viison_pickware_erp_barcode_label_printing-main-list');
        if (list.getSelectionModel().getSelection().length > 0) {
            this.showSelectedRemovalConfirmationAlert(labelListTab, this.getViisonSnippet('alert/confirm_clearing/selection/title'), this.getViisonSnippet('alert/confirm_clearing/selection/message'));
        } else {
            this.showDeleteAllConfirmationAlert(labelListTab, this.getViisonSnippet('alert/confirm_clearing/default/title'), this.getViisonSnippet('alert/confirm_clearing/default/message'));
        }
    },

    /**
     * Closes the main window/application.
     */
    onCancel: function () {
        this.mainWindow.close();
    },

    /**
     * Saves the given record in the database.
     *
     * @param record The record, which shall be saved.
     */
    onSaveBarcodeLabel: function (record) {
        record.save({
            scope: this,
            callback: function (data, operation) {
                if (operation.success) {
                    // Display a success notification
                    Shopware.Notification.createGrowlMessage(this.getViisonSnippet('notification/update_barcode_label/success/title'), this.getViisonSnippet('notification/update_barcode_label/success/message'), 'Pickware');
                } else {
                    // Display a error notification
                    Shopware.Notification.createGrowlMessage(this.getViisonSnippet('notification/update_barcode_label/error/title'), this.getViisonSnippet('notification/update_barcode_label/error/message'), 'Pickware');
                }
            },
        });
    },

    /**
     * Adds a new barcode label for the given article to the grid.
     *
     * @param Shopware.apps.ViisonPickwareERPBarcodeLabelPrinting.view.ArticleLabels labelListTab
     * @param articleRecord The selected article record, for which a barcode label should be generated.
     */
    onAddArticleBarcodeLabel: function (labelListTab, articleRecord) {
        this.addBarcodeLabel(labelListTab, {
            type: 'article',
            identifiers: [
                articleRecord.get('number'),
            ],
        }, {
            title: this.getViisonSnippet('notification/failed_to_add_article_to_barcode_labels/title'),
            message: this.getViisonSnippet('notification/failed_to_add_article_to_barcode_labels/message'),
        });
    },

    /**
     * @param Shopware.apps.ViisonPickwareERPBarcodeLabelPrinting.view.BinLocationLabels labelListTab
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.model.BinLocation binLocation
     */
    onAddBinLocationBarcodeLabel: function (labelListTab, binLocation) {
        this.addBarcodeLabel(labelListTab, {
            type: 'bin_location',
            identifiers: [
                binLocation.get('id'),
            ],
        }, {
            title: this.getViisonSnippet('notification/add_bin_location_barcode_label/failure/title'),
            message: this.getViisonSnippet('notification/add_bin_location_barcode_label/failure/message'),
        });
    },

    /**
     * @param Shopware.apps.ViisonPickwareERPBarcodeLabelPrinting.view.LabelListTab labelListTab
     * @param Object params
     * @param Object failureNotification
     */
    addBarcodeLabel: function (labelListTab, params, failureNotification) {
        Ext.Ajax.request({
            url: ViisonCommonApp.assembleBackendUrl('/ViisonPickwareERPBarcodeLabelPrinting/enqueueBarcodeLabel'),
            method: 'POST',
            jsonData: params,
            callback: function (options, success) {
                if (!success) {
                    Shopware.Notification.createGrowlMessage(failureNotification.title, failureNotification.message, 'Pickware');
                } else {
                    labelListTab.labelStore.load();
                }
            },
        });
    },

    /**
     * Displays an alert asking the user to confirm clearing the list and does so,
     * if confirmed.
     *
     * @param Shopware.apps.ViisonPickwareERPBarcodeLabelPrinting.view.LabelListTab labelListTab
     * @param String title
     * @param String message
     */
    showDeleteAllConfirmationAlert: function (labelListTab, title, message) {
        Ext.Msg.confirm(title, message, function (buttonId) {
            if (buttonId !== 'yes') {
                return;
            }

            Ext.Ajax.request({
                method: 'POST',
                url: ViisonCommonApp.assembleBackendUrl(labelListTab.labelStore.deleteAllLabelsUrl),
                params: {
                    filter: Ext.JSON.encode(this.getDefaultFilters(labelListTab.labelStore)),
                },
                scope: this,
                callback: function (options, success) {
                    if (success) {
                        labelListTab.labelStore.load();

                        Shopware.Notification.createGrowlMessage(this.getViisonSnippet('notification/clear_list/success/title'), this.getViisonSnippet('notification/clear_list/success/message'), 'Pickware');
                    } else {
                        Shopware.Notification.createGrowlMessage(this.getViisonSnippet('notification/clear_list/error/title'), this.getViisonSnippet('notification/clear_list/error/message'), 'Pickware');
                    }
                },
            });
        }, this);
    },

    /**
     * Displays an alert asking the user to confirm the removal of all selected barcode labels and,
     * if confirmed, performs the deletion.
     *
     * @param Shopware.apps.ViisonPickwareERPBarcodeLabelPrinting.view.LabelListTab labelListTab
     * @param String title
     * @param String message
     */
    showSelectedRemovalConfirmationAlert: function (labelListTab, title, message) {
        var list = labelListTab.down('viison_pickware_erp_barcode_label_printing-main-list');

        // Get all selected barcode labels
        var labels = list.getSelectionModel().getSelection();
        if (labels.length === 0) {
            return;
        }

        // Present a confirm alert before removing the selected entries from the list
        Ext.Msg.confirm(title, message, function (buttonId) {
            if (buttonId !== 'yes') {
                return;
            }

            this.deleteLabels(labelListTab, labels);
        }, this);
    },

    /**
     * Deletes the given barcodeLabel from the database.
     *
     * @param Shopware.apps.ViisonPickwareERPBarcodeLabelPrinting.view.LabelListTab labelListTab
     * @param Shopware.apps.ViisonPickwareERPBarcodeLabelPrinting.model.BarcodeLabel barcodeLabel
     */
    onDeleteBarcodeLabel: function (labelListTab, barcodeLabel) {
        this.deleteLabels(labelListTab, [barcodeLabel]);
    },

    /**
     * Deletes all barcode labels.
     *
     * @param {Object} labelListTab
     * @param {Object} labels
     */
    deleteLabels: function (labelListTab, labels) {
        var list = labelListTab.down('viison_pickware_erp_barcode_label_printing-main-list');
        var store = list.store;

        list.setLoading(true);

        store.remove(labels);
        store.sync({
            scope: this,
            success: function (callback) {
                list.setLoading(false);

                if (Ext.isFunction(callback)) {
                    callback(true);
                }
            },
            failure: function () {
                list.setLoading(false);
                store.rejectChanges();

                Shopware.Notification.createGrowlMessage(
                    this.getViisonSnippet('notification/delete_selection/error/title'),
                    this.getViisonSnippet('notification/delete_selection/error/message'),
                    'Pickware'
                );
            },
        });
    },

    /**
     * Gets the field values of the settings form and uses them to create an URL parameter
     * string containing its values, which is then returned.
     *
     * @param form
     * @return A string containing URL parameters.
     */
    getFormParameters: function (form) {
        // Get all values from the main form
        var values = form.down('[name="main-form"]').getForm().getFieldValues();

        // Use the values to create the default params
        return 'x=' + values.startX + '&y=' + values.startY + '&template=' + values.template;
    },

});
