// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPBinLocationEditor.controller.Main', {

    extend: 'Ext.app.Controller',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_bin_location_editor/main',

    /**
     * @Override
     */
    init: function () {
        this.control({
            'viison_pickware_erp_bin_location_editor-main': {
                cancel: this.onCancel,
                save: this.onSave,
            },
            'viison_pickware_erp_bin_location_editor-main viison_pickware_erp_bin_location_editor-main-bin_location_field': {
                editBinLocation: this.onEditBinLocation,
            },
            'viison_pickware_erp_bin_location_editor-bin_location_creation-panel': {
                cancel: this.onCancelBinLocationCreation,
                save: this.onSaveNewBinLocation,
            },
            'viison_pickware_erp_bin_location_editor-bin_location_selection': {
                cancel: this.onCancelBinLocationSelection,
                save: this.onSaveSelectedBinLocation,
            },
            'viison_pickware_erp_bin_location_editor-bin_location_selection-bin_location_list': {
                binLocationSelectionChanged: this.onBinLocationSelectionChanged,
                createBinLocation: this.onCreateBinLocation,
            },
            'viison_pickware_erp_bin_location_editor-bin_location_selection-bin_location_mapping_list': {
                binLocationMappingSelectionChanged: this.onBinLocationMappingSelectionChanged,
            },
        });

        // Create and load the comment store
        this.commentStore = Ext.create('Shopware.apps.ViisonPickwareERPBinLocationEditor.store.Comment', {});
        this.commentStore.load();

        this.callParent(arguments);
    },

    /**
     * Creates, shows and returns a new editor window, which is passed the given article detail id,
     * default purchase price, bin location and the comment store as well as the optional callback.
     *
     * @param int articleDetailId
     * @param float defaultPurchasePrice
     * @param Object warehouse
     * @param Shopware.apps.ViisonPickwareERPArticleWarehouseFieldSet.store.BinLocationArticleDetail binLocationMappingStore
     * @param Function callback (optional)
     * @param Shopware.apps.ViisonPickwareERPArticleWarehouseFieldSet.model.BinLocationArticleDetail selectedBinLocationMapping (optional)
     */
    createEditor: function (
        articleDetailId,
        defaultPurchasePrice,
        warehouse,
        binLocationMappingStore,
        callback,
        selectedBinLocationMapping
    ) {
        return this.getView('Shopware.apps.ViisonPickwareERPBinLocationEditor.view.Main').create({
            articleDetailId: articleDetailId,
            defaultPurchasePrice: defaultPurchasePrice,
            warehouse: warehouse,
            binLocationMappingStore: binLocationMappingStore,
            selectedBinLocationMapping: selectedBinLocationMapping,
            commentStore: this.commentStore,
            saveCallback: callback || Ext.emptyFn,
        }).show();
    },

    /**
     * @param Shopware.apps.ViisonPickwareERPBinLocationEditor.view.Main window
     */
    onCancel: function (window) {
        window.close();
    },

    /**
     * Gets the values of the active tab's form and uses them to save the respective stock changes. Upon success, the
     * article detail's inStock, physical stock and reserved stock are updated and the editor window is closed.
     *
     * @param Shopware.apps.ViisonPickwareERPBinLocationEditor.view.Main window
     */
    onSave: function (window) {
        var articleDetailId = window.articleDetailId;
        var defaultPurchasePrice = window.defaultPurchasePrice;
        var warehouse = window.warehouse;
        var panel = window.items.first().getActiveTab();
        var formValues = panel.getForm().getValues();

        // Validate form
        if (!panel.getForm().isValid()) {
            return;
        }

        // Determine the URL and parameters for the request as well as the notification messages
        var url;
        var notificationMessages;
        var params = {
            articleDetailId: articleDetailId,
            comment: formValues.comment,
            warehouseId: warehouse.id,
            warehouseConfigId: warehouse.articleDetailConfigurations.id,
            purchasePrice: defaultPurchasePrice,
        };
        switch (panel.itemId) {
            case 'incoming':
                url = ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPBinLocationEditor/saveIncomingStock');
                params.incomingStock = parseInt(formValues.incomingStock, 10);
                if (formValues.purchasePrice) {
                    params.purchasePrice = parseFloat(formValues.purchasePrice);
                }
                params.binLocationId = (formValues.binLocation) ? formValues.binLocation.id : warehouse.nullBinLocationId;
                notificationMessages = {
                    successTitle: this.getViisonSnippet('notification/save_stock_changes/success/title'),
                    failureTitle: this.getViisonSnippet('notification/save_stock_changes/failure/title'),
                    successMessage: this.getViisonSnippet('notification/save_stock_changes/incoming/success/message'),
                    failureMessage: this.getViisonSnippet('notification/save_stock_changes/incoming/failure/message'),
                };
                break;
            case 'outgoing':
                url = ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPBinLocationEditor/saveOutgoingStock');
                params.outgoingStock = parseInt(formValues.outgoingStock, 10);
                params.binLocationId = formValues.binLocation;
                notificationMessages = {
                    successTitle: this.getViisonSnippet('notification/save_stock_changes/success/title'),
                    failureTitle: this.getViisonSnippet('notification/save_stock_changes/failure/title'),
                    successMessage: this.getViisonSnippet('notification/save_stock_changes/outgoing/success/message'),
                    failureMessage: this.getViisonSnippet('notification/save_stock_changes/outgoing/failure/message'),
                };
                break;
            case 'relocation':
                url = ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPBinLocationEditor/saveRelocation');
                params.relocatedStock = parseInt(formValues.relocatedStock, 10);
                params.sourceBinLocationId = formValues.sourceBinLocation;
                var destinationWarehouse = panel.getForm().findField('destinationWarehouse').getSelectedRecord();
                params.destinationBinLocationId = (formValues.destinationBinLocation) ? formValues.destinationBinLocation.id : destinationWarehouse.get('nullBinLocationId');
                notificationMessages = {
                    successTitle: this.getViisonSnippet('notification/save_stock_changes/success/title'),
                    failureTitle: this.getViisonSnippet('notification/save_stock_changes/failure/title'),
                    successMessage: this.getViisonSnippet('notification/save_stock_changes/relocation/success/message'),
                    failureMessage: this.getViisonSnippet('notification/save_stock_changes/relocation/failure/message'),
                };
                break;
            case 'warehouse_stock_limits':
                url = ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPBinLocationEditor/saveStockLimits');
                params.minimumStock = parseInt(panel.down('#minimumStockField').value, 10);
                params.targetStock = parseInt(panel.down('#targetStockField').value, 10);
                notificationMessages = {
                    successTitle: this.getViisonSnippet('notification/save_stock_changes/stock_limits/success/title'),
                    failureTitle: this.getViisonSnippet('notification/save_stock_changes/stock_limits/failure/title'),
                    successMessage: this.getViisonSnippet('notification/save_stock_changes/stock_limits/success/message'),
                    failureMessage: this.getViisonSnippet('notification/save_stock_changes/stock_limits/failure/message'),
                };
                break;
            default:
                return;
        }

        // Send the request
        window.setLoading(true);
        Ext.Ajax.request({
            method: 'POST',
            url: url,
            params: params,
            scope: this,
            callback: function (options, success, response) {
                // Decode and validate response
                var json = response.responseText && Ext.JSON.decode(response.responseText, true);
                success = success && json && json.success;
                if (!success) {
                    window.setLoading(false);
                    Shopware.Notification.createGrowlMessage(
                        notificationMessages.failureTitle,
                        notificationMessages.failureMessage,
                        'ViisonPickwareERP'
                    );

                    return;
                }

                Shopware.Notification.createGrowlMessage(
                    notificationMessages.successTitle,
                    notificationMessages.successMessage,
                    'ViisonPickwareERP'
                );

                // Close editor
                window.close();
                if (Ext.isFunction(window.saveCallback)) {
                    window.saveCallback(params.purchasePrice, json.data.inStock, json.data.pickwarePhysicalStockForSale, json.data.stockMin);
                }
            },
        });
    },

    /**
     * Creates and shows a new bin location editor window, which is passed the selected warehouse, as well as a callback
     * that sets the selected bin location in the respective field.
     *
     * @param Shopware.apps.ViisonPickwareERPBinLocationEditor.view.main.BinLocationField field
     */
    onEditBinLocation: function (field) {
        var warehouse = field.warehouse;
        var binLocationMappingStore = field.up('viison_pickware_erp_bin_location_editor-main').binLocationMappingStore;
        var binLocationMappings;
        Ext.each(binLocationMappingStore.getGroups(), function (group) {
            if (group.name === warehouse.id) {
                binLocationMappings = group.children;

                return false;
            }

            return undefined;
        });

        // Create a callback, which updates the bin location displayed in the panel
        var callback = function (newBinLocation) {
            field.setValue(newBinLocation);
        };

        this.getView('Shopware.apps.ViisonPickwareERPBinLocationEditor.view.BinLocationSelection').create({
            binLocationMappings: binLocationMappings,
            warehouse: warehouse,
            saveCallback: callback,
        }).show();
    },

    /**
     * @param Shopware.apps.ViisonPickwareERPBinLocationEditor.view.BinLocationCreation.Panel panel
     */
    onCancelBinLocationCreation: function (panel) {
        panel.up('viison_pickware_erp_bin_location_editor-bin_location_creation').close();
    },

    /**
     * Creates a new temporary bin location with the code entered in the given panel and syncs it to the server. If the
     * sync is successful, the selection list is added a search query for entered code to reload the list and show only
     * the newly created bin location (otherwise it might end up on a random results page).
     *
     * @param Shopware.apps.ViisonPickwareERPBinLocationEditor.view.BinLocationCreation.Panel panel
     */
    onSaveNewBinLocation: function (panel) {
        var window = panel.up('viison_pickware_erp_bin_location_editor-bin_location_creation');
        var selectionWindow = window.selectionWindow;
        if (!panel.getForm().isValid()) {
            return;
        }

        // Close the creation window
        window.close();

        // Create a new bin location using the form data
        var binLocationCode = panel.getForm().getValues().code;
        var binLocation = Ext.create('Shopware.apps.ViisonPickwareERPWarehouseManagement.model.BinLocation', {});
        binLocation.set('code', binLocationCode);
        binLocation.set('warehouseId', selectionWindow.warehouse.id);

        // Add the new bin location to the store and save it
        selectionWindow.setLoading(true);
        selectionWindow.binLocationStore.add(binLocation);
        selectionWindow.binLocationStore.sync({
            callback: function (batch) {
                selectionWindow.setLoading(false);
                // Check the response
                var operation = batch.operations[0];
                if (operation.success) {
                    // Trigger a search for the create location
                    var searchField = selectionWindow.down('textfield[name=searchfield]');
                    searchField.setValue(binLocation.get('code'));
                } else {
                    // Revert changes and show error notification
                    selectionWindow.binLocationStore.rejectChanges();
                    var duplicateLocation = operation.request.proxy.reader.jsonData.uniqueConstraintViolation;
                    var message = duplicateLocation ? this.getViisonSnippet('notification/save_new_bin_location/error/message/duplicate_location') : this.getViisonSnippet('notification/save_new_bin_location/error/message/unknown_error');

                    message = Ext.String.format(message, binLocationCode);
                    Shopware.Notification.createGrowlMessage(
                        this.getViisonSnippet('notification/save_new_bin_location/error/title'),
                        message,
                        'ViisonPickwareERP'
                    );
                }
            },
        });
    },

    /**
     * @param Shopware.apps.ViisonPickwareERPBinLocationEditor.view.BinLocationSelection window
     */
    onCancelBinLocationSelection: function (window) {
        window.close();
    },

    /**
     * Closes the given window and, if it has a saveCallback, calls it by passing the currently selection bin location.
     *
     * @param Shopware.apps.ViisonPickwareERPBinLocationEditor.view.BinLocationSelection window
     */
    onSaveSelectedBinLocation: function (window) {
        var binLocationMappingList = window.down('viison_pickware_erp_bin_location_editor-bin_location_selection-bin_location_mapping_list');
        var binLocationList = window.down('viison_pickware_erp_bin_location_editor-bin_location_selection-bin_location_list');
        var binLocation;
        if (binLocationMappingList.getSelectionModel().getSelection().length > 0) {
            var binLocationMapping = binLocationMappingList.getSelectionModel().getSelection()[0];
            binLocation = Ext.create('Shopware.apps.ViisonPickwareERPWarehouseManagement.model.BinLocation', {
                id: binLocationMapping.get('binLocation').id,
                warehouseId: binLocationMapping.get('binLocation').warehouseId,
                code: binLocationMapping.get('binLocation').code,
                warehouse: binLocationMapping.get('warehouse'),
            });
        } else {
            binLocation = binLocationList.getSelectionModel().getSelection()[0];
        }

        window.close();
        if (Ext.isFunction(window.saveCallback)) {
            window.saveCallback(binLocation);
        }
    },

    /**
     * Creates and shows a new window bin location creation window.
     *
     * @param Shopware.apps.ViisonPickwareERPBinLocationEditor.view.BinLocationSelection.BinLocationList panel
     */
    onCreateBinLocation: function (panel) {
        this.getView('Shopware.apps.ViisonPickwareERPBinLocationEditor.view.BinLocationCreation').create({
            selectionWindow: panel.up('viison_pickware_erp_bin_location_editor-bin_location_selection'),
        }).show();
    },

    /**
     * Enables/disables the save button based on the current selection.
     *
     * @param Shopware.apps.ViisonPickwareERPBinLocationEditor.view.BinLocationSelection.BinLocationList panel
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.model.BinLocation[] selectedBinLocations
     */
    onBinLocationSelectionChanged: function (panel, selectedBinLocations) {
        var window = panel.up('viison_pickware_erp_bin_location_editor-bin_location_selection');
        if (selectedBinLocations.length > 0) {
            var binLocationMappingList = window.down('viison_pickware_erp_bin_location_editor-bin_location_selection-bin_location_mapping_list');
            binLocationMappingList.getSelectionModel().deselectAll(true);
        }
        var saveButton = window.child('toolbar[dock=bottom]').down('button[action=save]');
        saveButton.setDisabled(selectedBinLocations.length === 0);
    },

    /**
     * Enables/disables the save button based on the current selection.
     *s
     * @param Shopware.apps.ViisonPickwareERPBinLocationEditor.view.BinLocationSelection.BinLocationMappingList panel
     * @param Shopware.apps.ViisonPickwareERPArticleWarehouseFieldSet.model.BinLocationArticleDetail[] selectedBinLocationMappings
     */
    onBinLocationMappingSelectionChanged: function (panel, selectedBinLocationMappings) {
        var window = panel.up('viison_pickware_erp_bin_location_editor-bin_location_selection');
        if (selectedBinLocationMappings.length > 0) {
            var binLocationList = window.down('viison_pickware_erp_bin_location_editor-bin_location_selection-bin_location_list');
            binLocationList.getSelectionModel().deselectAll(true);
        }
        var saveButton = window.child('toolbar[dock=bottom]').down('button[action=save]');
        saveButton.setDisabled(selectedBinLocationMappings.length === 0);
    },

});
