// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPBinLocationEditor.view.main.BinLocationField', {

    extend: 'Ext.form.field.Text',
    alias: 'widget.viison_pickware_erp_bin_location_editor-main-bin_location_field',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_bin_location_editor/main',

    /**
     * Set the field read only by default, to allow editing it only via the edit and clear button.
     */
    readOnly: true,
    /**
     * Set this field to the raw data of the warehouse whose bin locations should be available when editing this field.
     */
    warehouse: null,

    /**
     * @Override
     */
    initComponent: function () {
        this.emptyText = this.getViisonSnippet('null_bin_location');
        if (Ext.isObject(this.value)) {
            // Save the full value and change the actual value property to avoid problems due to conversions
            // in the parent initializer
            this.selectedBinLocation = this.value;
            this.value = this.formatBinLocationDisplayText(this.value);
        }

        this.on('afterrender', function () {
            this.addEditIcon();
            this.addClearIcon();
        }, this);

        this.callParent(arguments);
    },

    /**
     * @Override
     */
    setValue: function (value) {
        // Only pass the 'code' to the parent implementation
        if (Ext.isObject(value)) {
            this.selectedBinLocation = value;
            value = this.formatBinLocationDisplayText(value);
        } else if (!value) {
            this.selectedBinLocation = null;
        }

        // Update the visibility of the clear icon
        if (this.clearIcon) {
            this.clearIcon[(this.selectedBinLocation) ? 'show' : 'hide']();
        }

        this.callParent([value]);
    },

    /**
     * @Override
     */
    getSubmitData: function () {
        // Return an object containing the full selected bin location
        var result = {};
        result[this.name] = (this.selectedBinLocation) ? this.selectedBinLocation.getData() : null;

        return result;
    },

    /**
     * @Override
     */
    getModelData: function () {
        return this.getSubmitData();
    },

    /**
     * Adds a new 'pencil' icon button to this text field.
     */
    addEditIcon: function () {
        // Check for an existing edit icon and remove it
        try {
            if (this.editIcon) {
                this.editIcon.removeListener('click');
                this.editIcon.remove();
            }
        } catch (e) {
            // Ignore
        }

        this.editIcon = this.addIcon(
            'sprite-pencil',
            this.getViisonSnippet('main/bin_location_field/button/edit/tooltip'),
            function () {
                this.fireEvent('editBinLocation', this);
            }
        );
    },

    /**
     * Adds a new 'minus' icon button to this text field.
     */
    addClearIcon: function () {
        // Check for an existing clear icon and remove it
        try {
            if (this.clearIcon) {
                this.clearIcon.removeListener('click');
                this.clearIcon.remove();
            }
        } catch (e) {
            // Ignore
        }

        this.clearIcon = this.addIcon(
            'sprite-minus-circle-frame',
            this.getViisonSnippet('main/bin_location_field/button/clear/tooltip'),
            function () {
                this.setValue(null);
            },
            20
        );

        // Hide the clear icon, if no bin location is set (yet)
        if (!this.selectedBinLocation) {
            this.clearIcon.hide();
        }
    },

    /**
     * Creates a new 'span' element containing an icon having the given 'iconCls' and 'tooltip'
     * and adds it to this text field. That is, it is added 'on top' of the fields content,
     * positioned on the right end of the field.
     *
     * @param String iconCls
     * @param String tooltip
     * @param Function handler
     * @param Number rightMargin (optional)
     * @return Ext.Element
     */
    addIcon: function (iconCls, tooltip, handler, rightMargin) {
        // Create and add the edit icon
        var icon = new Ext.Element(document.createElement('span'));
        icon.set({
            cls: iconCls,
            style: 'position: absolute; top: 6px; right: ' + ((rightMargin || 0) + 6) + 'px; width: 16px; height: 16px; z-index: 1; display: block; cursor: pointer;',
            'data-qtip': tooltip,
        });
        if (this.getEl()) {
            this.getEl().setStyle('position', 'relative');
        }
        icon.addListener('click', function () {
            handler.apply(this);
        }, this);
        icon.insertAfter(this.inputEl);

        return icon;
    },

    /**
     * @param BinLocation binLocation
     * @returns string binLocation's code, or "Unknown bin location" if it's the null bin location
     */
    formatBinLocationDisplayText: function (binLocation) {
        return binLocation.get('isNullBinLocation') ? this.getViisonSnippet('null_bin_location') : binLocation.get('code');
    },

});
