// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPCancellation.controller.Editor', {

    extend: 'Ext.app.Controller',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_cancellation/main',

    refs: [
        { ref: 'editor', selector: 'viison_pickware_erp_cancellation-editor' },
        { ref: 'itemLists', selector: 'viison_pickware_erp_cancellation-editor-items-item_list' },
    ],

    /**
     * @type {Shopware.apps.ViisonPickwareERPCancellation.view.Window}
     */
    mainWindow: null,

    /**
     * Optional callback, which will be called after the cancellation has been processed successfully.
     *
     * @type {function}
     */
    afterCancellationCallback: Ext.emptyFn,

    /**
     * @type {Shopware.apps.ViisonPickwareERPCancellation.model.Cancellation}
     */
    cancellationRecord: null,

    /**
     * @type {boolean}
     */
    columnUpdateActive: false,

    /**
     * @override
     */
    init: function () {
        this.callParent(arguments);

        this.control({
            'viison_pickware_erp_cancellation-editor': {
                closeWindow: this.onCloseWindow,
                previewDocument: this.onPreviewDocument,
                performCancellation: this.onPerformCancellation,
            },
            'viison_pickware_erp_cancellation-editor-footer': {
                shippingCostsCheckboxChanged: this.onShippingCostsCheckboxChanged,
            },
            'viison_pickware_erp_cancellation-editor-items-item_list_container': {
                disableEnableItemGroup: this.onDisableEnableItemGroup,
                addPseudoItem: this.onAddPseudoItem,
            },
            'viison_pickware_erp_cancellation-editor-items-item_list': {
                itemChanged: this.onItemChanged,
                columnhide: this.onColumnHide,
                columnshow: this.onColumnShow,
                deletePseudoItem: this.onDeletePseudoItem,
            },
            'viison_pickware_erp_cancellation-editor-document_sidebar field': {
                change: this.onChangeDocumentData,
            },
        });
    },

    /**
     * @override
     *
     * Scope refs to main window
     *
     * @param {string} selector
     * @returns {Ext.Component}
     */
    getActiveReference: function (selector) {
        return Ext.ComponentQuery.query(selector, this.mainWindow)[0];
    },

    /**
     * @returns {Shopware.apps.ViisonPickwareERPCancellation.model.Cancellation}
     */
    getCancellationRecord: function () {
        return this.cancellationRecord;
    },

    /**
     * @param {number} itemGroupId
     * @returns {Shopware.apps.ViisonPickwareERPCancellation.model.ItemGroup}
     */
    getItemGroupById: function (itemGroupId) {
        return this.cancellationRecord.getItemGroups().getById(itemGroupId);
    },

    /**
     * @param {number} orderId
     * @param {number} returnShipmentId
     * @param {boolean} autoPreselectAllReturnShipments
     * @param {number[]} preselectedOrderDetailIds
     * @param {function} afterCancellationCallback
     */
    createWindow: function (orderId, returnShipmentId, autoPreselectAllReturnShipments, preselectedOrderDetailIds, afterCancellationCallback) {
        if (typeof afterCancellationCallback !== 'undefined') {
            this.afterCancellationCallback = afterCancellationCallback;
        }

        this.mainWindow = this.getView('Window').create();
        this.mainWindow.show();

        this.loadData(orderId, returnShipmentId, autoPreselectAllReturnShipments, preselectedOrderDetailIds);
    },

    /**
     * @param {number} orderId
     * @param {number} returnShipmentId
     * @param {boolean} autoPreselectAllReturnShipments
     * @param {number[]} preselectedOrderDetailIds
     */
    loadData: function (orderId, returnShipmentId, autoPreselectAllReturnShipments, preselectedOrderDetailIds) {
        this.mainWindow.showLoadMask();

        // Chain loading of the taxes store and the cancellation record, in order to ensure
        // that all data is available when rendering the window content
        this.getStore('Tax').load({
            scope: this,
            callback: function () {
                this.getModel('Cancellation').load(-1, {
                    params: {
                        orderId: orderId,
                        returnShipmentId: returnShipmentId,
                        autoPreselectAllReturnShipments: autoPreselectAllReturnShipments ? 1 : 0,
                        preselectedOrderDetailIds: JSON.stringify(preselectedOrderDetailIds),
                    },
                    scope: this,
                    callback: function (cancellationRecord) {
                        this.setCancellationRecord(cancellationRecord);

                        setTimeout(function () {
                            this.mainWindow.hideLoadMask();
                        }.bind(this), 400);
                    },
                });
            },
        });
    },

    /**
     * @param {Shopware.apps.ViisonPickwareERPCancellation.model.Cancellation} cancellation
     */
    setCancellationRecord: function (cancellation) {
        this.cancellationRecord = cancellation;
        this.cancellationRecord.recalculateTotalAmount();

        this.mainWindow.add(this.getView('Editor').create());
        this.updateVisualizationOfColumnRefundQuantity();
    },

    onCloseWindow: function () {
        this.mainWindow.close();
    },

    onPreviewDocument: function () {
        var cancellationData = this.cancellationRecord.getProxy().getWriter().getRecordData(this.cancellationRecord);

        this.openWindowWithPost(
            ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPCancellation/showDocumentPreview'),
            cancellationData
        );
    },

    onPerformCancellation: function () {
        this.mainWindow.showLoadMask(this.getViisonSnippet('window/loading_mask/perform_cancellation/text'));

        this.cancellationRecord.save({
            scope: this,
            callback: function (data, operation) {
                if (operation.success) {
                    Shopware.Notification.createGrowlMessage(
                        this.getViisonSnippet('notification/perform_cancellation/success/title'),
                        Ext.String.format(
                            this.getViisonSnippet('notification/perform_cancellation/success/message'),
                            this.cancellationRecord.get('orderNumber')
                        ),
                        'ViisonPickwareERP'
                    );
                    this.afterCancellationCallback();
                    this.mainWindow.close();
                } else {
                    Shopware.Notification.createGrowlMessage(
                        this.getViisonSnippet('notification/perform_cancellation/failure/title'),
                        Ext.String.format(
                            this.getViisonSnippet('notification/perform_cancellation/failure/message'),
                            this.cancellationRecord.get('orderNumber'),
                            operation.request.proxy.getReader().rawData.message
                        ),
                        'ViisonPickwareERP'
                    );
                }
                this.mainWindow.hideLoadMask();
            },
        });
    },

    /**
     * Updates the cancellation record, when the user enables / disables an item group.
     *
     * @param {Shopware.apps.ViisonPickwareERPCancellation.view.editor.items.ItemListContainer} itemListContainer
     * @param {boolean} enabled
     * @param {Shopware.apps.ViisonPickwareERPCancellation.model.ItemGroup} itemGroup
     */
    onDisableEnableItemGroup: function (itemListContainer, enabled, itemGroup) {
        itemGroup.set('cancelItemGroup', enabled);
        this.cancellationRecord.recalculateTotalAmount();
    },

    /**
     * @param {Shopware.apps.ViisonPickwareERPCancellation.view.editor.items.ItemListContainer} itemListContainer
     * @param {Shopware.apps.ViisonPickwareERPCancellation.model.ItemGroup} itemGroup
     */
    onAddPseudoItem: function (itemListContainer, itemGroup) {
        itemGroup.getItems().add(
            Ext.create(
                'Shopware.apps.ViisonPickwareERPCancellation.model.Item',
                {
                    isPseudoItem: true,
                    isDeletable: true,
                    articleNumber: '',
                    articleName: 'Optionale Retourenkosten etc.',
                    quantity: 1,
                    cancelQuantity: null,
                    refundQuantity: 1,
                    taxId: this.getStore('Tax').first().get('id'),
                    price: 0.0,
                }
            )
        );
    },

    /**
     * Asks the user if a given pseudo item should indeed be deleted and, if the user confirms,
     * removed it from the list of pseudo items.
     *
     * @param {Shopware.apps.ViisonPickwareERPCancellation.view.editor.items.ItemList} itemList
     * @param {Shopware.apps.ViisonPickwareERPCancellation.model.Item} pseudoItem
     */
    onDeletePseudoItem: function (itemList, pseudoItem) {
        var store = itemList.getStore();

        Ext.Msg.confirm(
            this.getViisonSnippet('confirm/delete_pseudo_item/title'),
            this.getViisonSnippet('confirm/delete_pseudo_item/text'),
            function (decision) {
                if (decision !== 'yes') {
                    return;
                }

                store.remove(pseudoItem);
                itemList.displayPseudoItemsInfoIfNecessary();
            },
            this
        );
    },

    /**
     * Updates the cancellation record, when the user changes the checked state of the shipping
     * costs checkbox.
     *
     * @param {Ext.form.field.Checkbox} checkbox
     * @param {boolean} value
     */
    onShippingCostsCheckboxChanged: function (checkbox, value) {
        this.cancellationRecord.set('cancelShippingCosts', value);
        this.cancellationRecord.recalculateTotalAmount();
    },

    onItemChanged: function () {
        this.cancellationRecord.recalculateTotalAmount();

        // After a value of a record in a row has changed other values may change as well due to conditions and effects
        // of multiple properties of the record (e.g. changing 'refundQuantity' value may change the 'quantity' value as
        // well). This behavior in combination with conditional CSS class during render causes view bugs after editing a
        // row. Hence we need to refresh the grid view after a row was edited. This causes the rows to rerender
        // correctly.
        var itemLists = this.getEditor().query('viison_pickware_erp_cancellation-editor-items-item_list');
        Ext.each(itemLists, function (itemList) {
            itemList.getView().refresh();
        });
    },

    /**
     * Updates the column state of all item list, when the user hides a column in one list.
     *
     * @param {Ext.grid.header.Container} columnHeader
     * @param {Ext.grid.column.Column} column
     */
    onColumnHide: function (columnHeader, column) {
        this.updateColumnState(column, function (itemListColumn) {
            itemListColumn.setVisible(false);
        });
    },

    /**
     * Updates the column state of all item list, when the user shows a column in one list.
     *
     * @param {Ext.grid.header.Container} columnHeader
     * @param {Ext.grid.column.Column} column
     */
    onColumnShow: function (columnHeader, column) {
        this.updateColumnState(column, function (itemListColumn) {
            itemListColumn.setVisible(true);
        });
    },

    /**
     * Applies the state of a given item list column to the respective column of all other item lists.
     *
     * @param {Ext.grid.column.Column} column
     * @param {function} callback
     */
    updateColumnState: function (column, callback) {
        if (this.columnUpdateActive) {
            return;
        }

        this.columnUpdateActive = true;
        var itemLists = this.getEditor().query('viison_pickware_erp_cancellation-editor-items-item_list');
        Ext.each(itemLists, function (itemList) {
            var itemListColumn = this.findColumnIndex(itemList.columns, column.dataIndex);
            if (column.id !== itemListColumn.id) {
                itemListColumn.suspendEvents();
                callback(itemListColumn);
                itemListColumn.resumeEvents();
            }
        }, this);
        this.getEditor().doLayout();
        this.columnUpdateActive = false;
    },

    /**
     * @param {Ext.grid.column.Column[]} columns
     * @param {number} dataIndex
     * @returns {number}
     */
    findColumnIndex: function (columns, dataIndex) {
        var columnWithSearchedDataIndex = null;

        Ext.each(columns, function (column) {
            if (column.dataIndex === dataIndex) {
                columnWithSearchedDataIndex = column;
            }
        }, this);

        return columnWithSearchedDataIndex;
    },

    /**
     * @param {Ext.form.field.Field} field
     * @param {mixed} newValue
     */
    onChangeDocumentData: function (field, newValue) {
        this.cancellationRecord.updateDocumentData(field.name, newValue);
        this.updateVisualizationOfColumnRefundQuantity();
    },

    updateVisualizationOfColumnRefundQuantity: function () {
        if (this.cancellationRecord.getDocumentData().get('createDocument')) {
            this.getEditor().removeCls('is--create-no-document-mode');
        } else {
            this.getEditor().addCls('is--create-no-document-mode');
        }
    },

    /**
     * Opens a given url in a new browser window / tab passing the given data object as
     * jsonified POST data by creating and submitting a temporary form element.
     *
     * @param {string} url
     * @param {object} data
     */
    openWindowWithPost: function (url, data) {
        var form = document.createElement('form');
        form.target = '_blank';
        form.method = 'POST';
        form.action = url;
        form.style.display = 'none';

        var input = document.createElement('input');
        input.type = 'hidden';
        input.name = 'data';
        input.value = JSON.stringify(data);
        form.appendChild(input);

        document.body.appendChild(form);
        form.submit();
        document.body.removeChild(form);
    },

});
