// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPCancellation.model.Cancellation', {

    extend: 'Ext.data.Model',

    idProperty: 'orderId',

    fields: [
        { name: 'orderId', type: 'int' },
        { name: 'orderNumber', type: 'string' },
        { name: 'shippingCosts', type: 'float' },
        { name: 'shippingCostsGross', type: 'float' },
        { name: 'shippingCostsNet', type: 'float' },
        { name: 'shippingCostsTaxRate', type: 'float' },
        { name: 'cancelShippingCosts', type: 'boolean' },
        { name: 'totalAmount', type: 'float', default: 0.0 },
    ],

    associations: [
        {
            type: 'hasOne',
            model: 'Shopware.apps.ViisonPickwareERPCancellation.model.DocumentData',
            associationKey: 'documentData',
            name: 'getDocumentData',
            getterName: 'getDocumentData',
            setterName: 'setDocumentData',
        }, {
            type: 'hasMany',
            model: 'Shopware.apps.ViisonPickwareERPCancellation.model.ItemGroup',
            associationKey: 'itemGroups',
            name: 'getItemGroups',
        },
    ],

    proxy: {
        type: 'ajax',
        api: {
            read: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPCancellation/getCancellationData'),
            update: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPCancellation/performCancellation'),
        },
        reader: {
            type: 'json',
            root: 'data',
            totalProperty: 'total',
        },
        writer: Ext.create(
            'Shopware.apps.ViisonCommonData.proxy.writer.WriterWithSupportForHasOneAssociations',
            {
                root: 'data',
            }
        ),
    },

    /**
     * @returns {boolean}
     */
    hasShippingCosts: function () {
        return this.get('shippingCosts') !== 0.0;
    },

    recalculateTotalAmount: function () {
        var totalAmount = 0.0;

        this.getItemGroups().each(function (itemGroup) {
            totalAmount += itemGroup.getTotalAmount();
        }, this);

        if (this.get('cancelShippingCosts')) {
            totalAmount += this.get('shippingCosts');
        }

        this.set('totalAmount', totalAmount);
        this.fireEvent('totalAmountChanged', this.get('totalAmount'));
    },

    /**
     *
     * @param {string} key
     * @param {mixed} value
     */
    updateDocumentData: function (key, value) {
        this.getDocumentData().set(key, value);
        if (key === 'createDocument') {
            this.fireEvent('createDocumentChanged', value, this);
        }
    },

    /**
     * @returns {boolean}
     */
    containsItemsToBeCancelled: function () {
        var containsItemsToBeCancelled = false;

        this.getItemGroups().each(function (itemGroup) {
            if (itemGroup.containsItemsToBeCancelled()) {
                containsItemsToBeCancelled = true;
            }
        }, this);

        return containsItemsToBeCancelled;
    },

});
