// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPCancellation.view.editor.items.ItemList', {

    extend: 'Ext.grid.Panel',
    alias: 'widget.viison_pickware_erp_cancellation-editor-items-item_list',
    cls: 'viison_pickware_erp_cancellation-editor-items-item_list viison-common--grid has--vertical-lines',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
        'Shopware.apps.ViisonPickwareERPCancellation.mixin.DataRetrieval',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_cancellation/main',

    stateful: true,
    stateId: 'viison_pickware_erp_cancellation-editor-items-item_list',

    selModel: 'viisonCommonTabableCellModel',

    border: false,
    bodyStyle: {
        border: 'none !important',
    },

    viewConfig: {
        enableTextSelection: true,
        trackOver: true,
        selectedItemCls: '',
        selectedCellCls: '',
        markDirty: false,
        getRowClass: function (record, rowIndex) {
            this.removeRowCls(rowIndex, 'is--inactive');
            if (record.get('isPseudoItem') && record.get('quantity') === 0) {
                return 'is--inactive';
            }

            return '';
        },
        listeners: {
            render: function (view) {
                view.refundQuantityTooltip = Ext.create('Ext.tip.ToolTip', {
                    target: view.el,
                    delegate: '.x-grid-cell.has--icon.is--warning ',
                    trackMouse: true,
                    renderTo: Ext.getBody(),
                    html: ViisonCommonApp.getSnippet(
                        'item_list/columns/refund_quantity/tooltip',
                        'backend/viison_pickware_erp_cancellation/main'
                    ),
                });
            },
            beforedestroy: function (view) {
                if (view.refundQuantityTooltip) {
                    view.refundQuantityTooltip.destroy();
                }
            },
        },
    },

    /**
     * @type {number}
     */
    itemGroupId: null,

    /**
     * @override
     */
    initComponent: function () {
        var itemGroupRecord = this.getItemGroupById(this.itemGroupId);

        this.store = itemGroupRecord.getItems();
        this.columns = this.createColumns();
        this.plugins = this.createPlugins();
        this.disabled = itemGroupRecord.isDisabled();

        if (itemGroupRecord.isTypePseudo()) {
            this.dockedItems = [
                {
                    xtype: 'container',
                    dock: 'bottom',
                    itemId: 'pseudoItemsInfo',
                    cls: 'is--item-list-info--pseudo-items',
                    hidden: this.store.getCount() > 0,
                    html: '<span>' + this.getViisonSnippet('item_list/info/pseudo_items') + '</span>',
                },
            ];
        }

        this.callParent(arguments);
    },

    /**
     * @returns {object[]}
     */
    createColumns: function () {
        var itemGroupRecord = this.getItemGroupById(this.itemGroupId);

        var editableColumns = [];
        if (itemGroupRecord.isTypeReturnShipment()) {
            editableColumns.push('refundQuantity');
        } else if (itemGroupRecord.isTypePseudo()) {
            editableColumns = editableColumns.concat([
                'articleNumber',
                'articleName',
                'quantity',
                'refundQuantity',
                'taxId',
                'price',
            ]);
        } else {
            editableColumns = editableColumns.concat([
                'cancelQuantity',
                'refundQuantity',
            ]);
        }

        var actionColumnItems = [];
        if (itemGroupRecord.isTypePseudo()) {
            actionColumnItems.push({
                iconCls: 'sprite-minus-circle',
                tooltip: this.getViisonSnippet('item_list/columns/button/delete_pseudo_item/tooltip'),
                scope: this,
                getClass: function (value, meta, record) {
                    if (!record.get('isDeletable')) {
                        return 'x-hide-visibility';
                    }

                    return undefined;
                },
                handler: function (grid, rowIndex, columnIndex, column, event) {
                    var returnShipmentItem = grid.getStore().getAt(rowIndex);

                    if (event.keyCode === 13) {
                        return;
                    }

                    if (returnShipmentItem.get('isDeletable')) {
                        this.fireEvent('deletePseudoItem', this, returnShipmentItem);
                    }
                },
            });
        }

        return [
            {
                header: this.getViisonSnippet('item_list/columns/article_number'),
                dataIndex: 'articleNumber',
                width: 100,
                tdCls: editableColumns.indexOf('articleNumber') === -1 ? null : 'is--editable',
                editor: editableColumns.indexOf('articleNumber') === -1 ? null : {
                    xtype: 'textfield',
                    cls: 'is--viison-common-grid--cell-editor',
                    allowBlank: true,
                    enableKeyEvents: true,
                },
            },
            {
                header: this.getViisonSnippet('item_list/columns/article_name'),
                dataIndex: 'articleName',
                flex: 1,
                tdCls: editableColumns.indexOf('articleName') === -1 ? null : 'is--editable',
                editor: editableColumns.indexOf('articleName') === -1 ? null : {
                    xtype: 'textfield',
                    cls: 'is--viison-common-grid--cell-editor',
                    allowBlank: true,
                    enableKeyEvents: true,
                },
            },
            {
                header: this.getViisonSnippet('item_list/columns/quantity'),
                dataIndex: 'quantity',
                align: 'right',
                width: 80,
                tdCls: editableColumns.indexOf('quantity') === -1 ? null : 'is--editable',
                editor: editableColumns.indexOf('quantity') === -1 ? null : {
                    xtype: 'numberfield',
                    cls: 'is--viison-common-grid--cell-editor is--align-right',
                    allowBlank: true,
                    enableKeyEvents: true,
                },
            },
            {
                header: this.getViisonSnippet('item_list/columns/cancel_quantity'),
                dataIndex: 'cancelQuantity',
                tdCls: editableColumns.indexOf('cancelQuantity') === -1 ? null : 'is--editable',
                align: 'right',
                width: 80,
                editor: editableColumns.indexOf('cancelQuantity') === -1 ? null : {
                    xtype: 'numberfield',
                    cls: 'is--viison-common-grid--cell-editor is--align-right',
                    allowBlank: false,
                    minValue: 0,
                    enableKeyEvents: true,
                },
                renderer: function (value) {
                    if (itemGroupRecord.isTypePseudo()) {
                        return '-';
                    }

                    return value;
                },
            },
            {
                header: this.getViisonSnippet('item_list/columns/refund_quantity'),
                dataIndex: 'refundQuantity',
                tdCls: 'is--relevant-for-document-creation ' + (editableColumns.indexOf('refundQuantity') === -1 ? null : 'is--editable'),
                align: 'right',
                width: 80,
                editor: editableColumns.indexOf('refundQuantity') === -1 ? null : {
                    xtype: 'numberfield',
                    cls: 'is--viison-common-grid--cell-editor is--align-right',
                    allowBlank: false,
                    minValue: 0,
                    enableKeyEvents: true,
                },
                renderer: function (value, metaData, record) {
                    if (!itemGroupRecord.isTypePseudo() && record.get('cancelQuantity') !== value) {
                        metaData.tdCls = 'has--icon is--warning';
                    }

                    return value;
                },
            },
            {
                header: this.getViisonSnippet('item_list/columns/tax_rate'),
                dataIndex: 'taxId',
                tdCls: 'is--relevant-for-document-creation ' + (editableColumns.indexOf('taxId') === -1 ? null : 'is--editable'),
                align: 'right',
                width: 70,
                editor: editableColumns.indexOf('taxId') === -1 ? null : {
                    xtype: 'combobox',
                    cls: 'is--viison-common-grid--cell-editor',
                    queryMode: 'local',
                    allowBlank: false,
                    store: this.getTaxesStore(),
                    displayField: 'name',
                    valueField: 'id',
                },
                renderer: function (value, metaData, record) {
                    if (value !== null) {
                        var taxRecord = this.getTaxesStore().getById(value);
                        if (taxRecord) {
                            return taxRecord.get('name');
                        }
                    }

                    return record.get('taxRate') + '%';
                },
            },
            {
                header: this.getViisonSnippet('item_list/columns/price'),
                dataIndex: 'price',
                align: 'right',
                width: 90,
                renderer: ViisonCurrencyFormatter.renderer,
                tdCls: 'is--relevant-for-document-creation ' + (editableColumns.indexOf('price') === -1 ? null : 'is--editable'),
                editor: editableColumns.indexOf('price') === -1 ? null : {
                    xtype: 'numberfield',
                    cls: 'is--viison-common-grid--cell-editor is--align-right',
                    allowBlank: true,
                    decimalPrecision: 2,
                    decimalSeparator: ',',
                    enableKeyEvents: true,
                },
            },
            {
                header: this.getViisonSnippet('item_list/columns/total_price'),
                dataIndex: 'totalPrice',
                tdCls: 'is--relevant-for-document-creation',
                align: 'right',
                width: 90,
                renderer: ViisonCurrencyFormatter.renderer,
            },
            {
                xtype: 'actioncolumn',
                width: 30,
                align: 'center',
                items: actionColumnItems,
            },
        ];
    },

    /**
     * @returns {Ext.grid.plugin.CellEditing[]}
     */
    createPlugins: function () {
        return [
            Ext.create('Ext.grid.plugin.CellEditing', {
                clicksToEdit: 1,
                listeners: {
                    scope: this,
                    beforeedit: this.onBeforeEdit,
                    edit: this.onEdit,
                },
            }),
        ];
    },

    /**
     * Applies the max value constraints for quantity and cancel quantity.
     *
     * @param {Ext.grid.plugin.CellEditing} plugin
     * @param {object} editorEvent
     */
    onBeforeEdit: function (plugin, editEvent) {
        var editor = plugin.getEditor(editEvent.record, editEvent.column);

        if (editEvent.field === 'cancelQuantity') {
            editor.field.setMaxValue(editEvent.record.get('quantity'));
        }
    },

    /**
     * Applies the new values to the edited item.
     *
     * @param {Ext.grid.plugin.CellEditing} plugin
     * @param {object} editorEvent
     */
    onEdit: function (plugin, editorEvent) {
        this.fireEvent('itemChanged', editorEvent.record);
    },

    displayPseudoItemsInfoIfNecessary: function () {
        var itemGroupRecord = this.getItemGroupById(this.itemGroupId);

        if (itemGroupRecord.isTypePseudo()) {
            if (this.getStore().getCount() > 0) {
                this.down('#pseudoItemsInfo').setVisible(false);
            } else {
                this.down('#pseudoItemsInfo').setVisible(true);
            }
        }
    },

});
