// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPOrder.controller.Detail', {

    override: 'Shopware.apps.Order.controller.Detail',

    /* eslint-disable no-unused-vars */
    /**
     * @override
     *
     * Ensures that all relevant parts of the order detail window are reloaded, when a user clicks
     * one of the several save buttons of the oder detail window. For this purpose the default
     * callback used by Shopware when saving a order record is patched by chaining another callback,
     * which reloads all relevant data.
     *
     * @param {Shopware.apps.Order.model.Order} orderRecord
     * @param {string} successMessage
     * @param {string} errorMessage
     * @param {object} options
     */
    saveRecord: function (orderRecord, successMessage, errorMessage, options) {
        var detailWindow = this.getDetailWindow();
        var updateDetailWindow = this.viisonPickwareERPUpdateDetailWindow;

        var originalOrderSaveMethod = orderRecord.save;
        orderRecord.save = function (saveOptions) {
            var originalCallback = saveOptions.callback;
            saveOptions.callback = function (data, operation) {
                /* eslint-disable no-empty */
                // Use try-catch to circumvent a bug in (at least) Shopware <= 5.5, which throws an error, when a
                // user saves the order of an order detail window, which is not bound to the order overview window
                // (in detail: in case that the order list store does not exist).
                try {
                    originalCallback.call(this, data, operation);
                } catch (error) {
                }
                /* eslint-enable no-empty */
                updateDetailWindow.call(this, orderRecord, detailWindow);
            };

            originalOrderSaveMethod.call(this, saveOptions);
        };

        this.callParent(arguments);

        orderRecord.save = originalOrderSaveMethod;
    },
    /* eslint-enable no-unused-vars */

    /**
     * Updates a given order detail window by reloading all order data
     * and updating relevant parts of the order detail window.
     *
     * @param {Shopware.apps.Order.model.Order} orderRecord
     * @param {Shopware.apps.Order.view.detail.Window} orderRecord
     */
    viisonPickwareERPUpdateDetailWindow: function (orderRecord, detailWindow) {
        detailWindow.setLoading(true);

        // ExtJs automatically shares proxies via stores of the same type,
        // hence we need to backup initial proxy parameters, if they exist
        var extraParamBackup = null;
        if (orderRecord.store && orderRecord.store.getProxy().extraParams) {
            extraParamBackup = orderRecord.store.getProxy().extraParams;
        }

        // Load order data and update relevant parts
        var tempOrderStore = Ext.create('Shopware.apps.Order.store.Order');
        tempOrderStore.getProxy().extraParams = {
            orderID: orderRecord.get('id'),
        };
        tempOrderStore.load({
            callback: function (records) {
                var record = records[0];

                // Update order positions
                orderRecord.getPositions().loadRecords(record.getPositions().getRange(), {
                    addRecords: false, // Remove old records first
                });

                // Update order documents
                orderRecord.getReceipt().loadRecords(record.getReceipt().getRange(), {
                    addRecords: false, // Remove old records first
                });

                // Update invoice amount
                orderRecord.set('invoiceAmount', record.get('invoiceAmount'));
                orderRecord.set('invoiceAmountNet', record.get('invoiceAmountNet'));

                // Update shipping costs
                orderRecord.set('invoiceShipping', record.get('invoiceShipping'));
                orderRecord.set('invoiceShippingNet', record.get('invoiceShippingNet'));

                // Rerender order positions and order documents lists
                detailWindow.doLayout();

                // Update order details view (overview tab)
                var overviewPanel = detailWindow.query('order-overview-panel');
                if (overviewPanel.length > 0) {
                    overviewPanel[0].detailsForm.loadRecord(orderRecord); // Update invoice amount
                    overviewPanel[0].editForm.loadRecord(orderRecord); // Update shipping costs
                }

                detailWindow.setLoading(false);
            },
        });

        // Reset proxy parameters
        tempOrderStore.getProxy().extraParams = extraParamBackup;
    },
});
