// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPOrderDocumentCreation.controller.Detail', {
    override: 'Shopware.apps.Order.controller.Detail',
    viisonPickwareERPOrderDocumentCreationSkipPickListAlert: false,

    /**
     * If a pick list is about to be created, we show a warehouse selection dialog and inject the resulting warehouse id
     * into the store. When the store is synced and the documents are created, the warehouse id will be used.
     *
     * @param order
     * @param config
     * @param panel
     */
    onCreateDocument: function (order, config, panel) {
        if (this.viisonPickwareERPOrderDocumentCreationSkipPickListAlert) {
            this.callParent(arguments);
            this.viisonPickwareERPOrderDocumentCreationSkipPickListAlert = false;

            return;
        }

        if (!this.isPickListDocumentSelected(panel)) {
            this.callParent(arguments);

            return;
        }

        ViisonPickwareBinLocationSelectionAlert.selectWarehouse(
            ViisonCommonApp.getSnippet('warehouseSelection/title', 'backend/viison_pickware_erp_order_document_creation/pick_list'),
            ViisonCommonApp.getSnippet('warehouseSelection/message', 'backend/viison_pickware_erp_order_document_creation/pick_list'),
            Ext.Msg.OKCANCEL,
            function (buttonId, warehouse) {
                if (buttonId !== 'ok') {
                    return;
                }

                ViisonPickwareERPOrderEventBus.on('collectBatchStoreParams', function (store) {
                    store.getProxy().extraParams.pickListWarehouseId = warehouse.get('id');
                }, this, {
                    single: true,
                });

                this.viisonPickwareERPOrderDocumentCreationSkipPickListAlert = true;
                this.onCreateDocument(order, config, panel);
            },
            this
        );
    },

    /**
     * If a pick list is about to be previewed, we show a warehouse selection dialog and use the resulting warehouse id
     * to pass it to the backend order controller (no store sync is used). Call parent otherwise.
     *
     * @param order
     * @param config
     * @param panel
     */
    onDocumentPreview: function (order, config, panel) {
        if (!this.isPickListDocumentSelected(panel)) {
            this.callParent(arguments);

            return;
        }

        ViisonPickwareBinLocationSelectionAlert.selectWarehouse(
            ViisonCommonApp.getSnippet('warehouseSelection/title', 'backend/viison_pickware_erp_order_document_creation/pick_list'),
            ViisonCommonApp.getSnippet('warehouseSelection/previewMessage', 'backend/viison_pickware_erp_order_document_creation/pick_list'),
            Ext.Msg.OKCANCEL,
            function (buttonId, warehouse) {
                if (buttonId !== 'ok') {
                    return;
                }

                // Since Shopware just calls the backend controller with a 'preview' parameter (instead of
                // syncing the store), we have to overwrite this call manually
                window.open(
                    ViisonCommonApp.assembleBackendUrl('Order/createDocument')
                    + '?orderId=' + order.get('id')
                    + '&preview=1'
                    + '&taxFree=' + config.get('taxFree')
                    + '&temp=1'
                    + '&documentType=' + config.get('documentType')
                    + '&pickListWarehouseId=' + warehouse.get('id')
                );
            },
            this
        );
    },

    isPickListDocumentSelected: function (panel) {
        var pickListDocumentId = ViisonCommonApp.getConfig('viisonPickListDocumentId', 'ViisonPickwareERPOrderDocumentCreation');
        if (!pickListDocumentId) {
            // Signal the backend user if the pick list document template could not be found (e.g. template was renamed)
            Shopware.Notification.createGrowlMessage(
                ViisonCommonApp.getSnippet('pickListNotFound/growl/title', 'backend/viison_pickware_erp_order_document_creation/pick_list'),
                ViisonCommonApp.getSnippet('pickListNotFound/growl/message', 'backend/viison_pickware_erp_order_document_creation/pick_list'),
                'ViisonPickwareERPOrderDocumentCreation'
            );
        }
        var formValues = panel.getValues();

        return formValues.documentType === pickListDocumentId;
    },
});
