// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPOrderDocumentDeletion.controller.Detail', {

    override: 'Shopware.apps.Order.controller.Detail',

    /**
     * @Override
     */
    init: function () {
        this.callParent(arguments);
        this.control({
            'order-document-list': {
                deleteDocument: this.onDeleteDocument,
            },
        });
    },

    /**
     * If the third, optional parameter is given and true, the given document is deleted
     * and removed from the local store. Otherwise a confirm alert is presented first.
     *
     * @param Shopware.apps.Order.view.list.Document panel
     * @param Shopware.apps.Order.model.Receipt record
     * @param Boolean skipConfirmation (optional)
     */
    onDeleteDocument: function (panel, record, skipConfirmation) {
        var documentName = record.getDocType().getAt(0).get('name') + ' ' + record.get('documentId');
        if (skipConfirmation !== true) {
            // Ask for confirmation
            var title = ViisonCommonApp.getSnippet(
                'alert/confirm_deletion/title',
                'backend/viison_pickware_erp_order_document_deletion/main'
            );
            var text = Ext.String.format(
                ViisonCommonApp.getSnippet(
                    'alert/confirm_deletion/text',
                    'backend/viison_pickware_erp_order_document_deletion/main'
                ),
                documentName
            );
            Ext.Msg.confirm(title, text, function (decision) {
                if (decision === 'yes') {
                    this.onDeleteDocument(panel, record, true);
                }
            }, this);

            return;
        }

        // Delete the document
        panel.setLoading(true);
        Ext.Ajax.request({
            method: 'POST',
            url: ViisonCommonApp.assembleBackendUrl(
                'ViisonPickwareERPOrderDocumentDeletion/deleteDocument'
            ),
            params: {
                documentId: record.get('id'),
            },
            callback: function (options, success, response) {
                panel.setLoading(false);
                var responseData = Ext.JSON.decode(response.responseText);
                if (responseData && responseData.success) {
                    // Remove document from local store
                    panel.getStore().remove(record);

                    var successTitle = ViisonCommonApp.getSnippet(
                        'notification/success/title',
                        'backend/viison_pickware_erp_order_document_deletion/main'
                    );
                    var message = Ext.String.format(
                        ViisonCommonApp.getSnippet(
                            'notification/success/message',
                            'backend/viison_pickware_erp_order_document_deletion/main'
                        ),
                        documentName
                    );
                    Shopware.Msg.createGrowlMessage(successTitle, message, 'Pickware');
                } else {
                    Shopware.Msg.createGrowlMessage(
                        ViisonCommonApp.getSnippet(
                            'notification/failure/title',
                            'backend/viison_pickware_erp_order_document_deletion/main'
                        ),
                        ViisonCommonApp.getSnippet(
                            'notification/failure/message',
                            'backend/viison_pickware_erp_order_document_deletion/main'
                        ),
                        'Pickware'
                    );
                }
            },
        });
    },

});
