// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPOrderDocumentMailer.controller.Detail', {

    override: 'Shopware.apps.Order.controller.Detail',

    /**
     * @Override
     */
    init: function () {
        this.callParent(arguments);
        this.control({
            'order-document-list': {
                sendDocument: this.onSendDocument,
            },
        });
    },

    /**
     * Fetches the email template for the given order document and uses it to create a
     * new mail composer window for sending the document.
     *
     * @param Shopware.apps.Order.view.list.Document panel
     * @param Shopware.apps.Order.model.Receipt orderDocument
     */
    onSendDocument: function (panel, orderDocument) {
        var order = panel.up('order-detail-window').record;
        var documentType = orderDocument.getDocType().first();

        // Get the email template
        panel.up('order-detail-window').setLoading(true);
        Ext.Ajax.request({
            method: 'POST',
            url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPOrderDocumentMailer/getMail'),
            params: {
                documentId: orderDocument.getId(),
            },
            scope: this,
            callback: function (options, success, response) {
                panel.up('order-detail-window').setLoading(false);
                // Check response
                var responseJson = Ext.JSON.decode(response.responseText, true);
                if (!responseJson || !responseJson.success || !responseJson.mail) {
                    Shopware.Notification.createGrowlMessage(
                        ViisonCommonApp.getSnippet(
                            'notification/get_mail_template/failure/title',
                            'backend/viison_pickware_erp_order_document_mailer/main'
                        ),
                        ViisonCommonApp.getSnippet(
                            'notification/get_mail_template/failure/message',
                            'backend/viison_pickware_erp_order_document_mailer/main'
                        ),
                        'Pickware'
                    );

                    return;
                }

                // Open a new mail composer showing the received email
                var onSendMail = this.onSendMail.bind(this);
                Ext.create('Shopware.apps.ViisonCommonMailComposer.view.Window', {
                    mail: responseJson.mail,
                    sendMail: function () {
                        onSendMail(this, order, orderDocument, documentType);
                    },
                }).show();
            },
        });
    },

    /**
     * Sends the email that has been composed in the given mail composer window.
     *
     * @param Shopware.apps.ViisonCommonMailComposer.view.Window mailWindow
     * @param Shopware.apps.Order.model.Order order
     * @param Shopware.apps.Order.model.Receipt orderDocument
     * @param Shopware.apps.Base.model.DocType documentType
     */
    onSendMail: function (mailWindow, order, orderDocument, documentType) {
        // Send the email
        mailWindow.setLoading(true);
        Ext.Ajax.request({
            method: 'POST',
            url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPOrderDocumentMailer/send'),
            jsonData: {
                mail: mailWindow.formData,
                documentId: orderDocument.getId(),
            },
            callback: function (options, success, response) {
                mailWindow.setLoading(false);
                // Check response
                var responseJson = Ext.JSON.decode(response.responseText, true);
                success = responseJson && responseJson.success;
                if (success) {
                    mailWindow.close();
                }

                // Determine notification title and message
                var title;
                var message;
                if (success) {
                    title = ViisonCommonApp.getSnippet(
                        'notification/send_mail/success/title',
                        'backend/viison_pickware_erp_order_document_mailer/main'
                    );
                    message = ViisonCommonApp.getSnippet(
                        'notification/send_mail/success/message',
                        'backend/viison_pickware_erp_order_document_mailer/main'
                    );
                } else {
                    title = ViisonCommonApp.getSnippet(
                        'notification/send_mail/failure/title',
                        'backend/viison_pickware_erp_order_document_mailer/main'
                    );
                    message = ViisonCommonApp.getSnippet(
                        'notification/send_mail/failure/message',
                        'backend/viison_pickware_erp_order_document_mailer/main'
                    );
                }
                var documentName = documentType.get('name') + ' ' + orderDocument.get('documentId');
                message = Ext.String.format(
                    message,
                    documentName,
                    order.get('number'),
                    mailWindow.formData.toAddress
                );

                // Show the notification
                Shopware.Notification.createGrowlMessage(title, message, 'Pickware');
            },
        });
    },

});
