// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPOrderOverwriteProtection.controller.Detail', {

    override: 'Shopware.apps.Order.controller.Detail',

    /**
     * @override
     */
    onSavePosition: function (_, event, order) {
        // This method is the tunnel entry of 'tunneling' the 'pickwareLastChanged' date to the request parameters of the
        // store. The orderRecord is used as tunnel. See for
        // Shopware.apps.ViisonPickwareERPOrderOverwriteProtection.model.Position for tunnel exit.
        var positionStore = event.store;

        event.record.viisonParamsForSaveRequest = {
            // Use ISO format to also commit the time zone and avoid problem because of mismatching timezones of client
            // and server.
            pickwareLastChanged: order.get('pickwareLastChanged').toISOString(),
        };

        positionStore.addListener('write', function (store, operation) {
            if (!operation.success) {
                return;
            }
            var response = Ext.JSON.decode(operation.response.responseText);
            if (response.data.orderId !== order.get('id')) {
                return;
            }

            order.set('pickwareLastChanged', response.pickwareLastChanged);
        }, this, {
            single: true,
        });

        this.callParent(arguments);
    },

    /**
     * @override
     */
    onDeleteMultiplePositions: function (order, grid) {
        // This method is the tunnel entry of 'tunneling' the 'pickwareLastChanged' date to the request parameters of the
        // store. The orderRecord is used as tunnel. See for
        // Shopware.apps.ViisonPickwareERPOrderOverwriteProtection.model.Position for tunnel exit.
        var store = grid.getStore();

        var originalSyncClosure = store.sync.bind(store);
        store.sync = function (options) {
            options = options || {};

            this.getProxy().extraParams = this.getProxy().extraParams || {};
            Ext.apply(this.getProxy().extraParams, this.viisonParamsForSyncRequest);

            originalSyncClosure(options);
            store.sync = originalSyncClosure;
        };

        store.viisonParamsForSyncRequest = {
            // Use ISO format to also commit the time zone and avoid problem because of mismatching timezones of client
            // and server.
            pickwareLastChanged: order.get('pickwareLastChanged').toISOString(),
        };

        store.addListener('write', function (_, operation) {
            if (!operation.success) {
                return;
            }

            var updatedOrder = Ext.JSON.decode(operation.response.responseText).data;
            if (updatedOrder.id !== order.get('id')) {
                return;
            }

            order.set('pickwareLastChanged', updatedOrder.attribute.pickwareLastChanged);
        }, this, {
            single: true,
        });

        this.callParent(arguments);
    },

    /**
     * @override
     */
    onSaveExternalComment: function (orderRecord) {
        this.viisonAddChangeDateUpdateListener(orderRecord);

        this.callParent(arguments);
    },

    /**
     * @override
     */
    onSaveInternalComment: function (orderRecord) {
        this.viisonAddChangeDateUpdateListener(orderRecord);

        this.callParent(arguments);
    },

    /**
     * @override
     */
    saveRecord: function (orderRecord) {
        this.viisonAddChangeDateUpdateListener(orderRecord);

        this.callParent(arguments);
    },

    /**
     * Adds a one-time listener to the store of the orderRecord that automatically updates the 'pickwareLastChanged'
     * property after a store 'write' event.
     *
     * @param {Shopware.apps.Order.model.Order} orderRecord
     */
    viisonAddChangeDateUpdateListener: function (orderRecord) {
        var orderStore = orderRecord.store || this.getOrderList().store;

        // Yeah, Shopware does something terribly wrong so that the `store` property of the `orderRecord` may be null.
        // Restore the property here. It has something to do with the refresh of the store after successfully saving the
        // order.
        if (orderRecord.store === null) {
            orderRecord.store = orderStore;
        }
        if (orderRecord.stores.indexOf(orderStore) === -1) {
            orderRecord.stores.push(orderStore);
        }

        // Add a listener to the orderStore. The listener updates the "pickwareLastChanged" timestamp in the current
        // order detail window after the order has been saved. The "pickwareLastChanged" timestamp is used for optimistic
        // locking.
        orderStore.addListener('write', function (store, operation) {
            if (!operation.success) {
                return;
            }
            var updatedOrder = Ext.JSON.decode(operation.response.responseText);
            if (updatedOrder.id !== orderRecord.get('id')) {
                return;
            }

            orderRecord.set('pickwareLastChanged', updatedOrder.pickwareLastChanged);
        }, this, {
            single: true,
        });
    },
});
