// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPOrderReturnShipment.controller.Tab', {

    extend: 'Ext.app.Controller',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_order_return_shipment/main',

    refs: [
        { ref: 'returnShipmentsTab', selector: 'viison_pickware_erp_order_return_shipment-tab' },
        { ref: 'returnShipmentsList', selector: 'viison_pickware_erp_order_return_shipment-return_shipment_list' },
    ],

    /**
     * @type {Shopware.apps.Order.model.Order}
     */
    orderRecord: null,

    /**
     * @override
     */
    init: function () {
        this.control({
            'viison_pickware_erp_order_return_shipment-tab': {
                addReturnShipment: this.onAddReturnShipment,
                cancelReturnShipments: this.onCancelReturnShipments,
                finalizeReturnShipments: this.onFinalizeReturnShipments,
                reloadReturnShipments: this.onReloadReturnShipments,
                show: this.onActivated,
            },
            'viison_pickware_erp_order_return_shipment-return_shipment_list': {
                viewready: this.onReturnShipmentListViewReady,
                itemclick: this.onReturnShipmentListItemClicked,
                editReturnShipment: this.onReturnShipmentListEditReturnShipment,
                deleteReturnShipment: this.onReturnShipmentListDeleteReturnShipment,
                openDocument: this.onOpenDocument,
            },
            'viison_pickware_erp_order_return_shipment-return_shipment_item_list': {
                openArticle: this.onReturnShipmentItemListOpenArticle,
            },
        });

        this.callParent(arguments);
    },

    /**
     * @param {Shopware.apps.Order.model.Order} order
     * @param {Enlight.app.SubApplication} subApp
     * @return {Shopware.apps.ViisonPickwareERPOrderReturnShipment.view.detail.Tab}
     */
    createReturnShipmentsTab: function (order, subApp) {
        this.orderRecord = order;

        var returnShipmentsStore = this.getStore(
            'Shopware.apps.ViisonPickwareERPOrderReturnShipment.store.ReturnShipment'
        );
        returnShipmentsStore.setOrderId(order.get('id'));
        returnShipmentsStore.on('load', this.onLoadReturnShipmentsStore, this);

        return Ext.create('Shopware.apps.ViisonPickwareERPOrderReturnShipment.view.detail.Tab', {
            returnShipmentsStore: returnShipmentsStore,
            subApp: subApp,
        });
    },

    /**
     * Updates the state of the batch processing buttons (cancel and finalize) based on the content of the given
     * store of return shipments.
     *
     * @param {Shopware.apps.ViisonPickwareERPOrderReturnShipment.store.ReturnShipment} store
     */
    onLoadReturnShipmentsStore: function (store) {
        if (store.containsAtLeastOneCancellableReturnShipment()) {
            this.getReturnShipmentsTab().enableCancelButton();
        } else {
            this.getReturnShipmentsTab().disableCancelButton();
        }

        if (store.containsAtLeastOneFinalizableReturnShipment()) {
            this.getReturnShipmentsTab().enableFinalizeButton();
        } else {
            this.getReturnShipmentsTab().disableFinalizeButton();
        }
    },

    onReturnShipmentListViewReady: function () {
        this.loadReturnShipmentsStore();
    },

    /**
     *
     * @param {Ext.grid.Panel} panel
     * @param {Shopware.apps.ViisonPickwareERPOrderReturnShipment.model.ReturnShipment} returnShipment
     */
    onReturnShipmentListItemClicked: function (panel, returnShipment) {
        this.getReturnShipmentsTab().showReturnShipmentItemList(returnShipment.getItems());
    },

    onAddReturnShipment: function () {
        Shopware.app.Application.addSubApplication({
            name: 'Shopware.apps.ViisonPickwareERPReturnShipment',
            action: 'detail',
            orderId: this.orderRecord.get('id'),
            onUpdateCallback: this.loadReturnShipmentsStore.bind(this),
        });
    },

    onCancelReturnShipments: function () {
        Shopware.app.Application.addSubApplication({
            name: 'Shopware.apps.ViisonPickwareERPCancellation',
            orderId: this.orderRecord.get('id'),
            autoPreselectAllReturnShipments: true,
            afterCancellationCallback: this.loadReturnShipmentsStore.bind(this),
        });
    },

    onFinalizeReturnShipments: function () {
        var returnShipmentsToBeFinalized = this.getReturnShipmentsList().getStore().getFinalizableReturnShipments();
        if (returnShipmentsToBeFinalized.length === 0) {
            return;
        }

        var returnShipmentIds = [];
        var returnShipmentNumbers = [];
        Ext.Array.forEach(returnShipmentsToBeFinalized, function (returnShipment) {
            returnShipmentIds.push(returnShipment.get('id'));
            returnShipmentNumbers.push(returnShipment.get('number'));
        });

        var message = Ext.String.format(
            this.getViisonSnippet('confirm/finalize_return_shipments/text'),
            returnShipmentIds.length
        );

        var returnShipmentNotInStatusCancelledOrFinalized = returnShipmentsToBeFinalized.find(function (returnShipment) {
            return !returnShipment.isTypeCancelled() && !returnShipment.isTypeFinalized();
        });
        if (returnShipmentNotInStatusCancelledOrFinalized) {
            message += '<br /><br />';
            message += this.getViisonSnippet('confirm/finalize_return_shipments/warning');
        }

        var createStatusMailSendingCheckbox = Ext.create('Ext.form.field.Checkbox', {
            boxLabel: this.getViisonSnippet('confirm/finalize_return_shipments/automatic_status_mail_sending'),
            checked: true,
            style: {
                'margin-left': '20px',
            },
        });

        Ext.create('Shopware.apps.ViisonPickwareERPMessageBoxWithCheckboxes.view.MessageBoxWithCheckboxes').show({
            title: this.getViisonSnippet('confirm/finalize_return_shipments/title'),
            icon: Ext.Msg.QUESTION,
            msg: message,
            checkboxes: [
                createStatusMailSendingCheckbox,
            ],
            buttons: Ext.Msg.YESNO,
            scope: this,
            callback: function (decision) {
                if (decision !== 'yes') {
                    return;
                }

                this.getReturnShipmentsTab().showLoadMask(this.getViisonSnippet('action/finalize_return_shipments/loading_message'));
                Ext.Ajax.request({
                    url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPReturnShipment/batchFinalizeReturnShipments'),
                    method: 'POST',
                    params: {
                        returnShipmentIds: Ext.JSON.encode(returnShipmentIds),
                        shouldSendStatusMails: createStatusMailSendingCheckbox.checked,
                    },
                    scope: this,
                    callback: function (options, success, response) {
                        var result = Ext.JSON.decode(response.responseText, true);

                        if (result && result.success) {
                            Shopware.Notification.createGrowlMessage(
                                this.getViisonSnippet('growl_message/finalize_return_shipments/success/title'),
                                Ext.String.format(
                                    this.getViisonSnippet('growl_message/finalize_return_shipments/success/message'),
                                    returnShipmentNumbers.join(', '),
                                    returnShipmentIds.join(', ')
                                ),
                                'ViisonPickwareERP'
                            );
                        } else {
                            Shopware.Notification.createGrowlMessage(
                                this.getViisonSnippet('growl_message/finalize_return_shipments/failure/title'),
                                Ext.String.format(
                                    this.getViisonSnippet('growl_message/finalize_return_shipments/failure/message'),
                                    returnShipmentNumbers.join(', '),
                                    returnShipmentIds.join(', '),
                                    result.message
                                ),
                                'ViisonPickwareERP'
                            );
                        }

                        this.getReturnShipmentsTab().hideLoadMask();
                        this.loadReturnShipmentsStore();
                    },
                });
            },
        });
    },

    onReloadReturnShipments: function () {
        this.loadReturnShipmentsStore();
    },

    /**
     * @param {Shopware.apps.ViisonPickwareERPOrderReturnShipment.model.ReturnShipment} returnShipment
     */
    onReturnShipmentListEditReturnShipment: function (returnShipment) {
        Shopware.app.Application.addSubApplication({
            name: 'Shopware.apps.ViisonPickwareERPReturnShipment',
            action: 'detail',
            returnShipmentId: returnShipment.get('id'),
            onUpdateCallback: this.loadReturnShipmentsStore.bind(this),
        });
    },

    /**
     * Asks the user if a given return shipment should indeed be deleted and, if the user confirms,
     * deletes it by calling the respective server action.
     *
     * @param {Shopware.apps.ViisonPickwareERPOrderReturnShipment.model.ReturnShipment} returnShipmentToBeDeleted
     */
    onReturnShipmentListDeleteReturnShipment: function (returnShipmentToBeDeleted) {
        var message = Ext.String.format(
            this.getViisonSnippet('confirm/delete_return_shipment/text'),
            returnShipmentToBeDeleted.get('number'),
            returnShipmentToBeDeleted.get('id')
        );
        if (returnShipmentToBeDeleted.get('documentId') !== null) {
            message += '<br /><br />' + this.getViisonSnippet('confirm/delete_return_shipment/text_cancellation_invoice');
        }
        Ext.Msg.confirm(
            this.getViisonSnippet('confirm/delete_return_shipment/title'),
            message,
            function (decision) {
                if (decision !== 'yes') {
                    return;
                }

                this.getReturnShipmentsTab().showLoadMask(
                    this.getViisonSnippet('action/delete_return_shipment/loading_message')
                );
                Ext.Ajax.request({
                    url: ViisonCommonApp.assembleBackendUrl(
                        'ViisonPickwareERPOrderReturnShipment/deleteReturnShipment'
                    ),
                    method: 'POST',
                    params: {
                        returnShipmentId: returnShipmentToBeDeleted.get('id'),
                    },
                    scope: this,
                    callback: function (options, success, response) {
                        var result = Ext.JSON.decode(response.responseText, true);

                        if (result && result.success) {
                            Shopware.Notification.createGrowlMessage(
                                this.getViisonSnippet('growl_message/delete_return_shipment/success/title'),
                                Ext.String.format(
                                    this.getViisonSnippet('growl_message/delete_return_shipment/success/message'),
                                    returnShipmentToBeDeleted.get('number'),
                                    returnShipmentToBeDeleted.get('id')
                                ),
                                'ViisonPickwareERP'
                            );
                        } else {
                            Shopware.Notification.createGrowlMessage(
                                this.getViisonSnippet('growl_message/delete_return_shipment/failure/title'),
                                Ext.String.format(
                                    this.getViisonSnippet('growl_message/delete_return_shipment/failure/message'),
                                    returnShipmentToBeDeleted.get('number'),
                                    returnShipmentToBeDeleted.get('id'),
                                    result.message
                                ),
                                'ViisonPickwareERP'
                            );
                        }

                        this.getReturnShipmentsTab().hideLoadMask();
                        this.loadReturnShipmentsStore();
                    },
                });
            },
            this
        );
    },

    /**
     * @param {Shopware.apps.ViisonPickwareERPOrderReturnShipment.model.ReturnShipmentItem} returnShipmentItem
     */
    onReturnShipmentItemListOpenArticle: function (returnShipmentItem) {
        Shopware.app.Application.addSubApplication({
            name: 'Shopware.apps.Article',
            action: 'detail',
            params: {
                articleId: returnShipmentItem.get('articleId'),
            },
        });
    },

    loadReturnShipmentsStore: function () {
        var tab = this.getReturnShipmentsTab();

        if (tab) {
            tab.showLoadMask();
            tab.showReturnShipmentItemsInfo();
            tab.hideReturnShipmentItemsContainer();
            this.getStore('Shopware.apps.ViisonPickwareERPOrderReturnShipment.store.ReturnShipment').load({
                scope: this,
                callback: function (records) {
                    if (records.length > 0) {
                        tab.showReturnShipmentItemsContainer();
                    } else {
                        tab.hideReturnShipmentItemsContainer();
                    }
                    tab.hideLoadMask();
                },
            });
        }
    },

    onActivated: function () {
        this.toggleAddReturnShipmentButtonActive();
    },

    toggleAddReturnShipmentButtonActive: function () {
        var tab = this.getReturnShipmentsTab();
        var aReturnableOrderDetail = this.orderRecord.getPositions().getRange().find(function (orderDetail) {
            return orderDetail.get('shipped') > orderDetail.get('pickwareReturnedQuantity');
        });

        if (aReturnableOrderDetail) {
            tab.enableAddReturnShipmentButton();
        } else {
            tab.disableAddReturnShipmentButton();
        }
    },

    /**
     * Opens the cancellation Invoice document of a given return shipment in a new browser window / tab.
     *
     * @param {Shopware.apps.ViisonPickwareERPOrderReturnShipment.model.ReturnShipment} returnShipment
     */
    onOpenDocument: function (returnShipment) {
        window.open(ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPReturnShipment/openCorrectionOfInvoice', {
            documentId: returnShipment.get('documentId'),
        }));
    },

});
