// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPOrderShippedColumn.controller.Detail', {

    override: 'Shopware.apps.Order.controller.Detail',

    /**
     * A flag used to skip the alert that is asking whether the shipped quantity shall be changed.
     */
    skipAlert: false,

    /**
     * @Override
     */
    onAddPosition: function (record, grid) {
        this.callParent(arguments);
        /**
         * Set the shipped value of the freshly added row to 0 (it would be 'undefined' otherwise even if it is
         * displayed as '0' in the backend
         */
        var currentRecord = grid.getSelectionModel().getSelection()[0];
        currentRecord.raw.shipped = 0;
    },

    /**
     * @Override
     */
    onArticleSelect: function (editor, value, record) {
        this.callParent(arguments);
        // Transfer ArticleDetail attribute 'pickwareStockManagementDisabled' to new position as 'pickwareIsStockManaged'
        editor.context.record.set('pickwareIsStockManaged', !record.get('pickwareStockManagementDisabled'));
    },

    /**
     * @Override
     */
    onSavePosition: function (editor, context, order, options) {
        var isStockManaged = context.record.get('pickwareIsStockManaged');
        var newShipped = context.newValues.shipped;
        var shippedChanged = newShipped !== context.record.raw.shipped;
        if (this.skipAlert || !isStockManaged || !shippedChanged || (context.record.phantom && newShipped <= 0)) {
            this.skipAlert = false;
            this.callParent(arguments);

            return;
        }

        // Ask user for position update
        ViisonPickwareBinLocationSelectionAlert.selectWarehouse(
            ViisonCommonApp.getSnippet(
                'alert/title',
                'backend/viison_pickware_erp_order_shipped_column/main'
            ),
            ViisonCommonApp.getSnippet(
                'alert/message',
                'backend/viison_pickware_erp_order_shipped_column/main'
            ),
            Ext.Msg.OKCANCEL,
            function (buttonId, warehouse) {
                if (buttonId !== 'ok') {
                    // Continue editing
                    editor.startEdit(context.record, editor.context.column);

                    return;
                }

                // Add the ID of the selected warehouse to the order proxy's extra params
                var proxy = context.record.getProxy();
                proxy.extraParams = proxy.extraParams || {};
                proxy.extraParams.stockChangeWarehouseId = warehouse.get('id');

                // Prepare a callback for updating the raw record data
                var customOptions = options || {};
                var originalCallback = options.callback;
                customOptions.callback = function (orderRecord) {
                    context.record.raw.shipped = context.newValues.shipped;
                    if (Ext.isFunction(originalCallback)) {
                        originalCallback(orderRecord);
                    }
                };

                // Save the orders by calling the onSavePosition method again, but this time skipping the alert and
                // using the custom callback
                this.skipAlert = true;
                this.onSavePosition(editor, context, order, customOptions);
            },
            this
        );
    },

});
