// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPOrderStatusBasedPositionUpdate.controller.Batch', {

    override: 'Shopware.apps.Order.controller.Batch',

    /**
     * A flag used to skip the alert that is asking whether the order items shall be skipped.
     */
    viisonSkipStatusBasedPositionUpdateAlert: false,

    /**
     * Checks whether the order status set in the form is 'canceled' (4) or 'completely shipped' (7)
     * and, if it is, asks the user whether the orders' items shall be 'cancelled' or 'shipped', too.
     * Finally the parent method is called, either directly without asking the user or after the user
     * closed the alert.
     */
    onProcessChanges: function (form, options) {
        var newStatus = form.getValues().orderStatus;
        if (this.viisonSkipStatusBasedPositionUpdateAlert || (newStatus !== 4 && newStatus !== 7)) {
            this.callParent(arguments);
            this.viisonSkipStatusBasedPositionUpdateAlert = false;

            return;
        }

        // Ask user for position update
        var snippetNamespace = 'backend/viison_pickware_erp_order_status_based_position_update/main';
        if (newStatus === 4) {
            var cancelShippingCostsCheckbox = Ext.create('Ext.form.field.Checkbox', {
                boxLabel: ViisonCommonApp.getSnippet('alert/batch/checkbox/cancel_shipping_costs', snippetNamespace),
                checked: true,
                style: {
                    'margin-left': '20px',
                },
            });
            var createCancellationInvoiceCheckbox = Ext.create('Ext.form.field.Checkbox', {
                boxLabel: ViisonCommonApp.getSnippet('alert/batch/checkbox/create_cancellation_invoice', snippetNamespace),
                checked: true,
                style: {
                    'margin-left': '20px',
                },
            });
            ViisonMessageBoxWithCheckboxes.show({
                title: ViisonCommonApp.getSnippet('alert/title', snippetNamespace),
                msg: ViisonCommonApp.getSnippet('alert/batch/message/cancelled', snippetNamespace),
                icon: Ext.Msg.QUESTION,
                checkboxes: [
                    cancelShippingCostsCheckbox,
                    createCancellationInvoiceCheckbox,
                ],
                bottomMessage: ViisonCommonApp.getSnippet('alert/batch/message_bottom/cancelled', snippetNamespace),
                buttons: Ext.Msg.YESNOCANCEL,
                scope: this,
                callback: function (buttonId) {
                    if (buttonId === 'cancel') {
                        return;
                    }
                    if (buttonId === 'yes') {
                        // Register a one-off listener to add the ID of the selected warehouse to the batch store before it
                        // is synced
                        ViisonPickwareERPOrderEventBus.on(
                            'collectBatchStoreParams',
                            function (store) {
                                store.getProxy().extraParams.pickwareCancelUnshippedItemsOfOrder = 1;
                                if (cancelShippingCostsCheckbox.checked) {
                                    store.getProxy().extraParams.pickwareCancelShippingCosts = 1;
                                }
                                if (createCancellationInvoiceCheckbox.checked) {
                                    store.getProxy().extraParams.pickwareCreateCancellationInvoice = 1;
                                }
                            },
                            this,
                            {
                                single: true,
                            }
                        );
                    }

                    // Save the orders by calling the onProcessChanges method again, but this time skipping the alert
                    this.viisonSkipStatusBasedPositionUpdateAlert = true;
                    this.onProcessChanges(form, options);
                },
            });
        } else if (newStatus === 7) {
            ViisonPickwareBinLocationSelectionAlert.selectWarehouse(
                ViisonCommonApp.getSnippet('alert/title', snippetNamespace),
                ViisonCommonApp.getSnippet('alert/batch/message/completely_shipped', snippetNamespace),
                Ext.Msg.YESNOCANCEL,
                function (buttonId, warehouse) {
                    if (buttonId === 'cancel') {
                        return;
                    }
                    if (buttonId === 'yes') {
                        // Register a one-off listener to add the ID of the selected warehouse to the batch store before it
                        // is synced
                        ViisonPickwareERPOrderEventBus.on(
                            'collectBatchStoreParams',
                            function (store) {
                                store.getProxy().extraParams.stockChangeWarehouseId = warehouse.get('id');
                                store.getProxy().extraParams.pickwareMarkOrdersAsShipped = 1;
                            },
                            this,
                            {
                                single: true,
                            }
                        );
                    }

                    // Save the orders by calling the onProcessChanges method again, but this time skipping the alert
                    this.viisonSkipStatusBasedPositionUpdateAlert = true;
                    this.onProcessChanges(form, options);
                }, this
            );
        }
    },

});
