// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPOrderStatusBasedPositionUpdate.controller.Detail', {

    override: 'Shopware.apps.Order.controller.Detail',

    /**
     * A flag used to skip the alert that is asking whether the order items shall be skipped.
     */
    skipAlert: false,

    /**
     * Checks if the status was changed to 'cancelled' or 'completely shipped' and asks the
     * user whether the order items shall be 'cancelled' or 'shipped', too. Finally the parent
     * method is called, either directly without asking the user or after the user closed
     * the alert.
     *
     * @param record
     * @param options
     */
    onSaveOverview: function (record, options) {
        // Get the edit form of the overview and the status field
        var form = this.getDetailWindow().down('order-overview-panel').editForm.getForm();
        var statusField = form.getFields().findBy(function (item) {
            return item.name === 'status';
        });
        var oldStatus = statusField.viisonOldValue;
        var newStatus = form.getValues().status;
        if (this.skipAlert || oldStatus === newStatus || (newStatus !== 4 && newStatus !== 7)) {
            this.skipAlert = false;
            this.callParent(arguments);
            if (record.getProxy().extraParams) {
                delete record.getProxy().extraParams.pickwareCancelUnshippedItemsOfOrder;
                delete record.getProxy().extraParams.pickwareCancelShippingCosts;
                delete record.getProxy().extraParams.pickwareCreateCancellationInvoice;
                delete record.getProxy().extraParams.pickwareMarkOrderAsShipped;
                delete record.getProxy().extraParams.stockChangeWarehouseId;
            }
            statusField.viisonOldValue = newStatus;

            return;
        }

        // Ask user for position update
        var snippetNamespace = 'backend/viison_pickware_erp_order_status_based_position_update/main';
        if (newStatus === 4) {
            var invoice = record.getReceipt().getRange().find(function (document) {
                return document.get('typeId') === 1;
            });
            var orderHasInvoice = invoice !== undefined;
            var orderHasShippingCosts = record.get('invoiceShipping') !== 0;

            var cancelShippingCostsCheckbox = Ext.create('Ext.form.field.Checkbox', {
                boxLabel: ViisonCommonApp.getSnippet('alert/single/checkbox/cancel_shipping_costs/text', snippetNamespace),
                checked: orderHasShippingCosts,
                disabled: !orderHasShippingCosts,
                style: {
                    'margin-left': '20px',
                },
                helpText: orderHasShippingCosts ? '' : ViisonCommonApp.getSnippet('alert/single/checkbox/cancel_shipping_costs/disabled_tooltip', snippetNamespace),
            });
            var createCancellationInvoiceCheckbox = Ext.create('Ext.form.field.Checkbox', {
                boxLabel: ViisonCommonApp.getSnippet('alert/single/checkbox/create_cancellation_invoice/text', snippetNamespace),
                checked: orderHasInvoice,
                disabled: !orderHasInvoice,
                style: {
                    'margin-left': '20px',
                },
                helpText: orderHasInvoice ? '' : ViisonCommonApp.getSnippet('alert/single/checkbox/create_cancellation_invoice/disabled_tooltip', snippetNamespace),
            });

            ViisonMessageBoxWithCheckboxes.show({
                title: ViisonCommonApp.getSnippet('alert/title', snippetNamespace),
                msg: ViisonCommonApp.getSnippet('alert/single/message/cancelled', snippetNamespace),
                icon: Ext.Msg.QUESTION,
                checkboxes: [
                    cancelShippingCostsCheckbox,
                    createCancellationInvoiceCheckbox,
                ],
                buttons: Ext.Msg.YESNOCANCEL,
                width: 600,
                scope: this,
                callback: function (buttonId) {
                    if (buttonId === 'cancel') {
                        return;
                    }
                    record.getProxy().extraParams = record.getProxy().extraParams || {};
                    if (buttonId === 'yes') {
                        record.getProxy().extraParams.pickwareCancelUnshippedItemsOfOrder = 1;
                        if (cancelShippingCostsCheckbox.checked) {
                            record.getProxy().extraParams.pickwareCancelShippingCosts = 1;
                        }
                        if (createCancellationInvoiceCheckbox.checked) {
                            record.getProxy().extraParams.pickwareCreateCancellationInvoice = 1;
                        }
                    }

                    // Save the record by calling the onSaveOverview method again, but this time skipping the alert
                    this.skipAlert = true;
                    this.onSaveOverview(record, options);
                },
            });
        } else if (newStatus === 7) {
            ViisonPickwareBinLocationSelectionAlert.selectWarehouse(
                ViisonCommonApp.getSnippet('alert/title', snippetNamespace),
                ViisonCommonApp.getSnippet('alert/single/message/completely_shipped', snippetNamespace),
                Ext.Msg.YESNOCANCEL,
                function (buttonId, warehouse) {
                    if (buttonId === 'cancel') {
                        return;
                    }

                    // Add the ID of the selected warehouse to the record proxy's extra params
                    record.getProxy().extraParams = record.getProxy().extraParams || {};
                    if (buttonId === 'yes') {
                        record.getProxy().extraParams.stockChangeWarehouseId = warehouse.get('id');
                        record.getProxy().extraParams.pickwareMarkOrderAsShipped = 1;
                    }

                    // Save the record by calling the onSaveOverview method again, but this time skipping the alert
                    this.skipAlert = true;
                    this.onSaveOverview(record, options);
                },
                this
            );
        }
    },

});
