// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPOrderStatusBasedPositionUpdate.controller.List', {

    override: 'Shopware.apps.Order.controller.List',

    /**
     * Checks if the status was changed to 'cancelled' or 'completely delivered' and asks the
     * user whether the order items shall be 'cancelled' or 'shipped', too. Finally the parent
     * method is called, either directly without asking the user or after the user closed
     * the alert.
     *
     * @param {Ext.grid.plugin.RowEditing} editor
     * @param {object} event
     * @param {Shopware.apps.Order.store.Order} store
     */
    onSaveOrder: function (editor, event, store) {
        var order = event.record;
        var oldStatus = event.originalValues.status;
        var newStatus = event.newValues.status;
        var statusCompletelyDelivered = 7;
        var statusCancelledRejected = 4;
        if (oldStatus === newStatus || (newStatus !== statusCancelledRejected && newStatus !== statusCompletelyDelivered)) {
            this.callParent(arguments);
            this.viisonClearProxyParams(order);

            return;
        }

        // this.callParent() cannot be called asynchronous (in the callback of the MsgBox), therefore we save the
        // super method in a local variable an call that in the callback instead.
        // Determine the superMethod like ExtJS does
        var getCurrentFunction = function () {
            return getCurrentFunction.caller;
        };
        var method = getCurrentFunction();
        var superMethod = method.$previous || ((method = method.$owner ? method : method.caller) && method.$owner.superclass[method.$name]);

        var snippetNamespace = 'backend/viison_pickware_erp_order_status_based_position_update/main';
        if (newStatus === statusCancelledRejected) {
            var invoice = order.getReceipt().getRange().find(function (document) {
                return document.get('typeId') === 1;
            });
            var orderHasInvoice = invoice !== undefined;
            var orderHasShippingCosts = order.get('invoiceShipping') !== 0;
            var cancelShippingCostsCheckbox = Ext.create('Ext.form.field.Checkbox', {
                boxLabel: ViisonCommonApp.getSnippet('alert/single/checkbox/cancel_shipping_costs/text', snippetNamespace),
                checked: orderHasShippingCosts,
                disabled: !orderHasShippingCosts,
                style: {
                    'margin-left': '20px',
                },
                helpText: orderHasShippingCosts ? '' : ViisonCommonApp.getSnippet('alert/single/checkbox/cancel_shipping_costs/disabled_tooltip', snippetNamespace),
            });
            var createCancellationInvoiceCheckbox = Ext.create('Ext.form.field.Checkbox', {
                boxLabel: ViisonCommonApp.getSnippet('alert/single/checkbox/create_cancellation_invoice/text', snippetNamespace),
                checked: orderHasInvoice,
                disabled: !orderHasInvoice,
                style: {
                    'margin-left': '20px',
                },
                helpText: orderHasInvoice ? '' : ViisonCommonApp.getSnippet('alert/single/checkbox/create_cancellation_invoice/disabled_tooltip', snippetNamespace),
            });

            ViisonMessageBoxWithCheckboxes.show({
                title: ViisonCommonApp.getSnippet('alert/title', snippetNamespace),
                msg: ViisonCommonApp.getSnippet('alert/single/message/cancelled', snippetNamespace),
                icon: Ext.Msg.QUESTION,
                checkboxes: [
                    cancelShippingCostsCheckbox,
                    createCancellationInvoiceCheckbox,
                ],
                buttons: Ext.Msg.YESNOCANCEL,
                width: 600,
                scope: this,
                callback: function (buttonId) {
                    if (buttonId === 'cancel') {
                        event.record.set('status', event.originalValues.status);
                        this.viisonClearProxyParams(order);

                        return;
                    }
                    order.getProxy().extraParams = order.getProxy().extraParams || {};
                    if (buttonId === 'yes') {
                        order.getProxy().extraParams.pickwareCancelUnshippedItemsOfOrder = 1;
                        if (cancelShippingCostsCheckbox.checked) {
                            order.getProxy().extraParams.pickwareCancelShippingCosts = 1;
                        }
                        if (createCancellationInvoiceCheckbox.checked) {
                            order.getProxy().extraParams.pickwareCreateCancellationInvoice = 1;
                        }
                    }

                    superMethod.apply(this, [editor, event, store]);
                    this.viisonClearProxyParams(order);
                },
            });
        } else if (newStatus === 7) {
            ViisonPickwareBinLocationSelectionAlert.selectWarehouse(
                ViisonCommonApp.getSnippet('alert/title', snippetNamespace),
                ViisonCommonApp.getSnippet('alert/single/message/completely_shipped', snippetNamespace),
                Ext.Msg.YESNOCANCEL,
                function (buttonId, warehouse) {
                    if (buttonId === 'cancel') {
                        event.record.set('status', event.originalValues.status);
                        this.viisonClearProxyParams(order);

                        return;
                    }
                    // Add the ID of the selected warehouse to the order proxy's extra params
                    order.getProxy().extraParams = order.getProxy().extraParams || {};
                    if (buttonId === 'yes') {
                        order.getProxy().extraParams.stockChangeWarehouseId = warehouse.get('id');
                        order.getProxy().extraParams.pickwareMarkOrderAsShipped = 1;
                    }

                    superMethod.apply(this, [editor, event, store]);
                    this.viisonClearProxyParams(order);
                },
                this
            );
        }
    },

    /**
     * @param {Shopware.apps.Order.model.Order} order
     */
    viisonClearProxyParams: function (order) {
        if (!order || !order.getProxy().extraParams) {
            return;
        }

        // Reset the proxy parameters to their default. Hopefully the superMethod will always be
        // synchronous, otherwise this would break the functionality.
        delete order.getProxy().extraParams.pickwareMarkOrderAsShipped;
        delete order.getProxy().extraParams.stockChangeWarehouseId;
        delete order.getProxy().extraParams.pickwareCancelUnshippedItemsOfOrder;
        delete order.getProxy().extraParams.pickwareCancelShippingCosts;
        delete order.getProxy().extraParams.pickwareCreateCancellationInvoice;
    },

});
