// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

/**
 * A singleton that can observe changes within a ViisonPickwareERP plugin config form and,
 * in case the 'purchasePriceMode' changes, shows a growl notification, which informs the user
 * about converting the pruchase prices accordingly.
 */
Ext.define('Shopware.apps.ViisonPickwareERPPurchasePriceConverter.Converter', {

    singleton: true,
    alternateClassName: 'ViisonPickwarePurchasePriceConverter',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_purchase_price_converter/main',
    viisonConfigNamespace: 'ViisonPickwareERPPurchasePriceConverter',

    /**
     * @var Ext.panel.Panel modeChangeGrowlNotification
     */
    modeChangeGrowlNotification: null,

    constructor: function () {
        this.callParent(arguments);

        var converterConfig = this.getViisonConfig('ViisonPickwareERPPurchasePriceConverter');
        this.activePurchasePriceMode = converterConfig.purchasePriceModes.active;
        this.configuredPurchasePriceMode = converterConfig.purchasePriceModes.configured;
    },

    /**
     * Adds listeners on the 'load' and 'update' events to the given formStore, which check and compare
     * the 'purchasePriceMode' configured in the 'ViisonPickwareERP' plugin config form. If the mode
     * is changed, the listener on the the 'update' event triggers a growl notification informing the
     * user about the change an the consequences.
     *
     * @param Shopware.apps.Base.model.Form|Shopware.apps.Config.store.main.Form formStore
     */
    observeFormStore: function (formStore) {
        // Add listener on 'load' events, to be able to save the value of the 'purchasePriceMode'
        // element for later use
        formStore.on('load', function (store, records) {
            if (records.length === 0 || records[0].get('name') !== 'ViisonPickwareERP') {
                return;
            }

            // Save the currently configured purchasePriceMode
            this.configuredPurchasePriceMode = this.getPurchasePriceMode(records[0]);
        }, this);

        // Add listener on 'update' events, to be able to detect changes of the 'purchasePriceMode' element
        formStore.on('update', function (store, record, operation) {
            if (record.get('name') !== 'ViisonPickwareERP' || operation !== Ext.data.Model.COMMIT) {
                return;
            }

            // Save the new configured purchase price mode and update the growl notifications
            this.configuredPurchasePriceMode = this.getPurchasePriceMode(record);
            this.updatePurchasePriceModeChangeNotification();
        }, this);
    },

    /**
     * Loads the currently active and configured purchase price modes and, if the differ, shows a new
     * purchase price change notification.
     */
    checkPurchasePriceModeConfiguration: function () {
        Ext.Ajax.request({
            method: 'GET',
            url: ViisonCommonApp.assembleBackendUrl(
                'ViisonPickwareERPPurchasePriceConverter/getPurchasePriceModes'
            ),
            scope: this,
            success: function (response) {
                // Decode the response
                var responseData = Ext.JSON.decode(response.responseText, true);
                if (!responseData || !responseData.purchasePriceModes) {
                    return;
                }

                // Save the purchase price modes and update the growl notifications
                this.activePurchasePriceMode = responseData.purchasePriceModes.active || this.activePurchasePriceMode;
                this.configuredPurchasePriceMode = responseData.purchasePriceModes.configured || this.configuredPurchasePriceMode;
                this.updatePurchasePriceModeChangeNotification();
            },
        });
    },

    /**
     * Creates and shows a customised, sticky growl notification, which notifies the user about the
     * change of the purchase price mode and ask them to convert the purchase prices accordingly.
     */
    updatePurchasePriceModeChangeNotification: function () {
        // Close a currently displayed notification
        if (this.modeChangeGrowlNotification) {
            Shopware.Notification.closeGrowlMessage(this.modeChangeGrowlNotification, Shopware.Notification);
            this.modeChangeGrowlNotification = null;
        }

        // Check whether a new notification must be displayed
        var oldMode = this.activePurchasePriceMode;
        var newMode = this.configuredPurchasePriceMode;
        if (oldMode === newMode) {
            return;
        }

        // Show sticky growl notification
        this.modeChangeGrowlNotification = Shopware.Notification.createStickyGrowlMessage({
            title: this.getViisonSnippet('notifications/purchase_price_mode_changed/title'),
            text: Ext.String.format(
                this.getViisonSnippet('notifications/purchase_price_mode_changed/message'),
                this.getViisonSnippet(oldMode, 'backend/viison_pickware_erp_purchase_price_helper/main'),
                this.getViisonSnippet(newMode, 'backend/viison_pickware_erp_purchase_price_helper/main')
            ),
            log: true,
            btnDetail: {
                text: this.getViisonSnippet('notifications/purchase_price_mode_changed/buttons/convert'),
                scope: this,
                callback: function () {
                    this.onNotificationConvertPricesButton.apply(this, [oldMode, newMode]);
                },
            },
            onCloseButton: function () {
                this.onNotificationCloseButton.apply(this, [oldMode, newMode]);
            }.bind(this),
            scope: this,
            callback: function () {
                this.modeChangeGrowlNotification = null;
            },
        }, 'ViisonPickwareERP');
    },

    /**
     * Shows a confirm alert and, if the user confirms it, sends an AJAX request to convert
     * all purchase prices from the 'oldMode' to the 'newMode'.
     *
     * @param string oldMode
     * @param string newMode
     */
    onNotificationConvertPricesButton: function (oldMode, newMode) {
        var alertMessage = Ext.String.format(
            this.getViisonSnippet('alerts/convert_prices/message'),
            this.getViisonSnippet(oldMode, 'backend/viison_pickware_erp_purchase_price_helper/main'),
            this.getViisonSnippet(newMode, 'backend/viison_pickware_erp_purchase_price_helper/main')
        );
        Ext.Msg.confirm(
            this.getViisonSnippet('alerts/convert_prices/title'),
            alertMessage,
            function (buttonId) {
                if (buttonId === 'yes') {
                    // Confirmed, hence convert the purchase prices
                    Ext.Ajax.request({
                        method: 'POST',
                        url: ViisonCommonApp.assembleBackendUrl(
                            'ViisonPickwareERPPurchasePriceConverter/convertPurchasePrices'
                        ),
                        params: {
                            oldMode: oldMode,
                            newMode: newMode,
                        },
                        scope: this,
                        success: function (response) {
                            // Decode the response
                            var responseData = Ext.JSON.decode(response.responseText, true);
                            var success = responseData && responseData.success;

                            if (responseData && responseData.purchasePriceModes) {
                                // Save the purchase price modes and update the growl notifications
                                this.activePurchasePriceMode = responseData.purchasePriceModes.active || this.activePurchasePriceMode;
                                this.configuredPurchasePriceMode = responseData.purchasePriceModes.configured || this.configuredPurchasePriceMode;
                                this.updatePurchasePriceModeChangeNotification();
                            }

                            // Show success/failure notification
                            var title;
                            var message;
                            if (success) {
                                title = this.getViisonSnippet('notifications/convert_prices/success/title');
                                message = this.getViisonSnippet('notifications/convert_prices/success/message');
                            } else {
                                title = this.getViisonSnippet('notifications/convert_prices/failure/title');
                                message = (responseData && responseData.message) ? responseData.message : this.getViisonSnippet('notifications/convert_prices/failure/message');
                            }
                            Shopware.Notification.createGrowlMessage(title, message, 'ViisonPickwareERP', null, true);
                        },
                    });
                } else {
                    // Declined, hence show the notification again
                    this.updatePurchasePriceModeChangeNotification();
                }
            },
            this
        );
    },

    /**
     * Shows a confirm alert and, if the user confirms it, sends an AJAX request to hide
     * the change the notification until the mode changes again.
     *
     * @param string oldMode
     * @param string newMode
     */
    onNotificationCloseButton: function (oldMode, newMode) {
        Ext.Msg.show({
            title: this.getViisonSnippet('alerts/close_change_notification/title'),
            msg: Ext.String.format(
                this.getViisonSnippet('alerts/close_change_notification/message'),
                this.getViisonSnippet(oldMode, 'backend/viison_pickware_erp_purchase_price_helper/main'),
                this.getViisonSnippet(newMode, 'backend/viison_pickware_erp_purchase_price_helper/main')
            ),
            buttons: Ext.Msg.OK,
            buttonText: {
                cancel: this.getViisonSnippet('alerts/close_change_notification/buttons/hide_notification'),
            },
            icon: Ext.Msg.WARNING,
            scope: this,
            callback: function (buttonId) {
                if (buttonId !== 'cancel') {
                    return;
                }

                // Hide future notifications
                Ext.Ajax.request({
                    method: 'POST',
                    url: ViisonCommonApp.assembleBackendUrl(
                        'ViisonPickwareERPPurchasePriceConverter/hideChangeNotification'
                    ),
                    scope: this,
                    success: function (response) {
                        // Decode the response
                        var responseData = Ext.JSON.decode(response.responseText, true);
                        if (!responseData || !responseData.purchasePriceModes) {
                            return;
                        }

                        // Save the purchase price modes and update the growl notifications
                        this.activePurchasePriceMode = responseData.purchasePriceModes.active || this.activePurchasePriceMode;
                        this.configuredPurchasePriceMode = responseData.purchasePriceModes.configured || this.configuredPurchasePriceMode;
                        this.updatePurchasePriceModeChangeNotification();
                    },
                });
            },
        });
    },

    /**
     * @param Shopware.apps.Config.model.main.Form
     * @return string|null
     */
    getPurchasePriceMode: function (form) {
        // Try to find the the 'purchasePriceMode' element
        var modeIndex = form.getElements().findExact('name', 'purchasePriceMode');
        if (modeIndex === -1) {
            return null;
        }
        var element = form.getElements().getAt(modeIndex);

        // Determine the value
        return (element.getValues().count() > 0) ? element.getValues().first().get('value') : element.get('value');
    },

});
