// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPReturnShipment.controller.Detail', {

    extend: 'Ext.app.Controller',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_return_shipment/detail',

    refs: [
        {
            ref: 'returnShipmentItemList',
            selector: 'viison_pickware_erp_return_shipment-detail-tab-return_shipment_item_list',
        },
        {
            ref: 'attachmentList',
            selector: 'viison_pickware_erp_return_shipment-detail-tab-attachments-attachment_list',
        },
        {
            ref: 'attachmentListDropZone',
            selector: 'viison_pickware_erp_return_shipment-detail-tab-attachments-attachment_list html5fileupload',
        },
    ],

    /**
     * @type {Shopware.apps.ViisonPickwareERPReturnShipment.view.detail.Window}
     */
    mainWindow: null,

    /**
     * Optional callback, which will be called each time the return shipment record will be
     * updated based on a server response.
     *
     * @type {function}
     */
    onUpdateCallback: Ext.emptyFn,

    /**
     * @type {Shopware.apps.ViisonPickwareERPReturnShipment.model.detail.ReturnShipment}
     */
    returnShipmentRecord: null,

    /**
     * @type {Shopware.apps.ViisonPickwareERPWarehouseManagement.store.Warehouse}
     */
    warehouseStore: null,

    /**
     * @override
     */
    init: function () {
        this.callParent(arguments);

        this.control({
            'viison_pickware_erp_return_shipment-detail-content': {
                closeWindow: this.onCloseWindow,
                saveReturnShipment: this.onSaveReturnShipment,
                cancelReturnShipment: this.onCancelReturnShipment,
                finalizeReturnShipment: this.onFinalizeReturnShipment,
            },
            'viison_pickware_erp_return_shipment-detail-general_info-header': {
                targetWarehouseChange: this.onTargetWarehouseChange,
            },
            'viison_pickware_erp_return_shipment-detail-general_info-order': {
                openOrder: this.onOpenOrder,
            },
            'viison_pickware_erp_return_shipment-detail-general_info-customer': {
                openCustomer: this.onOpenCustomer,
            },
            'viison_pickware_erp_return_shipment-detail-general_info-correction_of_invoice': {
                openDocument: this.onOpenDocument,
            },
            'viison_pickware_erp_return_shipment-detail-tab-return_shipment_item_list': {
                addReturnShipmentItems: this.onAddReturnShipmentItems,
                openArticle: this.onOpenArticle,
                deleteReturnShipmentItem: this.onDeleteReturnShipmentItem,
                editReturnShipmentItem: this.onEditReturnShipmentItem,
            },
            'viison_pickware_erp_return_shipment-detail-tab-attachments-attachment_list': {
                addAttachment: this.onAddAttachment,
                downloadAttachment: this.onDownloadAttachment,
                deleteAttachments: this.onDeleteAttachments,
                attachmentsSelectionChange: this.onAttachmentsSelectionChange,
                attachmentBeforeUpload: this.onAttachmentBeforeUpload,
                attachmentUploaded: this.onAttachmentUploaded,
            },
            'viison_pickware_erp_return_shipment-detail-tab-attachments-attachment_list filefield': {
                change: this.onAddAttachment,
            },
            'viison_pickware_erp_return_shipment-detail-tab-internal_comments-add_comment_form': {
                addInternalComment: this.onAddInternalComment,
            },
            'viison_pickware_erp_return_shipment-detail-tab-internal_comments-internal_comment_list': {
                deleteInternalComment: this.onDeleteInternalComment,
            },
        });
    },

    /**
     * @override
     *
     * Scope refs to main window
     *
     * @param {string} selector
     * @returns {Ext.Component}
     */
    getActiveReference: function (selector) {
        return Ext.ComponentQuery.query(selector, this.mainWindow)[0];
    },

    /**
     * Creates the detail window and triggers loading the related return shipment.
     *
     * @param {number} returnShipmentId
     * @param {number} orderId
     * @param {function} onUpdateCallback
     */
    createWindow: function (returnShipmentId, orderId, onUpdateCallback) {
        this.onUpdateCallback = onUpdateCallback || Ext.emptyFn;

        this.mainWindow = this.getView('detail.Window').create();
        this.mainWindow.show();
        this.loadData(returnShipmentId, orderId);
    },

    /**
     * Loads the return shipment specified by the combination of a return shipment id and a related
     * order id. If a return shipment with the given id exists, its data will be returned by the server,
     * if not the server will return some "base data" based on the given order id without creating a new
     * return shipment. The return shipment is not created until the user clicks the save button of the
     * detail window for the first time.
     *
     * @param {number} returnShipmentId
     * @param {number} orderId
     */
    loadData: function (returnShipmentId, orderId) {
        this.mainWindow.showLoadMask(this.getViisonSnippet('window/loading_mask/initial_load/text'));

        this.warehouseStore = Ext.create('Shopware.apps.ViisonPickwareERPWarehouseManagement.store.Warehouse', {
            pageSize: 2000,
            sorters: [
                { property: 'warehouse.defaultReturnShipmentWarehouse', direction: 'DESC' },
                { property: 'warehouse.defaultWarehouse', direction: 'DESC' },
                { property: 'warehouse.name' },
            ],
        });

        // Chain loading of the warehouse store and the return shipment, in order to ensure
        // that all data is available when rendering the window content
        this.warehouseStore.load({
            scope: this,
            callback: function () {
                this.getModel('detail.ReturnShipment').load(-1, {
                    params: {
                        returnShipmentId: returnShipmentId,
                        orderId: orderId,
                    },
                    scope: this,
                    callback: function (returnShipmentRecord) {
                        this.returnShipmentRecord = returnShipmentRecord;

                        this.mainWindow.add(
                            this.getView('detail.Content').create()
                        );
                        this.mainWindow.reconfigureView();

                        // Delay hiding of the loading mask to prevent visual glitches
                        setTimeout(function () {
                            this.mainWindow.hideLoadMask();
                        }.bind(this), 400);
                    },
                });
            },
        });
    },

    /**
     * @returns {Shopware.apps.ViisonPickwareERPReturnShipment.model.detail.ReturnShipment}
     */
    getReturnShipmentRecord: function () {
        return this.returnShipmentRecord;
    },

    /**
     * @returns {Shopware.apps.ViisonPickwareERPWarehouseManagement.store.Warehouse}
     */
    getWarehouseStore: function () {
        return this.warehouseStore;
    },

    onCloseWindow: function () {
        this.mainWindow.close();
    },

    /**
     * Updates the target warehouse property of the return shipment. This event handler is called
     * when the user changed the selected target warehouse via the target warehouse combobox.
     *
     * @param {Shopware.apps.ViisonPickwareERPWarehouseManagement.store.Warehouse[]} warehouseSelection
     */
    onTargetWarehouseChange: function (warehouseSelection) {
        if (warehouseSelection.length > 0) {
            this.getReturnShipmentRecord().set('targetWarehouseId', warehouseSelection[0].get('id'));
        }
    },

    onSaveReturnShipment: function () {
        this.saveReturnShipment();
    },

    /**
     * Saves the current state of the return shipment by sending it to server and updates the return shipment
     * based on the returned data. If an optional mail configuration is returned by the server a mail composer
     * window will be show.
     */
    saveReturnShipment: function () {
        this.mainWindow.showLoadMask(this.getViisonSnippet('window/loading_mask/save_return_shipment/text'));

        this.getReturnShipmentRecord().save({
            scope: this,
            callback: function (data, operation) {
                if (operation.success) {
                    Shopware.Notification.createGrowlMessage(
                        this.getViisonSnippet('notification/save_return_shipment/success/title'),
                        Ext.String.format(
                            this.getViisonSnippet('notification/save_return_shipment/success/message'),
                            this.getReturnShipmentRecord().get('number')
                        ),
                        'ViisonPickwareERP'
                    );

                    this.updateReturnShipmentRecord(operation.response);

                    if (operation.request.proxy.getReader().rawData.mail) {
                        this.createMailComposerWindow(operation.request.proxy.getReader().rawData.mail);
                    }
                } else {
                    Shopware.Notification.createGrowlMessage(
                        this.getViisonSnippet('notification/save_return_shipment/failure/title'),
                        Ext.String.format(
                            this.getViisonSnippet('notification/save_return_shipment/failure/message'),
                            this.getReturnShipmentRecord().get('number'),
                            operation.request.proxy.getReader().rawData.message
                        ),
                        'ViisonPickwareERP'
                    );
                }

                this.mainWindow.hideLoadMask();
            },
        });
    },

    /**
     * Updates the return shipment based on the given server response.
     *
     * @param {object} response
     */
    updateReturnShipmentRecord: function (response) {
        this.returnShipmentRecord = this.getReturnShipmentRecord().getProxy().reader.read(response).records[0];
        this.fireEvent('returnShipmentRecordSaved');
        this.onUpdateCallback();
    },

    /**
     * Opens the cancellation dialog for the return shipment passing a callback, which be called after
     * the cancellation was performed successfully.
     */
    onCancelReturnShipment: function () {
        Shopware.app.Application.addSubApplication({
            name: 'Shopware.apps.ViisonPickwareERPCancellation',
            orderId: this.getReturnShipmentRecord().getOrder().get('id'),
            returnShipmentId: this.getReturnShipmentRecord().get('id'),
            afterCancellationCallback: this.reloadAfterCancellation.bind(this),
        });
    },

    /**
     * Updates the return shipment by reloading it from the server after the cancellation has been
     * performed successfully.
     */
    reloadAfterCancellation: function () {
        this.mainWindow.showLoadMask(this.getViisonSnippet('window/loading_mask/after_cancellation_reload/text'));

        Shopware.Notification.createGrowlMessage(
            this.getViisonSnippet('notification/cancel_return_shipment/success/title'),
            Ext.String.format(
                this.getViisonSnippet('notification/cancel_return_shipment/success/message'),
                this.getReturnShipmentRecord().get('number')
            ),
            'ViisonPickwareERP'
        );

        Ext.Ajax.request({
            url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPReturnShipment/getReturnShipment'),
            method: 'POST',
            params: {
                returnShipmentId: this.getReturnShipmentRecord().get('id'),
            },
            scope: this,
            callback: function (options, success, response) {
                var result = Ext.JSON.decode(response.responseText, true);

                if (result && result.success) {
                    this.updateReturnShipmentRecord(response);
                }

                this.mainWindow.hideLoadMask();
            },
        });
    },

    /**
     * Displays a confirm dialog asking the user if the return shipment should be indeed finalized and triggers
     * the finalization in case the user confirms the finalization.
     */
    onFinalizeReturnShipment: function () {
        if (this.getReturnShipmentRecord().canBeCanceled()) {
            Ext.Msg.confirm(
                this.getViisonSnippet('confirm/finalize_return_shipment/title'),
                Ext.String.format(
                    this.getViisonSnippet('confirm/finalize_return_shipment/text'),
                    this.getReturnShipmentRecord().get('number')
                ),
                function (decision) {
                    if (decision !== 'yes') {
                        return;
                    }

                    this.finalizeReturnShipment();
                },
                this
            );
        } else {
            this.finalizeReturnShipment();
        }
    },

    /**
     * Finalizes the return shipment by calling the related server action. The return shipment will be updated
     * based on the returned server data. If a optional mail configuration is returned by the server a mail composer
     * window will be show.
     */
    finalizeReturnShipment: function () {
        this.mainWindow.showLoadMask(this.getViisonSnippet('window/loading_mask/finalize_return_shipment/text'));

        Ext.Ajax.request({
            url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPReturnShipment/finalizeReturnShipment'),
            method: 'GET',
            params: {
                returnShipmentId: this.getReturnShipmentRecord().get('id'),
            },
            scope: this,
            callback: function (options, success, response) {
                var result = Ext.JSON.decode(response.responseText, true);

                if (result && result.success) {
                    Shopware.Notification.createGrowlMessage(
                        this.getViisonSnippet('notification/finalize_return_shipment/success/title'),
                        Ext.String.format(
                            this.getViisonSnippet('notification/finalize_return_shipment/success/message'),
                            this.getReturnShipmentRecord().get('number')
                        ),
                        'ViisonPickwareERP'
                    );

                    this.updateReturnShipmentRecord(response);

                    if (result.mail) {
                        this.createMailComposerWindow(result.mail);
                    }
                } else {
                    Shopware.Notification.createGrowlMessage(
                        this.getViisonSnippet('notification/finalize_return_shipment/failure/title'),
                        Ext.String.format(
                            this.getViisonSnippet('notification/finalize_return_shipment/failure/message'),
                            this.getReturnShipmentRecord().get('number'),
                            result.message
                        ),
                        'ViisonPickwareERP'
                    );
                }

                this.mainWindow.hideLoadMask();
            },
        });
    },

    /**
     * Opens the order detail window of the related order.
     */
    onOpenOrder: function () {
        Shopware.app.Application.addSubApplication({
            name: 'Shopware.apps.Order',
            action: 'detail',
            params: {
                orderId: this.getReturnShipmentRecord().getOrder().get('id'),
            },
        });
    },

    /**
     * Opens the customer detail window of the related customer.
     */
    onOpenCustomer: function () {
        Shopware.app.Application.addSubApplication({
            name: 'Shopware.apps.Customer',
            action: 'detail',
            params: {
                customerId: this.getReturnShipmentRecord().getCustomer().get('id'),
            },
        });
    },

    /**
     * Opens the related cancellation Invoice document (PDF) in a new browser window/tab.
     */
    onOpenDocument: function () {
        window.open(ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPReturnShipment/openCorrectionOfInvoice', {
            documentId: this.getReturnShipmentRecord().get('documentId'),
        }));
    },

    /**
     * Opens the return shipment items picker.
     */
    onAddReturnShipmentItems: function () {
        this.getController('ReturnShipmentItemsPicker').createWindow(
            this.getReturnShipmentRecord().getOrder().get('id'),
            this.getReturnShipmentItemList().getStore(),
            function (returnShipmentItemRecords) {
                this.getReturnShipmentItemList().getStore().add(returnShipmentItemRecords);
            }.bind(this)
        );
    },

    /**
     * Opens the article detail window of the article related to a given return shipment item.
     *
     * @param {Shopware.apps.ViisonPickwareERPReturnShipment.model.detail.ReturnShipmentItem} returnShipmentItemRecord
     */
    onOpenArticle: function (returnShipmentItemRecord) {
        Shopware.app.Application.addSubApplication({
            name: 'Shopware.apps.Article',
            action: 'detail',
            params: {
                articleId: returnShipmentItemRecord.get('articleId'),
            },
        });
    },

    /**
     * Displays a confirm window asking the user if a given return shipment item should indeed be deleted. If the user
     * confirms, the return shipment item will be removed from the items store.
     *
     * @param {Shopware.apps.ViisonPickwareERPReturnShipment.model.detail.ReturnShipmentItem} returnShipmentItemRecord
     */
    onDeleteReturnShipmentItem: function (returnShipmentItemRecord) {
        var message = Ext.String.format(
            this.getViisonSnippet('confirm/delete_return_shipment_item/text'),
            returnShipmentItemRecord.get('articleNumber')
        );
        Ext.Msg.confirm(
            this.getViisonSnippet('confirm/delete_return_shipment_item/title'),
            message,
            function (decision) {
                if (decision !== 'yes') {
                    return;
                }

                this.getReturnShipmentRecord().getItems().remove(returnShipmentItemRecord);
            },
            this
        );
    },

    /**
     * @param {Shopware.apps.ViisonPickwareERPReturnShipment.model.detail.ReturnShipmentItem} returnShipmentItemRecord
     */
    onEditReturnShipmentItem: function (returnShipmentItemRecord) {
        var returnShipmentItemList = this.getReturnShipmentItemList();
        returnShipmentItemList.cellEditingPlugin.startEdit(
            returnShipmentItemRecord, returnShipmentItemList.returnedQuantityColumn
        );
    },

    /**
     * Updates the state of the attachments download and delete buttons based on the given selection of
     * attachments. The download button will enabled only if the selection contains exactly one element
     * whereas the delete button will we enabled whenever the selection contains at least one attachment.
     *
     * @param {Shopware.apps.ViisonPickwareERPReturnShipment.model.detail.Attachment[]} selection
     */
    onAttachmentsSelectionChange: function (selection) {
        if (selection.length === 1) {
            this.getAttachmentList().enableDownloadButton();
        } else {
            this.getAttachmentList().disableDownloadButton();
        }

        if (selection.length > 0) {
            this.getAttachmentList().enableDeleteButton();
        } else {
            this.getAttachmentList().disableDeleteButton();
        }
    },

    /**
     * Displays a loading mask when a new attachment is being uploaded.
     */
    onAttachmentBeforeUpload: function () {
        this.mainWindow.showLoadMask(this.getViisonSnippet('window/loading_mask/upload_attachment/text'));
    },

    /**
     * Adds an uploaded attachment to the return shipment by passing the its media id to the server and updating
     * the attachments store based on the server response.
     *
     * @param {object} xhr
     */
    onAttachmentUploaded: function (xhr) {
        var attachmentData = Ext.JSON.decode(xhr.responseText).data;

        Ext.Ajax.request({
            url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPReturnShipment/addAttachment'),
            method: 'POST',
            params: {
                returnShipmentId: this.getReturnShipmentRecord().get('id'),
                mediaId: attachmentData.id,
            },
            scope: this,
            callback: function (options, success, response) {
                var result = Ext.JSON.decode(response.responseText, true);

                if (result && result.success) {
                    this.getReturnShipmentRecord().getAttachments().add(
                        this.getModel('detail.Attachment').create(result.data)
                    );

                    Shopware.Notification.createGrowlMessage(
                        this.getViisonSnippet('notification/upload_attachment/success/title'),
                        Ext.String.format(
                            this.getViisonSnippet('notification/upload_attachment/success/message'),
                            this.getReturnShipmentRecord().get('number')
                        ),
                        'ViisonPickwareERP'
                    );
                } else {
                    Shopware.Notification.createGrowlMessage(
                        this.getViisonSnippet('notification/upload_attachment/failure/title'),
                        Ext.String.format(
                            this.getViisonSnippet('notification/upload_attachment/failure/message'),
                            this.getReturnShipmentRecord().get('number'),
                            result.message
                        ),
                        'ViisonPickwareERP'
                    );
                }

                this.mainWindow.hideLoadMask();
            },
        });
    },

    /**
     * Triggers the upload of a new attachment, which was selected via the add new attachment button
     * of the attachments tab.
     *
     * @param {Ext.button.Button} uploadButton
     */
    onAddAttachment: function (uploadButton) {
        var fileField = uploadButton.getEl().down('input[type=file]').dom;

        uploadButton.reset();
        this.getAttachmentListDropZone().iterateFiles(fileField.files);
    },

    /**
     * Starts downloading a given attachment.
     *
     * @param {Shopware.apps.ViisonPickwareERPReturnShipment.model.detail.Attachment[]} selectedAttachments
     */
    onDownloadAttachment: function (selectedAttachments) {
        if (selectedAttachments.length === 1) {
            window.open(
                ViisonCommonApp.assembleBackendUrl('mediaManager/download', {
                    mediaId: selectedAttachments[0].get('mediaId'),
                }),
                '_blank'
            );
        }
    },

    /**
     * Displays a confirm dialog asking the user if a given selection of attachments should indeed be deleted and,
     * if the user confirms, removes them from the attachments store.
     *
     * @param {Shopware.apps.ViisonPickwareERPReturnShipment.model.detail.Attachment[]} selectedAttachments
     */
    onDeleteAttachments: function (selectedAttachments) {
        if (selectedAttachments.length < 1) {
            return;
        }

        Ext.Msg.confirm(
            this.getViisonSnippet('confirm/delete_attachments/title'),
            this.getViisonSnippet('confirm/delete_attachments/text'),
            function (decision) {
                if (decision !== 'yes') {
                    return;
                }

                this.getReturnShipmentRecord().getAttachments().remove(selectedAttachments);
            },
            this
        );
    },

    /**
     * Adds a given comment to the return shipment's store of internal comments.
     *
     * @param {string} comment
     */
    onAddInternalComment: function (comment) {
        this.getReturnShipmentRecord().getInternalComments().add(
            this.getModel('detail.InternalComment').create({
                comment: comment,
                created: new Date(),
                userName: userName, // Shopware global
            })
        );
    },

    /**
     * Asks the user if a given internal comment should indeed be deleted and, if the user confirms, removes
     * the internal comment from the return shipment's store of internal comments.
     *
     * @param {Shopware.apps.ViisonPickwareERPReturnShipment.model.detail.InternalComment} internalComment
     */
    onDeleteInternalComment: function (internalComment) {
        Ext.Msg.confirm(
            this.getViisonSnippet('confirm/delete_internal_comment/title'),
            this.getViisonSnippet('confirm/delete_internal_comment/text'),
            function (decision) {
                if (decision !== 'yes') {
                    return;
                }

                this.getReturnShipmentRecord().getInternalComments().remove(internalComment);
            },
            this
        );
    },

    /**
     * Creates a new mail composer window based on a given set of mail data.
     *
     * @param {object} mailData
     */
    createMailComposerWindow: function (mailData) {
        var self = this;

        // Open the email composer window
        Ext.create('Shopware.apps.ViisonCommonMailComposer.view.Window', {
            subApp: this.subApplication,
            hideAttachmentField: !(mailData.attachment || false),
            mail: mailData,
            sendMail: function () {
                this.close();
                self.sendMail(this.formData, this.mail.mailType);
            },
        }).show();
    },

    /**
     * Sends an new email based on a given set of mail data by calling the respective server action.
     *
     * @param {object} mailData
     * @param {string} mailType
     */
    sendMail: function (mailData, mailType) {
        this.mainWindow.showLoadMask(this.getViisonSnippet('window/loading_mask/send_mail/text'));
        Ext.Ajax.request({
            url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPReturnShipment/sendMail'),
            method: 'POST',
            params: {
                returnShipmentId: this.getReturnShipmentRecord().get('id'),
                mailData: Ext.JSON.encode(mailData),
                mailType: mailType,
            },
            scope: this,
            callback: function (options, success, response) {
                var result = Ext.JSON.decode(response.responseText, true);

                if (result && result.success) {
                    Shopware.Notification.createGrowlMessage(
                        this.getViisonSnippet('notification/send_mail/success/title'),
                        Ext.String.format(
                            this.getViisonSnippet('notification/send_mail/success/message'),
                            this.getReturnShipmentRecord().get('number')
                        ),
                        'ViisonPickwareERP'
                    );
                } else {
                    Shopware.Notification.createGrowlMessage(
                        this.getViisonSnippet('notification/send_mail/failure/title'),
                        Ext.String.format(
                            this.getViisonSnippet('notification/send_mail/failure/message'),
                            this.getReturnShipmentRecord().get('number'),
                            result.message
                        ),
                        'ViisonPickwareERP'
                    );
                }

                this.mainWindow.hideLoadMask();
            },
        });
    },

});
