// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPReturnShipment.controller.OrderPicker', {

    extend: 'Ext.app.Controller',

    /**
     * @type {Shopware.apps.ViisonPickwareERPReturnShipment.view.OrderPicker.Window}
     */
    mainWindow: null,

    /**
     * Callback, which will be called when the user clicks the select order button.
     *
     * @type {function}
     */
    actionCallback: Ext.emptyFn,

    refs: [
        { ref: 'orderList', selector: 'viison_pickware_erp_return_shipment-order_picker-order_list' },
    ],

    /**
     * @override
     */
    init: function () {
        this.callParent(arguments);

        this.control({
            'viison_pickware_erp_return_shipment-order_picker-window': {
                closeWindow: this.onCloseWindow,
                selectOrder: this.onSelectOrder,
            },
            'viison_pickware_erp_return_shipment-order_picker-order_list': {
                openCustomer: this.onOpenCustomer,
                openOrder: this.onOpenOrder,
                selectionchange: this.onSelectionChange,
                searchStrChange: this.onSearchStrChange,
            },
        });
    },

    /**
     * @override
     *
     * Scope refs to main window
     *
     * @param {string} selector
     * @returns {Ext.Component}
     */
    getActiveReference: function (selector) {
        return Ext.ComponentQuery.query(selector, this.mainWindow)[0];
    },

    /**
     * @param {function} actionCallback
     */
    createWindow: function (actionCallback) {
        this.actionCallback = actionCallback;
        this.mainWindow = this.getView('OrderPicker.Window').create();
        this.mainWindow.show();
    },

    onCloseWindow: function () {
        this.mainWindow.close();
    },

    onSelectOrder: function () {
        var selectedOrder = this.getOrderList().getSelectionModel().getSelection();

        this.mainWindow.close();
        this.actionCallback(selectedOrder[0]);
    },

    /**
     * @param {Shopware.apps.ViisonPickwareERPReturnShipment.model.OrderPicker.Order} orderRecord
     */
    onOpenCustomer: function (orderRecord) {
        Shopware.app.Application.addSubApplication({
            name: 'Shopware.apps.Customer',
            action: 'detail',
            params: {
                customerId: orderRecord.get('customerId'),
            },
        });
    },

    /**
     * @param {Shopware.apps.ViisonPickwareERPReturnShipment.model.OrderPicker.Order} orderRecord
     */
    onOpenOrder: function (orderRecord) {
        Shopware.app.Application.addSubApplication({
            name: 'Shopware.apps.Order',
            action: 'detail',
            params: {
                orderId: orderRecord.get('id'),
            },
        });
    },

    /**
     * Updates the states of the select button, when the user selects an order in the order list.
     *
     * @param {Ext.selection.RowModel} selectionModel
     * @param {Shopware.apps.ViisonPickwareERPReturnShipment.model.detail.Order[]} selection
     */
    onSelectionChange: function (selectionModel, selection) {
        if (selection.length > 0) {
            this.mainWindow.enableSelectButton();
        } else {
            this.mainWindow.disableSelectButton();
        }
    },

    /**
     * Reloads the order list based on a given search string.
     *
     * @param {string} searchStr
     */
    onSearchStrChange: function (searchStr) {
        var store = this.getOrderList().getStore();

        store.getProxy().extraParams.query = searchStr;
        store.loadPage(1);
        delete store.getProxy().extraParams.query;
    },

});
