// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPReturnShipment.controller.Overview', {

    extend: 'Ext.app.Controller',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_return_shipment/overview',

    /**
     * @type {Shopware.apps.ViisonPickwareERPReturnShipment.view.overview.Window}
     */
    mainWindow: null,

    refs: [
        { ref: 'returnShipmentList', selector: 'viison_pickware_erp_return_shipment-overview-return_shipment_list' },
    ],

    /**
     * @override
     */
    init: function () {
        this.callParent(arguments);

        this.control({
            'viison_pickware_erp_return_shipment-overview-return_shipment_list': {
                editReturnShipment: this.onEditReturnShipment,
                cancelReturnShipment: this.onCancelReturnShipment,
                deleteReturnShipment: this.onDeleteReturnShipment,
                openDocument: this.onOpenDocument,
                openCustomer: this.onOpenCustomer,
                openOrder: this.onOpenOrder,
                statusFilterChanged: this.onStatusFilterChanged,
                selectionchange: this.onSelectionChange,
                searchStrChange: this.onSearchStrChange,
                addReturnShipment: this.onAddReturnShipment,
                finalizeReturnShipments: this.onFinalizeReturnShipments,
            },
        });
    },

    /**
     * @override
     *
     * Scope refs to main window
     *
     * @param {string} selector
     * @returns {Ext.Component}
     */
    getActiveReference: function (selector) {
        return Ext.ComponentQuery.query(selector, this.mainWindow)[0];
    },

    createWindow: function () {
        this.mainWindow = this.getView('overview.Window').create();
        this.mainWindow.show();
    },

    /**
     * Opens the return shipment detail window for a given return shipment.
     *
     * @param {Shopware.apps.ViisonPickwareERPReturnShipment.model.overview.ReturnShipment} returnShipment
     */
    onEditReturnShipment: function (returnShipment) {
        Shopware.app.Application.addSubApplication({
            name: 'Shopware.apps.ViisonPickwareERPReturnShipment',
            action: 'detail',
            returnShipmentId: returnShipment.get('id'),
            onUpdateCallback: this.reloadReturnShipmentList.bind(this),
        });
    },

    /**
     * Opens the cancellation dialog for the given return shipment.
     *
     * @param {Shopware.apps.ViisonPickwareERPReturnShipment.model.overview.ReturnShipment} returnShipment
     */
    onCancelReturnShipment: function (returnShipment) {
        Shopware.app.Application.addSubApplication({
            name: 'Shopware.apps.ViisonPickwareERPCancellation',
            orderId: returnShipment.get('orderId'),
            returnShipmentId: returnShipment.get('id'),
            afterCancellationCallback: this.reloadReturnShipmentList.bind(this),
        });
    },

    reloadReturnShipmentList: function () {
        if (this.getReturnShipmentList()) {
            this.getReturnShipmentList().getStore().loadPage(1);
        }
    },

    /**
     * Asks the user if the given return shipment should indeed be deleted and, if the user confirms,
     * deletes it by calling the respective server action.
     *
     * @param {Shopware.apps.ViisonPickwareERPReturnShipment.model.overview.ReturnShipment} returnShipmentToBeDeleted
     */
    onDeleteReturnShipment: function (returnShipmentToBeDeleted) {
        var message = Ext.String.format(
            this.getViisonSnippet('confirm/delete_return_shipment/text'),
            returnShipmentToBeDeleted.get('number'),
            returnShipmentToBeDeleted.get('id')
        );
        if (returnShipmentToBeDeleted.get('documentId') !== null) {
            message += '<br /><br />' + this.getViisonSnippet('confirm/delete_return_shipment/text_cancellation_invoice');
        }
        Ext.Msg.confirm(
            this.getViisonSnippet('confirm/delete_return_shipment/title'),
            message,
            function (decision) {
                if (decision !== 'yes') {
                    return;
                }

                this.mainWindow.showLoadMask(this.getViisonSnippet('action/delete_return_shipment/loading_message'));
                Ext.Ajax.request({
                    url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPReturnShipment/deleteReturnShipment'),
                    method: 'POST',
                    params: {
                        returnShipmentId: returnShipmentToBeDeleted.get('id'),
                    },
                    scope: this,
                    callback: function (options, success, response) {
                        var result = Ext.JSON.decode(response.responseText, true);

                        if (result && result.success) {
                            Shopware.Notification.createGrowlMessage(
                                this.getViisonSnippet('growl_message/delete_return_shipment/success/title'),
                                Ext.String.format(
                                    this.getViisonSnippet('growl_message/delete_return_shipment/success/message'),
                                    returnShipmentToBeDeleted.get('number'),
                                    returnShipmentToBeDeleted.get('id')
                                ),
                                'ViisonPickwareERP'
                            );
                        } else {
                            Shopware.Notification.createGrowlMessage(
                                this.getViisonSnippet('growl_message/delete_return_shipment/failure/title'),
                                Ext.String.format(
                                    this.getViisonSnippet('growl_message/delete_return_shipment/failure/message'),
                                    returnShipmentToBeDeleted.get('number'),
                                    returnShipmentToBeDeleted.get('id'),
                                    result.message
                                ),
                                'ViisonPickwareERP'
                            );
                        }

                        this.mainWindow.hideLoadMask();
                        this.getReturnShipmentList().getStore().loadPage(1);
                    },
                });
            },
            this
        );
    },

    /**
     * Opens the cancellation Invoice document of a given return shipment in a new browser window / tab.
     *
     * @param {Shopware.apps.ViisonPickwareERPReturnShipment.model.overview.ReturnShipment} returnShipment
     */
    onOpenDocument: function (returnShipment) {
        window.open(ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPReturnShipment/openCorrectionOfInvoice', {
            documentId: returnShipment.get('documentId'),
        }));
    },

    /**
     * @param {Shopware.apps.ViisonPickwareERPReturnShipment.model.overview.ReturnShipment} returnShipment
     */
    onOpenCustomer: function (returnShipment) {
        Shopware.app.Application.addSubApplication({
            name: 'Shopware.apps.Customer',
            action: 'detail',
            params: {
                customerId: returnShipment.get('customerId'),
            },
        });
    },

    /**
     * @param {Shopware.apps.ViisonPickwareERPReturnShipment.model.overview.ReturnShipment} returnShipment
     */
    onOpenOrder: function (returnShipment) {
        Shopware.app.Application.addSubApplication({
            name: 'Shopware.apps.Order',
            action: 'detail',
            params: {
                orderId: returnShipment.get('orderId'),
            },
        });
    },

    /**
     * Updates the status filter property of the return shipments store based of a given set of status ids.
     *
     * @param {number[]} selectedStatusIds
     */
    onStatusFilterChanged: function (selectedStatusIds) {
        this.getStore('overview.ReturnShipment').setStatusFilter(selectedStatusIds);
    },

    /**
     * Updates the status of the batch buttons based on the given selection of return shipments.
     *
     * @param {Ext.selection.rowModel} selectionModel
     * @param {Shopware.apps.ViisonPickwareERPReturnShipment.model.overview.ReturnShipment[]} selection
     */
    onSelectionChange: function (selectionModel, selection) {
        if (selection.length > 0) {
            this.getReturnShipmentList().enableSelectionBasedButtons();
        } else {
            this.getReturnShipmentList().disableSelectionBasedButtons();
        }
    },

    /**
     * Opens the order picker and triggers the creation of a new return shipment based on the selected order.
     */
    onAddReturnShipment: function () {
        this.getController('OrderPicker').createWindow(function (orderRecord) {
            this.createReturnShipment(orderRecord);
        }.bind(this));
    },

    /**
     * Asks the user if the selected set of return shipments should indeed be finalized and, if the the user
     * confirms, finalizes them by calling the respective server action.
     */
    onFinalizeReturnShipments: function () {
        var selectedReturnShipments = this.getReturnShipmentList().getSelectionModel().getSelection();
        if (selectedReturnShipments.length === 0) {
            return;
        }

        var returnShipmentIds = [];
        var returnShipmentNumbers = [];
        Ext.Array.forEach(selectedReturnShipments, function (returnShipmentRecord) {
            returnShipmentIds.push(returnShipmentRecord.get('id'));
            returnShipmentNumbers.push(returnShipmentRecord.get('number'));
        });

        var message = Ext.String.format(
            this.getViisonSnippet('confirm/finalize_return_shipments/text'),
            returnShipmentIds.length
        );

        var returnShipmentNotInStatusCancelledOrFinalized = selectedReturnShipments.find(function (returnShipment) {
            return !returnShipment.isTypeCancelled() && !returnShipment.isTypeFinalized();
        });
        if (returnShipmentNotInStatusCancelledOrFinalized) {
            message += '<br /><br />';
            message += this.getViisonSnippet('confirm/finalize_return_shipments/warning');
        }

        var createStatusMailSendingCheckbox = Ext.create('Ext.form.field.Checkbox', {
            boxLabel: this.getViisonSnippet('confirm/finalize_return_shipments/automatic_status_mail_sending'),
            checked: true,
            style: {
                'margin-left': '20px',
            },
        });

        Ext.create('Shopware.apps.ViisonPickwareERPMessageBoxWithCheckboxes.view.MessageBoxWithCheckboxes').show({
            title: this.getViisonSnippet('confirm/finalize_return_shipments/title'),
            icon: Ext.Msg.QUESTION,
            msg: message,
            checkboxes: [
                createStatusMailSendingCheckbox,
            ],
            buttons: Ext.Msg.YESNO,
            scope: this,
            callback: function (decision) {
                if (decision !== 'yes') {
                    return;
                }

                this.mainWindow.showLoadMask(this.getViisonSnippet('action/finalize_return_shipments/loading_message'));
                Ext.Ajax.request({
                    url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPReturnShipment/batchFinalizeReturnShipments'),
                    method: 'POST',
                    params: {
                        returnShipmentIds: Ext.JSON.encode(returnShipmentIds),
                        shouldSendStatusMails: createStatusMailSendingCheckbox.checked,
                    },
                    scope: this,
                    callback: function (options, success, response) {
                        var result = Ext.JSON.decode(response.responseText, true);

                        if (result && result.success) {
                            Shopware.Notification.createGrowlMessage(
                                this.getViisonSnippet('growl_message/finalize_return_shipments/success/title'),
                                Ext.String.format(
                                    this.getViisonSnippet('growl_message/finalize_return_shipments/success/message'),
                                    returnShipmentNumbers.join(', '),
                                    returnShipmentIds.join(', ')
                                ),
                                'ViisonPickwareERP'
                            );
                        } else {
                            Shopware.Notification.createGrowlMessage(
                                this.getViisonSnippet('growl_message/finalize_return_shipments/failure/title'),
                                Ext.String.format(
                                    this.getViisonSnippet('growl_message/finalize_return_shipments/failure/message'),
                                    returnShipmentNumbers.join(', '),
                                    returnShipmentIds.join(', '),
                                    result.message
                                ),
                                'ViisonPickwareERP'
                            );
                        }

                        this.mainWindow.hideLoadMask();
                        this.getReturnShipmentList().getStore().loadPage(1);
                    },
                });
            },
        });
    },

    /**
     * Reloads the return shipments list based on a given search string.
     *
     * @param {string} searchStr
     */
    onSearchStrChange: function (searchStr) {
        var store = this.getReturnShipmentList().getStore();

        store.getProxy().extraParams.query = searchStr;
        store.loadPage(1);
    },

    /**
     * @param {Shopware.apps.ViisonPickwareERPReturnShipment.model.OrderPicker.Order} order
     */
    createReturnShipment: function (order) {
        Shopware.app.Application.addSubApplication({
            name: 'Shopware.apps.ViisonPickwareERPReturnShipment',
            action: 'detail',
            orderId: order.get('id'),
            onUpdateCallback: this.reloadReturnShipmentList.bind(this),
        });
    },

});
