// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPReturnShipment.view.detail.tab.ReturnShipmentItemList', {

    extend: 'Ext.grid.Panel',
    alias: 'widget.viison_pickware_erp_return_shipment-detail-tab-return_shipment_item_list',
    cls: 'viison_pickware_erp_return_shipment-detail-tab-return_shipment_item_list viison-common--grid has--vertical-lines',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_return_shipment/detail',

    stateful: true,
    stateId: 'viison_pickware_erp_return_shipment-detail-tab-return_shipment_item_list',

    selModel: 'viisonCommonTabableCellModel',

    style: {
        border: 'none !important',
    },
    bodyStyle: {
        borderTop: 'none !important',
        borderLeft: 'none !important',
        borderRight: 'none !important',
    },

    viewConfig: {
        enableTextSelection: true,
        trackOver: true,
        selectedItemCls: '',
        selectedCellCls: '',
        markDirty: false,
        deferEmptyText: false,
    },

    /**
     * @override
     */
    initComponent: function () {
        this.emptyText = Ext.String.format(
            '<strong>[0]</strong>[1]',
            this.getViisonSnippet('tab/return_shipment_item_list/empty_text/headline'),
            this.getViisonSnippet('tab/return_shipment_item_list/empty_text/body')
        );

        this.title = this.getViisonSnippet('tab/return_shipment_item_list/title');
        this.plugins = this.createPlugins();
        this.columns = this.createColumns();
        this.dockedItems = this.createDockedItems();
        this.store = this.subApp.getController('Detail').getReturnShipmentRecord().getItems();

        this.callParent(arguments);

        this.subApp.getController('Detail').on('returnShipmentRecordSaved', function () {
            this.reconfigureView();
        }, this);
    },

    /**
     * @returns {boolean}
     */
    itemsCanBeEdited: function () {
        var returnShipmentRecord = this.subApp.getController('Detail').getReturnShipmentRecord();

        return !(returnShipmentRecord.isTypeCancelled() || returnShipmentRecord.isTypeFinalized());
    },

    /**
     * @returns {object[]}
     */
    createColumns: function () {
        this.returnedQuantityColumn = Ext.create('Ext.grid.column.Column', {
            header: this.getViisonSnippet('tab/return_shipment_item_list/column/returned_quantity'),
            dataIndex: 'returnedQuantity',
            sortable: true,
            align: 'right',
            width: 120,
            editor: {
                xtype: 'numberfield',
                cls: 'is--viison-common-grid--cell-editor is--align-right',
                allowBlank: false,
                minValue: 0,
                enableKeyEvents: true,
            },
        });

        return [
            {
                header: this.getViisonSnippet('tab/return_shipment_item_list/column/article_number'),
                dataIndex: 'articleNumber',
                sortable: true,
                width: 120,
            }, {
                header: this.getViisonSnippet('tab/return_shipment_item_list/column/article_name'),
                dataIndex: 'articleName',
                sortable: true,
                flex: 1,
            }, this.returnedQuantityColumn, {
                header: this.getViisonSnippet('tab/return_shipment_item_list/column/written_off_quantity'),
                dataIndex: 'writtenOffQuantity',
                sortable: true,
                align: 'right',
                width: 120,
                editor: {
                    xtype: 'numberfield',
                    cls: 'is--viison-common-grid--cell-editor is--align-right',
                    allowBlank: false,
                    minValue: 0,
                    enableKeyEvents: true,
                },
            }, {
                xtype: 'actioncolumn',
                width: 80,
                align: 'center',
                items: [
                    {
                        iconCls: 'sprite-minus-circle x-action-col-icon',
                        tooltip: this.getViisonSnippet(
                            'tab/return_shipment_item_list/column/button/delete_return_shipment_item/tooltip'
                        ),
                        scope: this,
                        handler: function (grid, rowIndex) {
                            if (this.itemsCanBeEdited()) {
                                this.fireEvent('deleteReturnShipmentItem', grid.getStore().getAt(rowIndex));
                            }
                        },
                        getClass: function () {
                            if (!this.itemsCanBeEdited()) {
                                return 'is--disabled';
                            }

                            return '';
                        }.bind(this),
                    },
                    {
                        iconCls: 'sprite-inbox',
                        tooltip: this.getViisonSnippet(
                            'tab/return_shipment_item_list/column/button/open_article/tooltip'
                        ),
                        scope: this,
                        handler: function (grid, rowIndex) {
                            this.fireEvent('openArticle', grid.getStore().getAt(rowIndex));
                        },
                    },
                    {
                        iconCls: 'sprite-pencil',
                        tooltip: this.getViisonSnippet(
                            'tab/return_shipment_item_list/column/button/edit/tooltip'
                        ),
                        scope: this,
                        handler: function (grid, rowIndex) {
                            this.fireEvent('editReturnShipmentItem', grid.getStore().getAt(rowIndex));
                        },
                        getClass: function () {
                            if (!this.itemsCanBeEdited()) {
                                return 'is--disabled';
                            }

                            return '';
                        }.bind(this),
                    },
                ],
            },
        ];
    },

    /**
     * @returns {Ext.toolbar.Toolbar[]}
     */
    createDockedItems: function () {
        return [
            Ext.create('Ext.toolbar.Toolbar', {
                dock: 'top',
                ui: 'shopware-ui',
                cls: 'is--viison-common--toolbar',
                items: [
                    {
                        xtype: 'button',
                        itemId: 'addReturnShipmentItems',
                        text: this.getViisonSnippet(
                            'tab/return_shipment_item_list/toolbar/button/add_return_shipment_items/text'
                        ),
                        iconCls: 'icon-add',
                        margin: '0 5 0 0',
                        disabled: !this.itemsCanBeEdited(),
                        scope: this,
                        handler: function () {
                            this.fireEvent('addReturnShipmentItems');
                        },
                    },
                ],
            }),
        ];
    },

    /**
     * @returns {Ext.grid.plugin.CellEditing[]}
     */
    createPlugins: function () {
        this.cellEditingPlugin = Ext.create('Ext.grid.plugin.CellEditing', {
            clicksToEdit: 1,
            listeners: {
                scope: this,
                beforeedit: this.onBeforeEdit,
                edit: this.onEdit,
            },
        });

        return [
            this.cellEditingPlugin,
        ];
    },

    /**
     * Applies the max value constraints for returned quantity and written of quantity and ensure that
     * return shipment items of return shipments with status > canceled can not be edited.
     *
     * @param {Ext.grid.plugin.CellEditing} plugin
     * @param {object} editorEvent
     * @returns {boolean}
     */
    onBeforeEdit: function (plugin, editEvent) {
        var editor = plugin.getEditor(editEvent.record, editEvent.column);

        if (!this.itemsCanBeEdited()) {
            // Prevent return shipment items of return shipments with status > canceled from being edited
            return false;
        }

        if (editEvent.field === 'returnedQuantity') {
            editor.field.setMaxValue(editEvent.record.get('shippedQuantity'));
        }

        if (editEvent.field === 'writtenOffQuantity') {
            editor.field.setMaxValue(editEvent.record.get('returnedQuantity'));
        }

        return true;
    },

    /**
     * Updates the edited return shipment and ensures that the written of quantity never exceeds the value of
     * the related returned quantity.
     *
     * @param {Ext.grid.plugin.CellEditing} plugin
     * @param {object} editorEvent
     */
    onEdit: function (plugin, editorEvent) {
        if (editorEvent.originalValue !== editorEvent.value) {
            switch (editorEvent.field) {
                case 'returnedQuantity':
                    editorEvent.record.set('returnedQuantity', editorEvent.value);
                    if (editorEvent.record.get('writtenOffQuantity') > editorEvent.record.get('returnedQuantity')) {
                        editorEvent.record.set('writtenOffQuantity', editorEvent.record.get('returnedQuantity'));
                    }
                    break;
                case 'writtenOffQuantity':
                    editorEvent.record.set('writtenOffQuantity', editorEvent.value);
                    break;
                default:
            }
        }

        this.fireEvent('itemChanged', editorEvent.record);
    },

    reconfigureView: function () {
        this.down('#addReturnShipmentItems').setDisabled(!this.itemsCanBeEdited());
        this.reconfigure(this.subApp.getController('Detail').getReturnShipmentRecord().getItems());
    },

});
