// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPReturnShipment.view.overview.ReturnShipmentList', {

    extend: 'Ext.grid.Panel',
    alias: 'widget.viison_pickware_erp_return_shipment-overview-return_shipment_list',
    cls: 'viison_pickware_erp_return_shipment-overview-return_shipment_list viison-common--grid has--vertical-lines',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_return_shipment/overview',

    stateful: true,
    stateId: 'viison_pickware_erp_return_shipment-overview-return_shipment_list',

    style: {
        borderLeft: 'none !important',
        borderRight: 'none !important',
    },
    bodyStyle: {
        borderTop: 'none !important',
        borderLeft: 'none !important',
        borderRight: 'none !important',
    },

    viewConfig: {
        enableTextSelection: true,
        trackOver: true,
        markDirty: false,
    },

    /**
     * @override
     */
    initComponent: function () {
        this.emptyText = this.getViisonSnippet('return_shipment_list/empty_text');
        this.store = this.subApp.getStore('overview.ReturnShipment');
        this.columns = this.createColumns();
        this.dockedItems = this.createDockedItems();
        this.selModel = this.createSelectionModel();

        this.callParent(arguments);

        setTimeout(function () {
            // Load store in next tick, since otherwise the emptyText will
            // not be displayed properly
            this.store.load();
        }.bind(this), 100);
    },

    /**
     * @returns {Ext.selection.CheckboxModel}
     */
    createSelectionModel: function () {
        return Ext.create('Ext.selection.CheckboxModel', {
            mode: 'SIMPLE',
            checkOnly: false,
        });
    },

    /**
     * @returns {object[]}
     */
    createColumns: function () {
        return [
            {
                xtype: 'datecolumn',
                header: this.getViisonSnippet('return_shipment_list/column/created'),
                dataIndex: 'created',
                sortable: true,
                width: 100,
                format: 'd.m.Y H:i',
            }, {
                header: this.getViisonSnippet('return_shipment_list/column/number'),
                dataIndex: 'number',
                sortable: true,
                width: 80,
            }, {
                header: this.getViisonSnippet('return_shipment_list/column/customer_name'),
                dataIndex: 'customerName',
                sortable: true,
                flex: 1,
                renderer: function (value, metaData, record) {
                    var customerCompanyName = record.get('customerCompanyName');
                    if (!customerCompanyName) {
                        return value;
                    }

                    return value + ' (' + customerCompanyName + ')';
                },
            }, {
                header: this.getViisonSnippet('return_shipment_list/column/customer_email'),
                dataIndex: 'customerEmail',
                sortable: true,
                flex: 1,
                hidden: true,
            }, {
                header: this.getViisonSnippet('return_shipment_list/column/customer_number'),
                dataIndex: 'customerNumber',
                sortable: true,
                width: 80,
                hidden: true,
            }, {
                header: this.getViisonSnippet('return_shipment_list/column/customer_group_key'),
                dataIndex: 'customerGroupKey',
                sortable: true,
                width: 90,
                hidden: true,
            }, {
                header: this.getViisonSnippet('return_shipment_list/column/order_number'),
                dataIndex: 'orderNumber',
                sortable: true,
                width: 80,
            }, {
                header: this.getViisonSnippet('return_shipment_list/column/order_shop_name'),
                dataIndex: 'orderShopName',
                sortable: true,
                width: 80,
                hidden: true,
            }, {
                header: this.getViisonSnippet('return_shipment_list/column/shipping_method_name'),
                dataIndex: 'shippingMethodName',
                sortable: true,
                width: 100,
                hidden: true,
            }, {
                header: this.getViisonSnippet('return_shipment_list/column/order_status'),
                dataIndex: 'orderStatusName',
                sortable: true,
                width: 100,
                renderer: function (value) {
                    return ViisonStatusTranslator.translateOrderStatus(value);
                },
            }, {
                header: this.getViisonSnippet('return_shipment_list/column/payment_method_name'),
                dataIndex: 'paymentMethodName',
                sortable: true,
                width: 100,
            }, {
                header: this.getViisonSnippet('return_shipment_list/column/payment_status'),
                dataIndex: 'paymentStatusName',
                sortable: true,
                width: 100,
                renderer: function (value) {
                    return ViisonStatusTranslator.translatePaymentStatus(value);
                },
            }, {
                header: this.getViisonSnippet('return_shipment_list/column/number_of_returned_items'),
                dataIndex: 'numberOfReturnedItems',
                sortable: false,
                width: 75,
                align: 'right',
                hidden: true,
            }, {
                header: this.getViisonSnippet('return_shipment_list/column/amount'),
                dataIndex: 'amount',
                sortable: false,
                width: 90,
                align: 'right',
                renderer: ViisonCurrencyFormatter.renderer,
            }, {
                header: this.getViisonSnippet('return_shipment_list/column/status_name'),
                dataIndex: 'statusId',
                sortable: true,
                width: 90,
                renderer: function (value) {
                    return ViisonPickwareERPReturnShipmentReturnShipmentStatus.localizedDisplayName(value);
                },
            }, {
                header: this.getViisonSnippet('return_shipment_list/column/user_name'),
                dataIndex: 'userName',
                sortable: true,
                flex: 1,
                hidden: true,
            }, {
                xtype: 'actioncolumn',
                width: 140,
                align: 'center',
                items: [
                    {
                        iconCls: 'sprite-user',
                        tooltip: this.getViisonSnippet('return_shipment_list/button/open_customer/tooltip'),
                        scope: this,
                        handler: function (grid, rowIndex) {
                            this.fireEvent('openCustomer', grid.getStore().getAt(rowIndex));
                        },
                    }, {
                        iconCls: 'sprite-sticky-notes-pin',
                        tooltip: this.getViisonSnippet('return_shipment_list/button/open_order/tooltip'),
                        scope: this,
                        handler: function (grid, rowIndex) {
                            this.fireEvent('openOrder', grid.getStore().getAt(rowIndex));
                        },
                    }, {
                        iconCls: 'sprite-document-pdf-text',
                        handler: function (grid, rowIndex) {
                            var record = grid.getStore().getAt(rowIndex);

                            if (record.get('documentId')) {
                                this.fireEvent('openDocument', record);
                            }
                        }.bind(this),
                        getClass: function (html, metadata, record) {
                            if (!record.get('documentId')) {
                                this.items[2].tooltip = ViisonCommonApp.getSnippet(
                                    'return_shipment_list/column/action/open_document/disabled/info',
                                    'backend/viison_pickware_erp_return_shipment/overview'
                                );

                                return 'is--disabled';
                            }

                            this.items[2].tooltip = ViisonCommonApp.getSnippet(
                                'return_shipment_list/column/action/open_document/info',
                                'backend/viison_pickware_erp_return_shipment/overview'
                            );

                            return '';
                        },
                    }, {
                        iconCls: 'sprite-minus-circle',
                        tooltip: this.getViisonSnippet('return_shipment_list/button/delete_return_shipment/tooltip'),
                        scope: this,
                        handler: function (grid, rowIndex) {
                            this.fireEvent('deleteReturnShipment', grid.getStore().getAt(rowIndex));
                        },
                    }, {
                        iconCls: 'sprite-pencil',
                        tooltip: this.getViisonSnippet('return_shipment_list/button/edit_return_shipment/tooltip'),
                        scope: this,
                        handler: function (grid, rowIndex) {
                            this.fireEvent('editReturnShipment', grid.getStore().getAt(rowIndex));
                        },
                    },
                ],
            },
        ];
    },

    /**
     * @returns {object[]}
     */
    createDockedItems: function () {
        return [
            Ext.create('Ext.toolbar.Toolbar', {
                dock: 'top',
                ui: 'shopware-ui',
                cls: 'is--viison-common--toolbar',
                items: [
                    {
                        xtype: 'button',
                        text: this.getViisonSnippet('toolbar/button/add_return_shipment'),
                        iconCls: 'icon-add',
                        margin: '0 5 0 0',
                        scope: this,
                        handler: function () {
                            this.fireEvent('addReturnShipment');
                        },
                    },
                    {
                        xtype: 'button',
                        itemId: 'finalizationButton',
                        text: this.getViisonSnippet('toolbar/button/finalize_return_shipments'),
                        iconCls: 'icon-finalize-return-shipments',
                        disabled: true,
                        scope: this,
                        handler: function () {
                            this.fireEvent('finalizeReturnShipments');
                        },
                    },
                    '->',
                    {
                        xtype: 'label',
                        cls: 'is--viison-common--label is--status-filter-label',
                        text: 'Status:',
                        margin: '10 10 10 10',
                    },
                    {
                        xtype: 'combobox',
                        itemId: 'statusFilter',
                        cls: 'is--status-filter',
                        hideLabel: true,
                        width: 180,
                        multiSelect: true,
                        emptyText: 'Alle anzeigen',
                        store: this.subApp.getStore('ReturnShipmentStatus'),
                        queryMode: 'local',
                        displayField: 'displayName',
                        valueField: 'id',
                        listeners: {
                            collapse: function (combobox) {
                                this.fireEvent('statusFilterChanged', combobox.getValue());
                            },
                            scope: this,
                        },
                    },
                    {
                        xtype: 'textfield',
                        name: 'searchfield',
                        cls: 'searchfield',
                        width: 170,
                        enableKeyEvents: true,
                        emptyText: this.getViisonSnippet('toolbar/search/placeholder'),
                        listeners: {
                            buffer: 500,
                            change: function (field) {
                                this.fireEvent('searchStrChange', field.value);
                            },
                            scope: this,
                        },
                    },
                ],
            }),
            {
                xtype: 'viison_common_pagination_toolbar-toolbar',
                store: this.store,
            },
        ];
    },

    enableSelectionBasedButtons: function () {
        this.down('#finalizationButton').setDisabled(false);
    },

    disableSelectionBasedButtons: function () {
        this.down('#finalizationButton').setDisabled(true);
    },

});
