// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPStockInitialization.controller.Main', {

    extend: 'Enlight.app.Controller',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_stock_initialization/main',

    refs: [
        { ref: 'articleGrid', selector: 'viison_pickware_erp_stock_initialization-list' },
        { ref: 'main', selector: 'viison_pickware_erp_stock_initialization-main' },
    ],

    init: function () {
        this.control({
            'viison_pickware_erp_stock_initialization-list': {
                openArticle: this.onOpenArticle,
                editRow: this.onEditRow,
                applyChanges: this.onApplyChanges,
                initializeAll: this.onInitializeAll,
            },
        });

        this.mainWindow = this.getView('Main').create({
            store: this.getStore('Stock'),
        }).show();
    },

    /**
     * Opens a new article sub application window for the article with the given ID.
     *
     * @param articleId The id of the article whose details window shall be opened.
     */
    onOpenArticle: function (record) {
        Shopware.app.Application.addSubApplication({
            name: 'Shopware.apps.Article',
            action: 'detail',
            params: {
                articleId: record.get('articleId'),
            },
        });
    },

    /**
     * Start the row editing for the given record.
     */
    onEditRow: function (record) {
        this.getArticleGrid().rowEditing.startEdit(record, 0);
    },

    /**
     * Inserts stock entries one by one, chained with callbacks.
     *
     * @param Shopware.apps.ViisonPickwareERPStockInitialization.view.List panel
     */
    onApplyChanges: function (panel) {
        var selectedArticles = panel.getSelectionModel().getSelection();
        if (selectedArticles.length > 0) {
            this.getMain().setLoading(true);
            this.createStockEntries(selectedArticles);
        }
    },

    /**
     * Inserts stock entries one by one, chained with callbacks.
     *
     * @param {Shopware.apps.ViisonPickwareERPStockInitialization.view.List} panel
     * @param {boolean} skipAlert (optional)
     */
    onInitializeAll: function (panel, skipAlert) {
        if (skipAlert !== true) {
            Ext.Msg.confirm(
                this.getViisonSnippet('alert/initialize_all/title'),
                this.getViisonSnippet('alert/initialize_all/message'),
                function (buttonId) {
                    if (buttonId === 'yes') {
                        // Call this subscriber again to initialize all stocks
                        this.onInitializeAll(panel, true);
                    }
                },
                this
            );

            return;
        }

        var paginationToolbar = panel.down('[name=viisonStockInitializationPaginationToolbar]');
        if (!paginationToolbar) {
            return;
        }

        this.getMain().setLoading(true);
        var uninitializedArticlesCount = paginationToolbar.getStore().totalCount;
        var progressMessageBox = Ext.Msg.show({
            title: this.getViisonSnippet('progress_bar_title'),
            msg: Ext.String.format(this.getViisonSnippet('progress_bar_text'), uninitializedArticlesCount),
            buttons: Ext.Msg.CANCEL,
            progress: true,
        });
        progressMessageBox.setWidth(380);
        progressMessageBox.setHeight(160);
        progressMessageBox.center();
        progressMessageBox.updateProgress(0, 0 + ' / ' + uninitializedArticlesCount);

        this.callInitializeArticlesBatchWise(progressMessageBox, uninitializedArticlesCount, function (err) {
            progressMessageBox.close();
            this.getMain().setLoading(false);
            panel.store.reload();
            panel.getView().refresh();

            if (err) {
                Ext.Msg.alert({
                    title: this.getViisonSnippet('notification/initialize_all/fail_title'),
                    msg: Ext.String.format(this.getViisonSnippet('notification/initialize_all/fail_message'), err.message),
                    buttons: Ext.Msg.OK,
                    icon: Ext.Msg.ERROR,
                });
            }
        }.bind(this));
    },

    /**
     * A recursive function that calls for the controller to initialize all uninitialized articles. Calls the
     * controller only with a batch size (the controller decides which pending articles are initialized next).
     *
     * Recalls itself until the controller confirms that no pending articles are left, or a request fails.
     *
     * @param {Ext.window.MessageBox} progressMessageBox
     * @param {int} maxCount
     * @param callback
     */
    callInitializeArticlesBatchWise: function (progressMessageBox, maxCount, callback) {
        if (!progressMessageBox.isVisible()) {
            callback(null);

            return;
        }
        Ext.Ajax.request({
            url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPStockInitialization/initializeStocksForABatchOfUninitializedArticles'),
            method: 'GET',
            scope: this,
            success: function (response) {
                var json = Ext.JSON.decode(response.responseText, true);
                if (!json) {
                    callback(new Error('Could not parse JSON: ' + response.responseText));

                    return;
                }

                if (!json.success) {
                    callback(new Error('Error: ' + json.message));

                    return;
                }

                progressMessageBox.updateProgress(
                    (maxCount - json.pending) / maxCount,
                    (maxCount - json.pending) + ' / ' + maxCount
                );

                if (json.pending !== 0) {
                    this.callInitializeArticlesBatchWise(progressMessageBox, maxCount, callback);

                    return;
                }

                Shopware.Notification.createGrowlMessage(
                    this.getViisonSnippet('notification/initialize_all/title'),
                    this.getViisonSnippet('notification/initialize_all/success_message'),
                    'Pickware'
                );

                callback(null);
            },
            failure: function (response) {
                var errorString = response.status + ' ' + response.statusText;
                if (response.responseText) {
                    errorString += '\n' + response.responseText;
                }
                callback(new Error('Network failure: ' + errorString));
            },
        });
    },

    /**
     * Creates a new stock entry for the first item in the given articles array.
     *
     * @param articles An array containing those articles for which a new stock entry shall be created.
     */
    createStockEntries: function (articles) {
        var detailIds = Ext.Array.map(articles, function (article) {
            return article.get('detailsId');
        });

        Ext.Ajax.request({
            url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPStockInitialization/createStockEntry'),
            method: 'POST',
            jsonData: {
                detailIds: detailIds,
            },
            scope: this,
            callback: function () {
                this.getMain().setLoading(false);
                this.getArticleGrid().store.reload();
                this.getArticleGrid().getView().refresh();
            },
        });
    },

});
