// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPStockInitialization.view.List', {

    extend: 'Ext.grid.Panel',
    alias: 'widget.viison_pickware_erp_stock_initialization-list',
    // Use a bug in ExtJS which removes the background of the row editing row to make the values in all columns visible
    cls: 'x-tabpanel-child viison_pickware_erp_stock_initialization-list viison-common--grid has--vertical-lines',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_stock_initialization/main',

    viewConfig: {
        getRowClass: function (record) {
            return (!record.isSane()) ? 'is--warning' : '';
        },
    },

    listeners: {
        select: function (selectionModel, record) {
            // Deselect if record is not sane, wait 300ms for UX
            if (!record.isSane()) {
                setTimeout(function () {
                    selectionModel.deselect(record);
                }, 300);
            }
        },
    },

    initComponent: function () {
        var selectedRows = null;

        this.selModel = Ext.create('Ext.selection.CheckboxModel', {
            mode: 'SIMPLE',
            checkOnly: true,
        });

        this.rowEditing = Ext.create('Ext.grid.plugin.RowEditing', {
            clicksToEdit: 2,
            listeners: {
                scope: this,
                edit: function (editor, e) {
                    e.record.set('actualStock', e.newValues.instock + e.newValues.openOrders);
                    e.record.set('purchasePrice', e.newValues.purchasePrice);

                    e.record.save();
                    this.getView().refreshNode(e.record);
                },

                /*
                 * Fix selection behaviour of row editor
                 * When using the row editor, the selection of the gridpanel will be overwritten
                 * We store and recover the selection state to counter this effect
                 */
                beforeedit: function () {
                    var selectionModel = this.getSelectionModel();
                    selectedRows = selectionModel.getSelection();

                    var restoreSelection = function () {
                        selectionModel.select(selectedRows);
                    };

                    selectionModel.on('selectionchange', restoreSelection, selectionModel, { single: true });
                },
                canceledit: function () {
                    var selectionModel = this.getSelectionModel();
                    selectionModel.select(selectedRows);
                },
            },
        });

        this.plugins = [this.rowEditing];

        this.columns = [{
            header: this.getViisonSnippet('list/column/article_number'),
            sortable: true,
            dataIndex: 'orderNumber',
            flex: 1,
        }, {
            header: this.getViisonSnippet('list/column/article_name'),
            sortable: true,
            dataIndex: 'articleName',
            flex: 2,
            renderer: function (value, metaDatam, record) {
                var additionalText = record.get('additionalText');

                if (additionalText && additionalText.length > 0) {
                    return record.get('articleName') + ' - ' + additionalText;
                }

                return record.get('articleName');
            },
        }, {
            header: this.getViisonSnippet('list/column/supplier_name'),
            sortable: true,
            dataIndex: 'supplier',
            flex: 2,
        }, {
            header: this.getViisonSnippet('list/column/available_stock/label'),
            sortable: true,
            dataIndex: 'instock',
            align: 'right',
            flex: 1,
            editor: {
                xtype: 'numberfield',
                allowBlank: false,
            },
        }, {
            header: this.getViisonSnippet('list/column/reserved_stock/label'),
            sortable: true,
            dataIndex: 'openOrders',
            align: 'right',
            flex: 1,
        }, {
            header: this.getViisonSnippet('list/column/physical_stock/label'),
            sortable: true,
            dataIndex: 'actualStock',
            align: 'right',
            flex: 1,
            renderer: function (value, meta, record) {
                if (!record.isSane()) {
                    meta.tdCls = 'has--icon is--warning';
                }

                return value;
            },
        }, {
            header: ViisonPickwarePurchasePriceHelper.purchasePriceLabel(),
            sortable: true,
            dataIndex: 'purchasePrice',
            align: 'right',
            flex: 1,
            editor: {
                xtype: 'numberfield',
                allowBlank: false,
            },
            renderer: ViisonCurrencyFormatter.renderer,
        }, {
            xtype: 'actioncolumn',
            align: 'center',
            width: 60,
            items: [{
                iconCls: 'sprite-inbox',
                action: 'articleopen',
                tooltip: this.getViisonSnippet('list/column/action/open_article/tooltip'),
                scope: this,
                handler: function (view, rowIndex) {
                    this.fireEvent('openArticle', view.getStore().getAt(rowIndex));
                },
            }, {
                iconCls: 'sprite-pencil',
                action: 'startEdit',
                tooltip: this.getViisonSnippet('list/column/action/edit/tooltip'),
                scope: this,
                handler: function (view, rowIndex) {
                    this.fireEvent('editRow', view.getStore().getAt(rowIndex));
                },
            }],
        }];

        this.dockedItems = [
            this.createSearchBar(),
            this.createButtonPanel(),
            this.createPagingbar(),
        ];

        this.callParent(arguments);
    },

    /**
     * Creates and returns a new paging toolbar and adds a page size selection to it.
     *
     * @return The created page toolbar.
     */
    createPagingbar: function () {
        return {
            xtype: 'viison_common_pagination_toolbar-toolbar',
            name: 'viisonStockInitializationPaginationToolbar',
            store: this.store,
            paginationSteps: [100, 250, 500, 1000],
        };
    },

    createSearchBar: function () {
        // Create toolbar
        return {
            xtype: 'toolbar',
            dock: 'top',
            ui: 'shopware-ui is--viison-common--toolbar',
            items: [
                '->',
                {
                    xtype: 'textfield',
                    name: 'searchfield',
                    cls: 'searchfield',
                    width: 200,
                    enableKeyEvents: true,
                    emptyText: this.getViisonSnippet('search/initialText'),
                    listeners: {
                        buffer: 500,
                        scope: this,
                        change: function (field) {
                            this.applyFilter(field.value);
                        },
                    },
                },
            ],
        };
    },

    createButtonPanel: function () {
        return {
            xtype: 'toolbar',
            dock: 'bottom',
            ui: 'shopware-ui is--button-panel',
            padding: '10 10 10 10',
            items: [
                '->',
                {
                    xtype: 'button',
                    action: 'initializeAll',
                    text: this.getViisonSnippet('list/toolbar/button/initialize_all'),
                    cls: 'secondary',
                    scope: this,
                    handler: function () {
                        this.fireEvent('initializeAll', this);
                    },
                }, {
                    xtype: 'button',
                    action: 'applyChanges',
                    text: this.getViisonSnippet('list/toolbar/button/confirm'),
                    cls: 'primary',
                    scope: this,
                    handler: function () {
                        this.fireEvent('applyChanges', this);
                    },
                },
            ],
        };
    },

    // text filter for toolbar
    applyFilter: function (filterStr) {
        this.store.clearFilter(true); // silent clear
        this.store.getProxy().extraParams.filterStr = filterStr;
        this.store.currentPage = 1;
        this.store.load();
    },

});
