// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPStockOverview.controller.Main', {
    extend: 'Enlight.app.Controller',

    refs: [
        { ref: 'articleGrid', selector: 'viison_pickware_erp_stock_overview-list' },
        { ref: 'main', selector: 'viison_pickware_erp_stock_overview-main' },
        { ref: 'orderPanel', selector: 'viison_pickware_erp_stock_overview-order-list' },
        { ref: 'orderMain', selector: 'viison_pickware_erp_stock_overview-order-main' },
        { ref: 'stockPlanningPanel', selector: 'viison_pickware_erp_stock_overview-planning_interval' },
        { ref: 'binLocationList', selector: 'viison_pickware_erp_stock_overview-article-bin-location-list' },
    ],

    init: function () {
        this.initSalesFilterDateRange();
        this.control({
            'viison_pickware_erp_stock_overview-list': {
                openArticle: this.onOpenArticle,
                editRow: this.onEditRow,
                articleSelected: this.onArticleSelected,
                saveRecord: this.onSaveRecord,
            },
            'viison_pickware_erp_stock_overview-order-list': {
                openOrder: this.onOpenOrder,
            },
            'viison_pickware_erp_stock_overview-planning_interval': {
                planningRange: this.onChangePlanningRange,
            },
        });

        // Reset the orders store and respective view before loading the main stock store
        this.getStore('Stock').on('beforeload', function () {
            this.getStore('Orders').extraParams = {};
            this.getStore('Orders').removeAll();
            if (this.getOrderMain()) {
                this.getOrderMain().hideStats();
            }
        }, this);

        // Open the main window
        this.mainWindow = this.getView('Main').create({
            store: this.getStore('Stock'),
            salesStore: this.getStore('Sales'),
            orderStore: this.getStore('Orders'),
        }).show();

        var salesChart = this.getOrderMain().getSalesChart();
        var salesStore = this.getStore('Sales');
        salesChart.updateInterval(salesStore.planningFrom, salesStore.planningTo);

        // Instantiate bin location list controller
        this.getController('Shopware.apps.ViisonPickwareERPStockOverview.controller.BinLocationList');
    },

    initSalesFilterDateRange: function () {
        // calculate default range
        var from = this.beginOfYesterdayOneMonthAgo();
        var to = this.endOfYesterday();

        // prepare stores
        var salesStore = this.getStore('Sales');
        salesStore.planningFrom = from;
        salesStore.planningTo = to;

        var store = this.getStore('Stock');
        store.getProxy().extraParams.planningFrom = from;
        store.getProxy().extraParams.planningTo = to;
    },

    onChangePlanningRange: function (arg) {
        var from = arg.from;
        var to = arg.to;

        var store = this.getStore('Stock');
        store.getProxy().extraParams.planningFrom = from;
        store.getProxy().extraParams.planningTo = to;

        var salesStore = this.getStore('Sales');
        salesStore.planningFrom = from;
        salesStore.planningTo = to;
        salesStore.updateRange();

        var salesChart = this.getOrderMain().getSalesChart();
        salesChart.updateInterval(from, to);
        salesChart.redraw();

        this.getStore('Stock').load({
            scope: this,
            callback: function () {
                this.getArticleGrid().getView().refresh();
            },
        });

        this.getBinLocationList().showInfoPanel();
    },

    /**
     * Opens a new article sub application window for the article with the given ID.
     *
     * @param Ext.data.Record The id of the article whose details window shall be opened.
     */
    onOpenArticle: function (record) {
        Shopware.app.Application.addSubApplication({
            name: 'Shopware.apps.Article',
            action: 'detail',
            params: {
                articleId: record.get('articleId'),
            },
        });
    },

    onOpenOrder: function (record) {
        Shopware.app.Application.addSubApplication({
            name: 'Shopware.apps.Order',
            params: {
                orderId: record.get('id'),
                // Set focus to position tab
                focus: 'order-position-panel',
            },
        });
    },

    onSaveRecord: function (record) {
        record.set('not_sent', record.get('stock') - record.get('instock'));
        record.save({
            scope: this,
            callback: function () {
                // grid refresh to show if there is a problem after editing instock
                this.getArticleGrid().getView().refresh();
                this.onArticleSelected(record);
            },
        });
    },

    onShowSales: function (record) {
        this.getOrderMain().setDisabled(false);

        var store = this.getStore('Sales');
        store.getProxy().extraParams.articleDetail = record.get('id');
        var salesChart = this.getOrderMain().getSalesChart();
        var salesChartContainer = this.getOrderMain().getSalesChartContainer();
        salesChartContainer.setLoading(true);

        store.load({
            scope: this,
            callback: function () {
                salesChart.redraw();
                salesChartContainer.setLoading(false);
            },
        });
    },

    /**
     * @param {Shopware.apps.ViisonPickwareERPStockOverview.model.Article} record
     */
    onArticleSelected: function (record) {
        this.updateOrdersView(record);

        // Update bin location list
        this.getBinLocationList().setArticle(record);
    },

    updateOrdersView: function (record) {
        this.onShowSales(record);
        var view = this.getOrderPanel();
        var main = this.getOrderMain();
        view.setLoading(true);

        var store = this.getStore('Orders');
        store.currentPage = 1;
        store.getProxy().extraParams.isStockValid = record.isValidStock();
        store.getProxy().extraParams.articleDetail = record.get('id');

        store.load({
            scope: this,
            callback: function () {
                view.setLoading(false);

                // show description when stock is incorrect
                if (!record.isValidStock()) {
                    main.updateStats(record.get('not_sent'), record.get('open'));
                    main.showStats();
                } else {
                    main.hideStats();
                }
            },
        });
    },

    ISO8601Short: 'Y-m-d',
    endOfYesterday: function () {
        var now = new Date();
        var startOfToday = new Date(Ext.Date.format(now, this.ISO8601Short)); // start of day by removing hours, minutes, seconds

        return Ext.Date.add(startOfToday, Ext.Date.SECOND, -1); // end of yesterday => begin of today - 1 second
    },

    beginOfYesterdayOneMonthAgo: function () {
        var now = new Date();
        var startOfToday = new Date(Ext.Date.format(now, this.ISO8601Short)); // start of day by removing hours, minutes, seconds

        return Ext.Date.add(startOfToday, Ext.Date.DAY, -30); // go 30 days back => about a month
    },

    /**
     * Start the row editing for the given record.
     */
    onEditRow: function (record) {
        this.getArticleGrid().rowEditing.startEdit(record, 0);
    },

    /**
     * Reloads the article stocks list with optionally preserving the current selection. Please note,
     * that re-selecting will trigger a reload of the orders panel and the bin location list as well.
     *
     * @param {boolean} preserveSelection
     */
    reloadStockList: function (preserveSelection) {
        var selectedArticle = null;
        if (preserveSelection) {
            var selectedArticles = this.getArticleGrid().getSelectionModel().getSelection();
            if (selectedArticles.length > 0) {
                selectedArticle = selectedArticles[0];
            }
        }

        this.getArticleGrid().getStore().load({
            scope: this,
            callback: function () {
                if (selectedArticle) {
                    var index = this.getArticleGrid().getStore().find('id', selectedArticle.getId());
                    this.getArticleGrid().getView().select(index);
                }
            },
        });
    },

});
