// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPStockOverview.store.Sales', {

    extend: 'Ext.data.Store',
    storeId: 'viison_pickware_erp_stock_overview-salesStore',
    autoLoad: false,
    model: 'Shopware.apps.ViisonPickwareERPStockOverview.model.Sales',
    proxy: {
        type: 'ajax',
        api: {
            read: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPStockOverview/salesByDay'),
        },
        reader: {
            type: 'json',
            root: 'data',
        },
    },
    listeners: {
        load: {
            // add datapadding ( remove gaps within the time period )
            // calculate sliding window average
            fn: function (store, records) {
                store.addDataPadding(store, records);
                store.removeFirstYear();
                store.updateInterval();
            },
        },
    },

    firstYear: [],
    secondYear: [],
    planningFrom: new Date(),
    planningTo: new Date(),
    highestValueInStore: 0,

    /**
     * has to be called after the planning interval has changed
     */
    updateInterval: function () {
        var millisecondsPerDay = 1000 * 60 * 60 * 24;
        var millisecondsOfInterval = (this.planningTo.getTime() - this.planningFrom.getTime());
        var days = Math.round(millisecondsOfInterval / millisecondsPerDay);

        this.addFirstYear();
        this.setSlidingWindowSum('sales', 'salesAverage', Math.max(1, days));
        this.removeFirstYear();
    },

    updateRange: function () {
        this.updateInterval();
        this.data.each(function (r) {
            this.setHightlightMarkerIfRequired(r.get('date'), r);
        }, this);
    },

    /**
     * @constraint: this.data is order by date ASC
     */
    setSlidingWindowSum: function (attribute, newAttribute, windowSize) {
        var windowSum = 0;
        for (var i = 0; i < this.data.length; i += 1) {
            var record = this.data.get(i);
            windowSum += record.get(attribute);

            // handling records leaving the window
            if (i >= windowSize) {
                var outdatedRecord = this.data.get(i - windowSize);
                windowSum -= outdatedRecord.get(attribute);
            }

            // adjusting the divider for when the window starts
            record.set(newAttribute, windowSum);
        }
    },

    /**
     * called upon loading the store, adds missing days to fix holes in the graph
     * @param store
     * @param records
     */
    addDataPadding: function (store, records) {
        // gets the highest within the chart displayed value from the given records
        // only use here: addHightlightMarkerIfRequired
        this.highestValueInStore = Math.max.apply(null, Ext.Array.map(records, function (e) {
            return Math.max(e.get('sales'), e.get('stock'));
        }));

        // create a hashmap of all records
        var measurementsMap = {};
        for (var i = 0; i < records.length; i += 1) {
            var recordDateHash = Ext.Date.format(records[i].get('date'), 'Y-m-d');
            measurementsMap[recordDateHash] = records[i];
            store.remove(records[i]);
        }

        // defining start and end of year
        var today = new Date(Ext.Date.format(new Date(), 'Y-m-d'));
        var lastYear = Ext.Date.add(today, Ext.Date.YEAR, -1);
        var iterator = lastYear;

        // initial default, no line should be drawn if records start with these
        var stock = 0;
        var sales = 0;

        // iterating day by day through the year, creating records with defaults for days without data
        while (iterator < today) {
            var dateHash = Ext.Date.format(iterator, 'Y-m-d');
            var record = measurementsMap[dateHash];
            if (typeof record !== 'undefined') {
                stock = record.get('stock');
                sales = record.get('sales');
            } else {
                // Assumption for defaults:
                // if there is no data of sales, nothing was sold
                // if there is no data of stockchanges stock did not change
                sales = 0;
            }

            // create and add record
            var model = store.model.create({
                date: new Date(dateHash),
                sales: sales,
                stock: stock,
                salesAverage: 0,
                highlightFilterRangeHack: 0,
            });

            this.setHightlightMarkerIfRequired(iterator, model);

            store.add(model);

            // add one day
            iterator = Ext.Date.add(iterator, Ext.Date.DAY, 1);
        }
    },

    // This is a hack to show the selected date range to plan your minimum stock
    // it's ugly as f*** but in compliance with Samuel the fastest/easiest way to proceeed
    setHightlightMarkerIfRequired: function (now, record) {
        if (now <= this.planningTo && now > this.planningFrom) {
            // add marker after the first day of the range
            record.set('highlightFilterRangeHack', this.highestValueInStore * 1.1);
        } else {
            record.set('highlightFilterRangeHack', 0);
        }
    },

    // we are only interested in the second year (latest data)
    // but we need the data of the first year to calculate the beginning of the sliding window
    removeFirstYear: function () {
        var beginOfLastYearInterval = (new Date()) - (1000 * 365 * 24 * 60 * 60);
        this.firstYear = Ext.Array.filter(this.data.items, function (record) {
            return record.get('date') < beginOfLastYearInterval;
        });

        this.secondYear = Ext.Array.filter(this.data.items, function (record) {
            return record.get('date') >= beginOfLastYearInterval;
        });
        this.removeAll();
        this.add(this.secondYear);
    },

    // re-add the first year to to sliding window calculation
    addFirstYear: function () {
        this.removeAll();
        this.add(this.firstYear.concat(this.secondYear));
    },

});
