// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPStockOverview.view.List', {

    extend: 'Ext.grid.Panel',
    alias: 'widget.viison_pickware_erp_stock_overview-list',
    // Use a bug in ExtJS which removes the background of the row editing row to make the values in all columns visible
    cls: 'x-tabpanel-child viison-common--grid has--vertical-lines',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_stock_overview/main',

    style: 'border-left:none;border-top:none;',
    selType: 'rowmodel',
    viewConfig: {
        getRowClass: function (record) {
            return (!record.isValidStock()) ? 'is--warning' : '';
        },
    },
    listeners: {
        select: function (selectionModel, record) {
            // Deselect if record is not sane, wait 300ms for UX
            this.fireEvent('articleSelected', record);
        },
    },

    renderPlanningIndicatorIcon: function (value, meta, record) {
        var marginPercentage = 0.10;
        var targetStock = record.get('targetStock');
        var sales = record.get('salesWithinPlanningInterval');

        var isOverstocked = targetStock > sales * (1 + marginPercentage);
        var isUnderstocked = targetStock * (1 + marginPercentage) < sales;

        if (isUnderstocked) {
            meta.tdAttr = 'data-qtip="' + this.getViisonSnippet('list/column/target_stock_understocked') + '"';
            meta.tdCls = 'has--icon is--trend-down';
        }

        if (isOverstocked) {
            meta.tdAttr = 'data-qtip="' + this.getViisonSnippet('list/column/target_stock_overstocked') + '"';
            meta.tdCls = 'has--icon is--trend-up';
        }

        return value;
    },

    initComponent: function () {
        // When disabling sorting all sortable colums will be saved here in order to restore the original state later
        this.savedSortableColumns = [];

        this.rowEditing = Ext.create('Ext.grid.plugin.RowEditing', {
            clicksToEdit: 2,
            listeners: {
                scope: this,
                edit: function (editor, e) {
                    this.fireEvent('saveRecord', e.record);
                },
            },
        });

        this.plugins = [this.rowEditing];
        this.columns = [{
            header: this.getViisonSnippet('list/column/article_number'),
            sortable: true,
            dataIndex: 'orderNumber',
            flex: 1,
        }, {
            header: this.getViisonSnippet('list/column/article_name'),
            sortable: true,
            dataIndex: 'articleName',
            flex: 2,
        }, {
            header: this.getViisonSnippet('list/column/supplier_name'),
            sortable: true,
            dataIndex: 'supplier',
            flex: 1,
        }, {
            header: this.getViisonSnippet('list/column/available_stock/label'),
            dataIndex: 'instock',
            align: 'right',
            sortable: true,
            flex: 1,
            editor: {
                xtype: 'numberfield',
                allowBlank: false,
            },
        }, {
            header: this.getViisonSnippet('list/column/reserved_stock/label'),
            dataIndex: 'not_sent',
            align: 'right',
            sortable: true,
            flex: 1,
            renderer: function (value, meta, record) {
                if (!record.isValidStock()) {
                    meta.tdCls = 'has--icon is--warning';
                }

                return value;
            },
        }, {
            header: this.getViisonSnippet('list/column/physical_stock/label'),
            dataIndex: 'stock',
            align: 'right',
            sortable: true,
            flex: 1,
        }, {
            header: this.getViisonSnippet('list/column/sales_within_planning_interval'),
            dataIndex: 'salesWithinPlanningInterval',
            align: 'right',
            sortable: true,
            flex: 1,
        }, {
            header: this.getViisonSnippet('list/column/minimum_stock'),
            dataIndex: 'minimumStock',
            align: 'right',
            sortable: true,
            flex: 1,
        }, {
            header: this.getViisonSnippet('list/column/target_stock'),
            dataIndex: 'targetStock',
            align: 'right',
            sortable: true,
            flex: 1,
            renderer: this.renderPlanningIndicatorIcon,
        }, {
            xtype: 'actioncolumn',
            width: 50,
            items: [{
                iconCls: 'sprite-inbox',
                tooltip: this.getViisonSnippet('list/column/action/open_article/tooltip'),
                scope: this,
                handler: function (view, rowIndex) {
                    this.fireEvent('openArticle', view.getStore().getAt(rowIndex));
                },
            }, {
                iconCls: 'sprite-pencil',
                tooltip: this.getViisonSnippet('list/column/action/edit/tooltip'),
                scope: this,
                handler: function (view, rowIndex) {
                    this.fireEvent('editRow', view.getStore().getAt(rowIndex));
                },
            }],
        }];

        this.dockedItems = [
            this.getToolbar(),
            this.getPagingbar(),
        ];

        this.callParent(arguments);
    },

    /**
     * Creates and returns a new paging toolbar and adds a page size selection to it.
     *
     * @return Shopware.apps.ViisonCommonPaginationToolbar
     */
    getPagingbar: function () {
        return {
            xtype: 'viison_common_pagination_toolbar-toolbar',
            store: this.store,
            pageSizeLabel: this.getViisonSnippet('list/paging/page_size'),
        };
    },

    /**
     * Creates and returns a new paging toolbar and adds a page size selection to it.
     *
     * @return Ext.toolbar.Toolbar The created page toolbar.
     */
    getToolbar: function () {
        return {
            xtype: 'toolbar',
            dock: 'top',
            ui: 'shopware-ui',
            padding: '5 5 5 5',
            border: false,
            items: [
                {
                    xtype: 'checkboxfield',
                    margin: '0 0 0 8',
                    cls: 'viison-common--checkbox-with-boxlabel',
                    boxLabel: this.getViisonSnippet('label/filter/show_problems'),
                    listeners: {
                        scope: this,
                        change: function (field) {
                            this.filterProblems(field.value);
                        },
                    },
                },
                '->',
                {
                    xtype: 'textfield',
                    name: 'searchfield',
                    cls: 'searchfield',
                    width: 200,
                    enableKeyEvents: true,
                    emptyText: this.getViisonSnippet('search/initialText'),
                    listeners: {
                        buffer: 500,
                        scope: this,
                        change: function (field) {
                            this.applyFilter(field.value);
                        },
                    },
                }],
        };
    },

    disableSorting: function (disabled) {
        // save column sortable state
        if (disabled && this.savedSortableColumns.length === 0) {
            this.savedSortableColumns = this.columns.filter(function (col) {
                return col.sortable;
            });
        }

        // removes sort status (arrow indicating sorting direction)
        if (disabled) {
            this.store.sorters.clear();
        }

        Ext.Array.forEach(this.savedSortableColumns, function (col) {
            col.sortable = !disabled;
        });
    },

    // filters problems with stock
    filterProblems: function (applyFilter) {
        this.disableSorting(applyFilter);
        this.store.clearFilter(true); // silent clear
        this.store.getProxy().extraParams.filterStockProblems = applyFilter;
        this.store.currentPage = 1;
        this.store.load();
    },

    // text filter for toolbar
    applyFilter: function (filterStr) {
        this.store.clearFilter(true); // silent clear
        this.store.getProxy().extraParams.filterStr = filterStr;
        this.store.currentPage = 1;
        this.store.load();
    },

});
