// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPStockOverview.view.SalesChart', {

    extend: 'Ext.chart.Chart',
    alias: 'widget.viison_pickware_erp_stock_overview-sales_chart',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_stock_overview/main',

    theme: 'ViisonPickwareERP',
    cls: 'viison-common--chart',
    style: 'background-color:white;',
    shadow: false,

    legend: {
        isViisonPositionTopRight: true,
        labelFont: '9px Arial, Helvetica, sans-serif',
        labelColor: '#475c6a',
        position: 'right',
        boxStrokeWidth: 0,
        padding: 10,
        itemSpacing: 15,
    },

    beginOfChartPeriod: (new Date()) - (365 * 24 * 60 * 60),

    initComponent: function () {
        this.axes = this.createAxes();
        this.series = this.createSeries();

        this.callParent(arguments);
    },

    // we load 2 years of data to calcualte the slifing window,
    // but we only want to show one year
    updateInterval: function (from, to) {
        var days = Math.round((to - from) / 1000 / 60 / 60 / 24);
        var seriesItem = Ext.Array.filter(this.series.items, function (item) {
            return item.yField === 'salesAverage';
        }, this)[0];

        var salesSumText = this.getViisonSnippet('chart/series/sales_sum');
        var salesSumDaysText = this.getViisonSnippet('chart/series/sales_sum_days');
        seriesItem.title = salesSumText + ' (' + days + ' ' + salesSumDaysText + ')';
    },

    /**
     * @return {Object[]}
     */
    createAxes: function () {
        return [
            {
                type: 'Numeric',
                position: 'left',
                fields: [
                    'sales',
                    'stock',
                    'highlightFilterRangeHack',
                    'salesAverage',
                ],
            }, {
                type: 'Time',
                dateFormat: 'd/M',
                position: 'bottom',
                fields: ['date'],
                step: [Ext.Date.MONTH, 1],
            },
        ];
    },

    /**
     * @return {Object[]}
     */
    createSeries: function () {
        var physicalStockTooltipSnippet = this.getViisonSnippet('chart/tooltip/physical_stock');
        var salesTooltipSnippet = this.getViisonSnippet('chart/tooltip/sales');

        return [
            {
                title: this.getViisonSnippet('chart/series/sales_sum'),
                type: 'line',
                xField: 'date',
                yField: 'salesAverage',
                viisonLegendMarkerType: 'rect',
                smooth: true,
                showMarkers: false,
                tips: {
                    trackMouse: true,
                    width: 200,
                    baseCls: 'viison-common--chart--tooltip',
                    renderer: function (storeItem) {
                        /* eslint-disable indent */
                        var tooltipTemplate = new Ext.XTemplate(
                            '<tpl for=".">',
                                '<div class="is--header">{date}</div>',
                                '<div>',
                                    '<span class="is--label">',
                                        physicalStockTooltipSnippet + ':',
                                    '</span>',
                                    '<span class="is--value">{stock}</span>',
                                '</div>',
                                '<div>',
                                    '<span class="is--label">',
                                        salesTooltipSnippet + ':',
                                    '</span>',
                                    '<span class="is--value">{sales}</span>',
                                '</div>',
                            '</tpl>'
                        );
                        /* eslint-enable indent */

                        this.setTitle(tooltipTemplate.apply({
                            date: Ext.Date.format(storeItem.get('date'), 'l d.m.Y'),
                            stock: storeItem.get('stock'),
                            sales: storeItem.get('sales'),
                        }));
                    },
                },
            }, {
                title: this.getViisonSnippet('chart/series/sales_by_day'),
                type: 'line',
                xField: 'date',
                yField: 'sales',
                viisonLegendMarkerType: 'rect',
                smooth: true,
                showMarkers: false,
            }, {
                title: this.getViisonSnippet('chart/series/physical_stock'),
                type: 'area',
                xField: 'date',
                yField: ['stock'],
                viisonLegendMarkerType: 'rect',
                renderer: function (sprite, record, attr) {
                    return Ext.apply(attr, {
                        opacity: 0.7,
                        fill: '#90D5F2',
                        'stroke-width': 0,
                        lineWidth: 0,
                    });
                },
                getLegendColor: function () {
                    return '#90D5F2';
                },
            }, {
                type: 'area',
                xField: 'date',
                yField: 'highlightFilterRangeHack',
                viisonLegendMarkerType: 'rect',
                title: this.getViisonSnippet('chart/series/planning_interval'),
                renderer: function (sprite, record, attr) {
                    return Ext.apply(attr, {
                        opacity: 0.2,
                        fill: '#798D8D',
                        'stroke-width': 0,
                    });
                },
                getLegendColor: function () {
                    return '#EBEFED';
                },
            },
        ];
    },

});
