// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPStockTakeExport.controller.Main', {

    extend: 'Ext.app.Controller',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_stock_take_export/main',

    /**
     * @Override
     */
    init: function () {
        this.control({
            'viison_pickware_erp_stock_take_export-base-filter': {
                filterSinglePanel: this.onFilterSinglePanel,
                warehouseChanged: this.onWarehouseChanged,
                dateFromChanged: this.onDateFromChanged,
                dateToChanged: this.onDateToChanged,
                userChanged: this.onUserChanged,
                searchFieldChanged: this.onSearchFieldChanged,
            },
            'viison_pickware_erp_stock_take_export-complete-main': {
                export: this.onExportStockTakes,
            },
            'viison_pickware_erp_stock_take_export-complete-list': {
                openArticle: this.onOpenArticle,
                updateButtons: this.onUpdateCompletePanelButtons,
            },
            'viison_pickware_erp_stock_take_export-pending-main': {
                export: this.onExportPendingStockTakes,
                saveStockTakes: this.onRecordStockTakes,
                nullStockTakes: this.onRecordStockTakesUsingZeroStock,
            },
            'viison_pickware_erp_stock_take_export-pending-list': {
                openArticle: this.onOpenArticle,
                pageChange: this.onPageChange,
                sortChange: this.onSortChange,
                updateButtons: this.onUpdatePendingPanelButtons,
            },
            'viison_pickware_erp_stock_take_export-main': {
                tabChange: this.onTabChange,
            },
        });

        // Create and show main window
        this.mainWindow = this.getView('Main').create().show();

        this.callParent(arguments);
    },

    /**
     * @param int articleId
     */
    onOpenArticle: function (articleId) {
        Shopware.app.Application.addSubApplication({
            name: 'Shopware.apps.Article',
            action: 'detail',
            params: {
                articleId: articleId,
            },
        });
    },

    /**
     * @param Shopware.apps.ViisonPickwareERPStockTakeExport.view.Main window
     */
    filterBothGridsWithCurrentFilters: function (window) {
        var filterPanels = window.query('viison_pickware_erp_stock_take_export-base-filter');
        filterPanels.forEach(function (panel) {
            this.onFilterSinglePanel(panel);
        }, this);
    },

    /**
     * Enables or disabled the respective action buttons of the pending-panel. If the grid is empty, no buttons are
     * available. If grid entries are counted (newStock is set), saving these entries is enabled and nulling entries is
     * disabled and vice versa.
     *
     * @param Shopware.apps.ViisonPickwareERPStockTakeExport.view.pending.Main pendingPanel
     */
    onUpdatePendingPanelButtons: function (pendingPanel) {
        if (pendingPanel.gridPanel.store.data.items.length === 0) {
            pendingPanel.saveStockTakesButton.setDisabled(true);
            pendingPanel.nullStockTakesButton.setDisabled(true);
            pendingPanel.exportButton.setDisabled(true);

            return;
        }

        // Grid is not empty, enable one of the buttons
        var completeEntries = this.getChangedEntitiesFromPendingStockTakePanel(pendingPanel);
        var stockTakesAreCounted = completeEntries.length > 0;
        pendingPanel.saveStockTakesButton.setDisabled(!stockTakesAreCounted);
        pendingPanel.nullStockTakesButton.setDisabled(stockTakesAreCounted);
        pendingPanel.exportButton.setDisabled(false);
    },

    /**
     * Enables or disables the export button when the store is loaded.
     *
     * @param Shopware.apps.ViisonPickwareERPStockTakeExport.view.complete.Main pendingPanel
     */
    onUpdateCompletePanelButtons: function (completePanel) {
        completePanel.exportButton.setDisabled(completePanel.gridPanel.store.data.items.length === 0);
    },

    /**
     * Validates the filter form and, if valid, applies the filter settings to the store of the grid panel and loads the
     * store.
     *
     * @param Shopware.apps.ViisonPickwareERPStockTakeExport.view.base.Filter baseFilterPanel
     */
    onFilterSinglePanel: function (baseFilterPanel) {
        if (!baseFilterPanel.getForm().isValid()) {
            return;
        }

        // Assigning the filter values to the store will keep them assigned when reloading the store via pagination
        // toolbar.
        var gridPanel = baseFilterPanel.up('panel').gridPanel;
        gridPanel.getStore().getProxy().extraParams = this.getFilters(baseFilterPanel);
        gridPanel.getStore().load();
    },

    /**
     * Opens a new tab showing the export document created by applying the current filter settings to the given
     * 'complete' panel.
     *
     * @param Shopware.apps.ViisonPickwareERPStockTakeExport.view.complete.Main completePanel
     */
    onExportStockTakes: function (completePanel) {
        var url = ViisonCommonApp.assembleBackendUrl(
            'ViisonPickwareERPStockTakeExport/exportStockTakes',
            this.getFilters(completePanel)
        );
        window.open(url, '_blank');
    },

    /**
     * Opens a new tab showing the export document created by applying the current filter settings to the given
     * 'pending' panel.
     *
     * @param Shopware.apps.ViisonPickwareERPStockTakeExport.view.pending.Main pendingPanel
     */
    onExportPendingStockTakes: function (pendingPanel) {
        // Construct URL and open it in a new window/tab
        var url = ViisonCommonApp.assembleBackendUrl(
            'ViisonPickwareERPStockTakeExport/exportPendingStockTakes',
            this.getFilters(pendingPanel)
        );
        window.open(url, '_blank');
    },

    /**
     * Applies the counted article changes as stocktake entries and reloads the pending store (since articles are now
     * removed from this list).
     *
     * @param Shopware.apps.ViisonPickwareERPStockTakeExport.view.pending.Main pendingPanel
     * @param bool skipConfirmation
     * @param callbackFunction
     */
    onRecordStockTakes: function (pendingPanel, skipConfirmation, callbackFunction) {
        pendingPanel.setLoading(true);

        // Fetch the number of articles that are about to be saved and let the customer confirm this action. This array
        // must not be empty, since this action (button) is only enabled, if stock takes are actually counted.
        var completeEntries = this.getChangedEntitiesFromPendingStockTakePanel(pendingPanel);

        if (skipConfirmation) {
            this.callRecordStockTakesRequest(pendingPanel, completeEntries, callbackFunction);

            return;
        }

        var message = this.getViisonSnippet('button/saveStockTakes/confirmation');
        message = message.replace('%d', completeEntries.length);
        Ext.Msg.confirm(
            this.getViisonSnippet('button/saveStockTakes/text'), // Use the button text as title
            message,
            function (buttonId) {
                if (buttonId === 'yes') {
                    this.callRecordStockTakesRequest(pendingPanel, completeEntries);
                } else {
                    // Confirmation was denied. Do nothing
                    pendingPanel.setLoading(false);
                }
            },
            this
        );
    },

    /**
     * Calls the controller action to save the record (given by the completeEntries array).
     *
     * @param Shopware.apps.ViisonPickwareERPStockTakeExport.view.pending.Main pendingPanel
     * @param [] completeEntries
     * @param callbackFunction
     */
    callRecordStockTakesRequest: function (pendingPanel, completeEntries, callbackFunction) {
        var requestData = completeEntries.map(function (stockTakeEntry) {
            return {
                articleDetailBinLocationMappingId: stockTakeEntry.data.articleDetailBinLocationMappingId,
                newStock: stockTakeEntry.data.newStock,
                comment: stockTakeEntry.data.comment,
            };
        });

        Ext.Ajax.request({
            url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPStockTakeExport/recordStockTakes'),
            method: 'POST',
            jsonData: {
                countedEntities: requestData,
            },
            scope: this,
            success: function (response) {
                var responseData = Ext.JSON.decode(response.responseText, true);
                if (responseData && responseData.success) {
                    Shopware.Notification.createGrowlMessage(
                        this.getViisonSnippet('alert/title/success'),
                        this.getViisonSnippet('button/saveStockTakes/result/success'),
                        'ViisonPickwareERPStockTakeExport'
                    );

                    // Reload both grid panels since the pool of entries now changed
                    this.filterBothGridsWithCurrentFilters(pendingPanel.up('window'));
                } else {
                    Shopware.Notification.createGrowlMessage(
                        this.getViisonSnippet('alert/title/error'),
                        this.getViisonSnippet('button/saveStockTakes/result/error'),
                        'ViisonPickwareERPStockTakeExport'
                    );
                }
                pendingPanel.setLoading(false);

                // Execute custom callback if it was set
                if (callbackFunction) {
                    callbackFunction();
                }
            },
        });
    },

    /**
     * Writes stocktakes with amount 0 for all pending articles and reloads the complete store (since articles are
     * now removed from this list).
     *
     * @param Shopware.apps.ViisonPickwareERPStockTakeExport.view.pending.Main pendingPanel
     */
    onRecordStockTakesUsingZeroStock: function (pendingPanel) {
        pendingPanel.setLoading(true);

        // Use the totalCount of the pagination toolbar of the grid store to determine how many stock takes are nulled.
        var uncountedStockTakesCount = pendingPanel.gridPanel.dockedItems.items[1].store.totalCount;
        if (uncountedStockTakesCount === 0) {
            Shopware.Notification.createGrowlMessage(
                this.getViisonSnippet('alert/title/error'),
                this.getViisonSnippet('button/nullStockTakes/noPositions'),
                'ViisonPickwareERPStockTakeExport'
            );
            pendingPanel.setLoading(false);

            return;
        }

        var message = this.getViisonSnippet('button/nullStockTakes/confirmation');
        message = message.replace('%d', uncountedStockTakesCount);
        Ext.Msg.confirm(
            this.getViisonSnippet('button/nullStockTakes/text'), // Use the button text as title
            message,
            function (buttonId) {
                if (buttonId === 'yes') {
                    var progressWindow = Ext.create('Shopware.apps.ViisonPickwareERPStockTakeExport.view.pending.ProgressWindow');
                    progressWindow.progressBar.updateProgress(0, 0 + ' / ' + uncountedStockTakesCount, true);
                    progressWindow.show();

                    this.callRecordStockTakesUsingZeroStockRequestBatchwise(
                        pendingPanel,
                        progressWindow,
                        uncountedStockTakesCount,
                        uncountedStockTakesCount
                    );
                } else {
                    // Confirmation was denied. Do nothing
                    pendingPanel.setLoading(false);
                }
            },
            this
        );
    },

    /**
     * A recursive function that calls for the controller to null all pending article bin location mappings. Calls the
     * controller only with a batch size (the controller decided which pending article bin location mappings are nulled
     * next).
     *
     * Recalls itself until the controller confirms that no pending article bin locations mappings are left, or a
     * request fails. Calls a finishing method with a respective growl message in both cases.
     *
     * @param Shopware.apps.ViisonPickwareERPStockTakeExport.view.pending.Main pendingPanel
     * @param Shopware.apps.ViisonPickwareERPStockTakeExport.view.pending.ProgressWindow progressWindow
     * @param int leftToProcess
     * @param int maxCount
     */
    callRecordStockTakesUsingZeroStockRequestBatchwise: function (pendingPanel, progressWindow, leftToProcess, maxCount) {
        progressWindow.progressBar.updateProgress(
            (maxCount - leftToProcess) / maxCount,
            (maxCount - leftToProcess) + ' / ' + maxCount,
            true
        );

        // Reuse the store filter parameters and add batch size
        var filterParams = pendingPanel.gridPanel.getStore().getProxy().extraParams;
        filterParams.batchSize = 250;
        Ext.Ajax.request({
            url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPStockTakeExport/recordStockTakesUsingZeroStock'),
            method: 'POST',
            jsonData: filterParams,
            scope: this,
            success: function (actionResponse) {
                var actionResponseData = Ext.JSON.decode(actionResponse.responseText, true);
                if (actionResponseData && actionResponseData.success) {
                    if (actionResponseData.pending === 0) {
                        this.onRecordStockTakesUsingZeroStockRequestBatchwiseSuccess(
                            pendingPanel,
                            progressWindow,
                            maxCount
                        );
                    } else {
                        this.callRecordStockTakesUsingZeroStockRequestBatchwise(
                            pendingPanel,
                            progressWindow,
                            actionResponseData.pending,
                            maxCount
                        );
                    }
                } else {
                    this.onRecordStockTakesUsingZeroStockRequestBatchwiseFailed(pendingPanel, progressWindow, actionResponseData);
                }
            },
        });
    },

    /**
     * Is called after the batchwise stock taking with using zero stock succeeded. Shows a respective growl message.
     *
     * @param Shopware.apps.ViisonPickwareERPStockTakeExport.view.pending.Main pendingPanel
     * @param Shopware.apps.ViisonPickwareERPStockTakeExport.view.pending.ProgressWindow progressWindow
     * @param int maxCount
     */
    onRecordStockTakesUsingZeroStockRequestBatchwiseSuccess: function (pendingPanel, progressWindow, maxCount) {
        var message = this.getViisonSnippet('button/nullStockTakes/result/success');
        message = message.replace('%d', maxCount);
        Shopware.Notification.createGrowlMessage(
            this.getViisonSnippet('alert/title/success'),
            message,
            'ViisonPickwareERPStockTakeExport'
        );
        progressWindow.close();
        this.reloadPendingGridStoreWithKeepCountValues(pendingPanel);
        pendingPanel.setLoading(false);
    },

    /**
     * Is called after the batchwise stock taking with using zero stock failed. Shows a respective growl message.
     *
     * @param Shopware.apps.ViisonPickwareERPStockTakeExport.view.pending.Main pendingPanel
     * @param Shopware.apps.ViisonPickwareERPStockTakeExport.view.pending.ProgressWindow progressWindow
     * @param object
     */
    onRecordStockTakesUsingZeroStockRequestBatchwiseFailed: function (pendingPanel, progressWindow, actionResponseData) {
        var message = this.getViisonSnippet('button/nullStockTakes/result/error');
        if (actionResponseData && actionResponseData.error) {
            message += ' Error: ' + actionResponseData.error;
            message += ' Stacktrace: ' + actionResponseData.error_trace;
        }
        Shopware.Notification.createGrowlMessage(
            this.getViisonSnippet('alert/title/error'),
            message,
            'ViisonPickwareERPStockTakeExport'
        );
        progressWindow.close();
        pendingPanel.setLoading(false);
    },

    /**
     * Fetch all changed (counted) entities, reload the store of the grid then assign the changes (counts) to the
     * reloaded store. This way the manual changed (counts) are kept when reloading
     *
     * @param Shopware.apps.ViisonPickwareERPStockTakeExport.view.pending.Main pendingPanel
     */
    reloadPendingGridStoreWithKeepCountValues: function (pendingPanel) {
        var completeEntries = this.getChangedEntitiesFromPendingStockTakePanel(pendingPanel);

        pendingPanel.setLoading(true);
        pendingPanel.gridPanel.store.load({
            scope: this,
            callback: function (records) {
                // Reassign the newStock (count) value
                records.forEach(function (record) {
                    for (var i = 0; i < completeEntries.length; i += 1) {
                        if (completeEntries[i].data.articleDetailBinLocationMappingId === record.get('articleDetailBinLocationMappingId')) {
                            record.set('newStock', completeEntries[i].data.newStock);
                            break;
                        }
                    }
                });

                pendingPanel.setLoading(false);
            },
        });
    },

    /**
     * @param Shopware.apps.ViisonPickwareERPStockTakeExport.view.base.Filter filterPanel
     * @return object
     */
    getFilters: function (filterPanel) {
        return {
            warehouseId: this.getFilterWarehouse(filterPanel),
            userId: this.getFilterUser(filterPanel),
            fromDate: this.getFilterFromDate(filterPanel),
            toDate: this.getFilterToDate(filterPanel),
            query: this.getFilterQuery(filterPanel),
        };
    },

    /**
     * @param Shopware.apps.ViisonPickwareERPStockTakeExport.view.base.Filter filterPanel
     * @return int
     */
    getFilterWarehouse: function (filterPanel) {
        var warehouseComboBox = filterPanel.down('viison_pickware_erp_warehouse_management-warehouse_combo_box');

        return warehouseComboBox.getSelectedRecord().get('id');
    },

    /**
     * @param Shopware.apps.ViisonPickwareERPStockTakeExport.view.base.Filter filterPanel
     * @return DateTime
     */
    getFilterToDate: function (filterPanel) {
        var values = filterPanel.getForm().getFieldValues();

        return values.to ? values.to : null;
    },

    /**
     * @param Shopware.apps.ViisonPickwareERPStockTakeExport.view.base.Filter filterPanel
     * @return DateTime
     */
    getFilterFromDate: function (filterPanel) {
        var values = filterPanel.getForm().getFieldValues();

        return values.from ? values.from : null;
    },

    /**
     * @param Shopware.apps.ViisonPickwareERPStockTakeExport.view.base.Filter filterPanel
     * @return int
     */
    getFilterUser: function (filterPanel) {
        var values = filterPanel.getForm().getFieldValues();

        return values.user ? values.user : null;
    },

    /**
     * @param {Shopware.apps.ViisonPickwareERPStockTakeExport.view.base.Filter} filterPanel
     * @returns {string}
     */
    getFilterQuery: function (filterPanel) {
        var values = filterPanel.getForm().getFieldValues();

        return values.searchfield ? values.searchfield : null;
    },

    /**
     * Returns an array of all changed entities of the given pendingPanel.
     *
     * @param Shopware.apps.ViisonPickwareERPStockTakeExport.view.pending.Main pendingPanel
     * @return []
     */
    getChangedEntitiesFromPendingStockTakePanel: function (pendingPanel) {
        return pendingPanel.gridPanel.store.data.items.filter(function (stockTakeEntry) {
            return stockTakeEntry.data.newStock !== null;
        });
    },

    /**
     * This method is executed when a tab change to the complete panel is started. Ask for confirmation to save (or
     * dismiss) any changes to the current pending panel list and continue. Continue without confirmation if there are
     * no changes.
     *
     * @param Shopware.apps.ViisonPickwareERPStockTakeExport.view.Main mainPanel
     */
    onTabChange: function (mainPanel) {
        var pendingPanel = mainPanel.down('viison_pickware_erp_stock_take_export-pending-main');
        var tabPanel = mainPanel.down('tabpanel');
        var numberOfChangedEntities = this.getChangedEntitiesFromPendingStockTakePanel(pendingPanel).length;

        if (numberOfChangedEntities === 0) {
            // Continue tab change to the first tab
            tabPanel.suspendEvents();
            tabPanel.setActiveTab(0);
            tabPanel.resumeEvents();

            return;
        }

        var message = this.getViisonSnippet('alert/tabChange/message');
        message = message.replace('%d', numberOfChangedEntities);
        mainPanel.setLoading(true);

        Ext.create('Shopware.apps.ViisonPickwareERPStockTakeExport.view.base.LargeMessageBox').show({
            title: this.getViisonSnippet('alert/tabChange/title'),
            icon: Ext.Msg.QUESTION,
            msg: message,
            buttons: Ext.Msg.YESNOCANCEL,
            buttonText: {
                yes: this.getViisonSnippet('alert/continue/yes'),
                no: this.getViisonSnippet('alert/continue/no'),
                cancel: this.getViisonSnippet('alert/cancel'),
            },
            scope: this,
            callback: function (buttonId) {
                mainPanel.setLoading(false);
                if (buttonId === 'yes') {
                    // Record stock takes (which also reloads both stores
                    this.onRecordStockTakes(pendingPanel, true);
                    tabPanel.suspendEvents();
                    tabPanel.setActiveTab(0);
                    tabPanel.resumeEvents();
                } else if (buttonId === 'no') {
                    // Dismiss the current changes by reloading the store
                    this.onFilterSinglePanel(pendingPanel.filterPanel);
                    tabPanel.suspendEvents();
                    tabPanel.setActiveTab(0);
                    tabPanel.resumeEvents();
                } // Otherwise (which includes button 'cancel') do nothing
            },
        });
    },

    /**
     * Prompt a confirmation message before the page of the pending stock list is changed. If there are unsaved changes,
     * ask to dismiss or save them before changing the page (reloading the grid). Change the page directly, if there are
     * no unsaved changes.
     *
     * @param Shopware.apps.ViisonPickwareERPStockTakeExport.view.pending.List pendingList
     * @param int pageNumber
     */
    onPageChange: function (pendingList, pageNumber) {
        var pendingPanel = pendingList.up('panel');
        var mainPanel = pendingPanel.up('panel');
        var numberOfChangedEntities = this.getChangedEntitiesFromPendingStockTakePanel(pendingPanel).length;

        if (numberOfChangedEntities === 0) {
            // Continue tab change to the first tab
            pendingList.toolbar.suspendEvents();
            pendingList.store.loadPage(pageNumber);
            pendingList.toolbar.resumeEvents();

            return;
        }

        var message = this.getViisonSnippet('alert/pageChange/message');
        message = message.replace('%d', numberOfChangedEntities);
        mainPanel.setLoading(true);

        Ext.create('Shopware.apps.ViisonPickwareERPStockTakeExport.view.base.LargeMessageBox').show({
            title: this.getViisonSnippet('alert/pageChange/title'),
            icon: Ext.Msg.QUESTION,
            msg: message,
            buttons: Ext.Msg.YESNOCANCEL,
            buttonText: {
                yes: this.getViisonSnippet('alert/continue/yes'),
                no: this.getViisonSnippet('alert/continue/no'),
                cancel: this.getViisonSnippet('alert/cancel'),
            },
            scope: this,
            callback: function (buttonId) {
                mainPanel.setLoading(false);
                if (buttonId === 'yes') {
                    var callbackFunction = function () {
                        pendingList.toolbar.suspendEvents();
                        pendingList.store.loadPage(pageNumber);
                        pendingList.toolbar.resumeEvents();
                    };
                    // Record stock takes then go to the next page
                    this.onRecordStockTakes(pendingPanel, true, callbackFunction);
                } else if (buttonId === 'no') {
                    // Dismiss the current changes by simply loading the next page
                    pendingList.toolbar.suspendEvents();
                    pendingList.store.loadPage(pageNumber);
                    pendingList.toolbar.resumeEvents();
                } // Otherwise (which includes button 'cancel') do nothing
            },
        });
    },

    /**
     * Is called when the sorting changes (which causes a reload of the store). Check if there are unsaved changes and
     * prompt a confirmation.
     *
     * @param Shopware.apps.ViisonPickwareERPStockTakeExport.view.pending.List pendingList
     * @param Ext.data.Store store
     * @param array newSorters
     */
    onSortChange: function (pendingList, store, newSorters) {
        pendingList.lastSorters = newSorters;
        var pendingPanel = pendingList.up('panel');
        var mainPanel = pendingPanel.up('panel');
        var numberOfChangedEntities = this.getChangedEntitiesFromPendingStockTakePanel(pendingPanel).length;
        if (numberOfChangedEntities === 0) {
            // Continue with the sorting
            store.load();

            return;
        }

        var message = this.getViisonSnippet('alert/sortChange/message');
        message = message.replace('%d', numberOfChangedEntities);
        mainPanel.setLoading(true);

        Ext.create('Shopware.apps.ViisonPickwareERPStockTakeExport.view.base.LargeMessageBox').show({
            title: this.getViisonSnippet('alert/sortChange/title'),
            icon: Ext.Msg.QUESTION,
            msg: message,
            buttons: Ext.Msg.YESNOCANCEL,
            buttonText: {
                yes: this.getViisonSnippet('alert/continue/yes'),
                no: this.getViisonSnippet('alert/continue/no'),
                cancel: this.getViisonSnippet('alert/cancel'),
            },
            scope: this,
            callback: function (buttonId) {
                mainPanel.setLoading(false);
                if (buttonId === 'yes') {
                    // Record stock takes then continue with the sorting. Since the "recordStockTakes" function also
                    // reloads the store there is nothing more to do than update the sorter and call "recordStockTakes"
                    this.onRecordStockTakes(pendingPanel, true);
                } else if (buttonId === 'no') {
                    // Dismiss the current changes by simply loading the new sorted list
                    store.load();
                } // Otherwise (which includes button 'cancel') do nothing
            },
        });
    },

    /**
     * Prompt a confirmation pop up before the the filter is changed and store reloaded to save (or dimiss) any changes
     * to the current store.
     *
     * @param window
     * @param applyFilterChanges
     */
    promptConfirmationToReloadStores: function (window, applyFilterChanges) {
        var pendingPanel = window.down('viison_pickware_erp_stock_take_export-pending-main');
        var numberOfChangedEntities = this.getChangedEntitiesFromPendingStockTakePanel(pendingPanel).length;
        if (numberOfChangedEntities === 0) {
            applyFilterChanges();

            // Reload both stores after the filters are updated
            this.filterBothGridsWithCurrentFilters(window);

            return;
        }

        var message = this.getViisonSnippet('alert/reload/message');
        message = message.replace('%d', numberOfChangedEntities);
        window.setLoading(true);

        Ext.create('Shopware.apps.ViisonPickwareERPStockTakeExport.view.base.LargeMessageBox').show({
            title: this.getViisonSnippet('alert/reload/title'),
            icon: Ext.Msg.QUESTION,
            msg: message,
            buttons: Ext.Msg.YESNOCANCEL,
            cls: 'is--large-message',
            buttonText: {
                yes: this.getViisonSnippet('alert/reload/yes'),
                no: this.getViisonSnippet('alert/reload/no'),
                cancel: this.getViisonSnippet('alert/cancel'),
            },
            scope: this,
            callback: function (buttonId) {
                window.setLoading(false);
                if (buttonId === 'yes') {
                    applyFilterChanges();

                    // Record stock takes (which also reloads both stores
                    this.onRecordStockTakes(pendingPanel, true);
                } else if (buttonId === 'no') {
                    applyFilterChanges();

                    // Just reload both grids with the current filter, which dismisses the current changes of the store
                    this.filterBothGridsWithCurrentFilters(window);
                } // Otherwise (which includes button 'cancel') do nothing
            },
        });
    },

    /**
     * A change of a filter that is used in both filter panels will be ignored at first and a change event is thrown.
     * This controller checks if there are any pending changed in the pending panel and prompts a confirmation message
     * first before the filter is updated in both panels and the grids reloaded.
     *
     * If a change of a filter is only used in the complete panels, the filter can be applied and the store reloaded
     * without hesitation.
     */

    /**
     * This filter is used in both panels. Prompt confirmation.
     *
     * @param Shopware.apps.ViisonPickwareERPStockTakeExport.view.base.Filter baseFilterPanel
     * @param Warehouse newWarehouse
     */
    onWarehouseChanged: function (baseFilterPanel, newWarehouse) {
        var window = baseFilterPanel.up('window');
        var applyFilterChanges = function () {
            var panels = window.query('viison_pickware_erp_stock_take_export-base-stock_take_panel');
            panels.forEach(function (panel) {
                panel.filterPanel.updateWarehouse(newWarehouse);
            });
        };

        this.promptConfirmationToReloadStores(window, applyFilterChanges);
    },

    /**
     * This filter is used in both panels. Prompt confirmation.
     *
     * @param Shopware.apps.ViisonPickwareERPStockTakeExport.view.base.Filter baseFilterPanel
     * @param Date newDateFrom
     */
    onDateFromChanged: function (baseFilterPanel, newDateFrom) {
        var window = baseFilterPanel.up('window');
        var applyFilterChanges = function () {
            var panels = window.query('viison_pickware_erp_stock_take_export-base-stock_take_panel');
            panels.forEach(function (panel) {
                panel.filterPanel.updateDateFrom(newDateFrom);
            });
        };

        this.promptConfirmationToReloadStores(window, applyFilterChanges);
    },

    /**
     * This filter is used in the completed panel. Reload the store.
     *
     * @param Shopware.apps.ViisonPickwareERPStockTakeExport.view.base.Filter baseFilterPanel
     */
    onDateToChanged: function (baseFilterPanel) {
        this.onFilterSinglePanel(baseFilterPanel);
    },

    /**
     * This filter is used in the completed panel. Reload the store.
     *
     * @param Shopware.apps.ViisonPickwareERPStockTakeExport.view.base.Filter baseFilterPanel
     */
    onUserChanged: function (baseFilterPanel) {
        this.onFilterSinglePanel(baseFilterPanel);
    },

    /**
     * This filter is used in both panels. Prompt confirmation.
     *
     * @param {Shopware.apps.ViisonPickwareERPStockTakeExport.view.base.Filter} baseFilterPanel
     * @param {string} query
     */
    onSearchFieldChanged: function (baseFilterPanel, query) {
        var window = baseFilterPanel.up('window');
        var applyFilterChanges = function () {
            var panels = window.query('viison_pickware_erp_stock_take_export-base-stock_take_panel');
            panels.forEach(function (panel) {
                panel.filterPanel.updateSearchField(query);
            });
        };

        this.promptConfirmationToReloadStores(window, applyFilterChanges);
    },
});
