// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPStockTakeExport.view.base.Filter', {

    extend: 'Ext.form.Panel',
    alias: 'widget.viison_pickware_erp_stock_take_export-base-filter',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_stock_take_export/main',

    style: {
        backgroundColor: '#f0f2f4',
    },
    height: 'auto',
    border: false,

    /**
     * @Override
     */
    initComponent: function () {
        this.userStore = Ext.create('Shopware.apps.ViisonPickwareERPStockTakeExport.store.User');
        this.dockedItems = [
            this.createFilterForm(),
        ];
        this.userStore.load({
            scope: this,
            callback: function (records) {
                this.userComboBox.setValue(records[0].get('id'));
            },
        });

        // Filter (load) the stock take store after the warehouse filter (only non-optional filter) is loaded.
        this.warehouseComboBox.store.on('load', function () {
            this.fireEvent('filterSinglePanel', this);
        }, this);

        this.callParent(arguments);
    },

    /**
     * @return Ext.container.Container
     */
    createFilterForm: function () {
        this.warehouseComboBox = Ext.create('Shopware.apps.ViisonPickwareERPWarehouseManagement.view.WarehouseComboBox', {
            name: 'warehouseSelection',
            allowBlank: false,
            hideLabel: true,
            width: 180,
            listeners: {
                scope: this,
                select: function (combobox, record) {
                    // Ignore the change for now and let the controller handle the change
                    combobox.suspendEvents();
                    combobox.setValue(combobox.lastSelection[0]);
                    combobox.resumeEvents();
                    this.fireEvent('warehouseChanged', this, record);
                },
            },
        });

        this.userComboBox = Ext.create('Ext.form.field.ComboBox', {
            name: 'user',
            width: 160,
            store: this.userStore,
            queryMode: 'local',
            valueField: 'id',
            displayField: 'name',
            editable: false,
            hideLabel: true,
            listeners: {
                scope: this,
                select: function () {
                    this.fireEvent('userChanged', this);
                },
            },
        });

        return {
            xtype: 'toolbar',
            dock: 'top',
            ui: 'shopware-ui is--viison-common--toolbar',
            padding: '5 5 5 5',
            border: false,
            items: [
                {
                    xtype: 'label',
                    cls: 'is--viison-common--label',
                    text: ViisonCommonApp.getSnippet(
                        'warehouse_combobox/field_label',
                        'backend/viison_pickware_erp_warehouse_management/main'
                    ) + ':',
                    margin: '7 10 0 0',
                },
                this.warehouseComboBox,
                {
                    xtype: 'tbspacer',
                    width: 15,
                },
                {
                    xtype: 'label',
                    cls: 'is--viison-common--label',
                    text: this.getViisonSnippet('filter/dateFrom/label') + ':',
                    margin: '7 10 0 0',
                },
                {
                    xtype: 'datefield',
                    name: 'from',
                    format: 'd.m.Y',
                    width: 100,
                    maxValue: new Date(),
                    allowBlank: true,
                    hideLabel: true,
                    listeners: {
                        scope: this,
                        change: function (dateField, newValue, oldValue) {
                            // Ignore the change for now and let the controller handle the change
                            dateField.suspendEvents();
                            dateField.setValue(oldValue);
                            dateField.resumeEvents();
                            this.fireEvent('dateFromChanged', this, newValue);
                        },
                    },
                },
                {
                    xtype: 'tbspacer',
                    width: 15,
                },
                {
                    xtype: 'label',
                    cls: 'is--viison-common--label',
                    text: this.getViisonSnippet('filter/dateTo/label') + ':',
                    margin: '7 10 0 0',
                },
                {
                    xtype: 'datefield',
                    name: 'to',
                    format: 'd.m.Y',
                    width: 100,
                    maxValue: new Date(),
                    allowBlank: true,
                    hideLabel: true,
                    listeners: {
                        scope: this,
                        select: function (dateField, newValue) {
                            // If the date ranges are invalid (e.g. 'to' is before 'from'), correct the other date to
                            // the same value to make it valid again.
                            var dateFromFilter = this.query('[name=from]')[0];
                            if (dateFromFilter && dateFromFilter.value !== null && dateFromFilter.value > newValue) {
                                dateFromFilter.suspendEvents();
                                dateFromFilter.setValue(newValue);
                                this.fireEvent('dateFromChanged', this, newValue);
                                dateFromFilter.resumeEvents();
                            }

                            this.fireEvent('dateToChanged', this);
                        },
                    },
                },
                {
                    xtype: 'tbspacer',
                    width: 15,
                },
                {
                    xtype: 'label',
                    cls: 'is--viison-common--label',
                    text: this.getViisonSnippet('filter/user/label') + ':',
                    margin: '7 10 0 0',
                },
                this.userComboBox,
                {
                    xtype: 'tbspacer',
                    width: 15,
                },
                '->',
                {
                    xtype: 'textfield',
                    name: 'searchfield',
                    cls: 'searchfield',
                    width: 170,
                    enableKeyEvents: true,
                    emptyText: this.getViisonSnippet('filter/search_field/placeholder'),
                    listeners: {
                        buffer: 500,
                        change: function (field) {
                            this.fireEvent('searchFieldChanged', this, field.value);
                        },
                        scope: this,
                    },
                },
            ],
        };
    },

    /**
     * Updates the warehouse filter with the given value.
     *
     * @param Warehouse newWarehouse
     */
    updateWarehouse: function (newWarehouse) {
        var warehouseFilter = this.down('viison_pickware_erp_warehouse_management-warehouse_combo_box');
        // Since the change was triggered by the controller, we do not want events to be fired.
        warehouseFilter.suspendEvents();
        warehouseFilter.setValue(newWarehouse);
        warehouseFilter.resumeEvents();
    },

    /**
     * Updates the searchfield filter with the given value.
     * @param {string} newSearchQuery
     */
    updateSearchField: function (newSearchQuery) {
        var searchFieldFilter = this.query('[name=searchfield]')[0];
        searchFieldFilter.suspendEvents();
        searchFieldFilter.setValue(newSearchQuery);
        searchFieldFilter.resumeEvents();
    },

    /**
     * Updates the dateFrom filter with the given value.
     *
     * @param DateTime newWarehouse
     */
    updateDateFrom: function (newDateFrom) {
        var dateFromFilter = this.query('[name=from]')[0];
        if (dateFromFilter.getValue() !== newDateFrom) {
            // Since the change was triggered by the controller, we do not want events to be fired.
            dateFromFilter.suspendEvents();
            dateFromFilter.setValue(newDateFrom);
            dateFromFilter.resumeEvents();

            // If the date ranges are invalid (e.g. 'to' is before 'from'), correct the other date to the same value to
            // make it valid again.
            var dateToFilter = this.query('[name=to]')[0];
            if (dateToFilter && dateToFilter.value !== null && dateToFilter.value < dateFromFilter.value) {
                dateToFilter.suspendEvents();
                dateToFilter.setValue(newDateFrom);
                dateToFilter.resumeEvents();
            }
        }
    },
});
