// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPStockTakeExport.view.pending.List', {

    extend: 'Ext.grid.Panel',
    alias: 'widget.viison_pickware_erp_stock_take_export-pending-list',
    cls: 'viison-common--grid has--vertical-lines has--no-borders',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_stock_take_export/main',

    flex: 1,
    selModel: 'viisonCommonTabableCellModel',

    viewConfig: {
        enableTextSelection: true,
        // Remove the dirty flag indicator after a cell was edited
        markDirty: false,
    },
    lastSorters: [],

    /**
     * @Override
     */
    initComponent: function () {
        this.store = Ext.create('Shopware.apps.ViisonPickwareERPStockTakeExport.store.PendingStockTake');
        this.store.on('beforeload', function (store, operation) {
            // Check if the sorters have changed. If so, block the sort by returning false and let the controller handle
            // the next steps.
            var newSorters = operation.sorters;
            if (
                newSorters.length > 0
                && (
                    newSorters.length !== this.lastSorters.length
                    || newSorters[0].property !== this.lastSorters[0].property
                    || newSorters[0].direction !== this.lastSorters[0].direction
                )
            ) {
                this.fireEvent('sortChange', this, store, newSorters);

                return false;
            }

            return true;
        }, this);
        this.store.on('load', function () {
            this.fireEvent('updateButtons', this.up('panel'));
        }, this);

        this.columns = this.getColumns();
        this.dockedItems = [
            this.createPagingToolbar(),
        ];
        this.plugins = this.createPlugins();

        this.callParent(arguments);
    },

    /**
     * @return Ext.grid.column.Column[]
     */
    getColumns: function () {
        var columns = [
            {
                dataIndex: 'articleNumber',
                header: this.getViisonSnippet('grid/column/number'),
                flex: 1,
            }, {
                dataIndex: 'articleName',
                header: this.getViisonSnippet('grid/column/name'),
                flex: 2,
            }, {
                dataIndex: 'binLocationCode',
                header: this.getViisonSnippet('grid/column/binLocationCode'),
                flex: 1,
            }, {
                xtype: 'datecolumn',
                dataIndex: 'lastStockTakeDate',
                header: this.getViisonSnippet('grid/column/lastStockTakeDate'),
                format: 'd.m.Y',
                flex: 1,
            }, {
                dataIndex: 'newStock',
                header: this.getViisonSnippet('grid/column/pendingNewInstock'),
                align: 'right',
                flex: 1,
                sortable: false,
                editor: {
                    xtype: 'numberfield',
                    cls: 'is--viison-common-grid--cell-editor is--align-right',
                    allowBlank: true,
                    minValue: 0,
                    step: 1,
                },
            }, {
                dataIndex: 'comment',
                header: this.getViisonSnippet('grid/column/comment'),
                sortable: false,
                hidden: true,
                flex: 2,
                editor: {
                    xtype: 'textfield',
                    cls: 'is--viison-common-grid--cell-editor',
                },
            }, {
                xtype: 'actioncolumn',
                width: 30,
                align: 'center',
                items: [
                    {
                        iconCls: 'sprite-inbox',
                        tooltip: this.getViisonSnippet('tooltip/openArticle'),
                        scope: this,
                        handler: function (view, rowIndex) {
                            this.fireEvent('openArticle', view.getStore().getAt(rowIndex).get('articleId'));
                        },
                    },
                ],
            },
        ];

        if (ViisonCommonApp.getConfig('showCurrentStockWhenStocktaking', 'ViisonPickwareERPStockTakeExport')) {
            columns.splice(4, 0, {
                dataIndex: 'expectedStock',
                header: this.getViisonSnippet('grid/column/expectedStock'),
                align: 'right',
                flex: 1,
            });
            columns.splice(6, 0, {
                header: this.getViisonSnippet('grid/column/changeAmount'),
                align: 'right',
                flex: 1,
                renderer: function (value, meta, record) {
                    if (record.get('newStock') === null) {
                        meta.tdCls = '';

                        return '';
                    }

                    var changeAmount = record.get('newStock') - record.get('expectedStock');
                    if (changeAmount < 0) {
                        return '- ' + Math.abs(changeAmount);
                    }
                    if (changeAmount > 0) {
                        return '+ ' + Math.abs(changeAmount);
                    }

                    return '-';
                },
            });
        }

        return columns;
    },

    /**
     * Creates and returns a new paging toolbar and adds a page size selection to it.
     *
     * @return Shopware.apps.ViisonCommonPaginationToolbar
     */
    createPagingToolbar: function () {
        this.toolbar = Ext.create('Shopware.apps.ViisonCommonPaginationToolbar.view.Toolbar', {
            store: this.store,
        });
        this.toolbar.on('beforechange', function (toolbar, pageNumber) {
            this.fireEvent('pageChange', this, pageNumber);

            // Block this page change and let the controller handle it
            return false;
        }, this);

        return this.toolbar;
    },

    /**
     * @returns {Ext.grid.plugin.CellEditing[]}
     */
    createPlugins: function () {
        return [
            Ext.create('Ext.grid.plugin.CellEditing', {
                clicksToEdit: 1,
                listeners: {
                    scope: this,
                    edit: this.onEdit,
                },
            }),
        ];
    },

    /**
     * Is fired after (any) editable field was edited.
     *
     * @param Ext.grid.plugin.CellEditing plugin
     * @param editorEvent
     */
    onEdit: function (plugin, editorEvent) {
        if (editorEvent.field !== 'newStock') {
            return;
        }

        // Inform the controller to enabled/disabled action buttons.
        this.fireEvent('updateButtons', this.up('panel'));

        // Remove the comment, if the newStock value was cleared
        if (editorEvent.value === null) {
            editorEvent.record.set('comment', null);

            return;
        }

        // If the newStock is newly set (not null) and no comment is set, insert the default comment
        if (editorEvent.record.get('comment') === null || editorEvent.record.get('comment') === '') {
            editorEvent.record.set('comment', this.getViisonSnippet('stocktake/defaultComment'));
        }
    },

});
