// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPStockValuation.controller.Main', {

    extend: 'Ext.app.Controller',
    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_stock_valuation/main',

    refs: [
        {
            ref: 'reportsList',
            selector: 'viison_pickware_erp_stock_valuation-reports_list',
        },
        {
            ref: 'report',
            selector: 'viison_pickware_erp_stock_valuation-report',
        },
        {
            ref: 'filter',
            selector: 'viison_pickware_erp_stock_valuation-filter',
        },
    ],

    /**
     * @Override
     */
    init: function () {
        this.control({
            'viison_pickware_erp_stock_valuation-reports_list': {
                select: this.onReportSelected,
                deleteReport: this.onDeleteReport,
            },
            'viison_pickware_erp_stock_valuation-filter button[action=generatePreview]': {
                click: this.onGeneratePreview,
            },
        });

        // Create and show the main window
        this.mainWindow = this.getView('Main').create({
            reportStore: this.getStore('Report'),
            reportRowStore: this.getStore('ReportRow'),
            reportMethodStore: this.getStore('ReportMethod'),
        });
        this.mainWindow.show();

        this.callParent(arguments);
    },

    onReportSelected: function (_, report) {
        this.getController('Report').showReport(report);
    },

    onDeleteReport: function (report) {
        Ext.Msg.confirm(
            this.getViisonSnippet('confirm/delete_report/title'),
            this.getViisonSnippet('confirm/delete_report/message'),
            function (buttonId) {
                if (buttonId !== 'yes') {
                    return;
                }
                this.mainWindow.setLoading(true);
                Ext.Ajax.request({
                    method: 'GET',
                    url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPStockValuation/deleteReport'),
                    timeout: 600000, // 10 minutes
                    params: {
                        reportId: report.get('id'),
                    },
                    scope: this,
                    callback: function () {
                        this.getStore('Report').reload();
                        var displayedReport = this.getController('Report').report;
                        if (displayedReport && displayedReport.get('id') === report.get('id')) {
                            this.getController('Report').showReport(null);
                        }
                        this.mainWindow.setLoading(false);
                    },
                });
            },
            this
        );
    },

    onGeneratePreview: function () {
        var progressModal = Ext.MessageBox.show({
            title: this.getViisonSnippet('progress_modal/title'),
            msg: this.getViisonSnippet('progress_modal/text'),
            progress: true,
            progressText: this.getTaskDescription('report-created'),
            buttons: 0,
            closable: false,
        });
        // Seems like there is a bug that causes the progress modal to be too small. Increase the height by 10px so
        // the progress bar isn't cut off.
        progressModal.setHeight(progressModal.height + 10);
        Ext.Ajax.request({
            method: 'POST',
            url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPStockValuation/createReportPreview'),
            timeout: 600000, // 10 minutes
            params: this.getFilter().getSelectedValues(),
            scope: this,
            callback: function (options, success, response) {
                var responseData = Ext.JSON.decode(
                    response.responseText,
                    // do not throw exception in case of error:
                    true
                );
                if (!success || !responseData || !responseData.success) {
                    var message = (responseData && responseData.message) || this.getViisonSnippet('error/unknown');
                    Shopware.Msg.createGrowlMessage(
                        this.getViisonSnippet('notification/generate_preview/error/title'),
                        this.getViisonSnippet('notification/generate_preview/error/message') + ' ' + message,
                        'Pickware'
                    );
                    progressModal.hide();
                } else {
                    var tempReportStore = Ext.create('Shopware.apps.ViisonPickwareERPStockValuation.store.Report');
                    tempReportStore.loadRawData([responseData.report]);
                    var report = tempReportStore.first();
                    this.generateReport(report, progressModal);
                    progressModal.updateProgress(
                        report.get('progress'),
                        this.getTaskDescription(report.get('nextGenerationStep'))
                    );
                }
            },
        });
    },

    generateReport: function (report, progressModal) {
        Ext.Ajax.request({
            method: 'POST',
            url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPStockValuation/performNextReportGenerationStep'),
            timeout: 600000, // 10 minutes
            params: {
                reportId: report.get('id'),
            },
            scope: this,
            callback: function (options, success, response) {
                var responseData = Ext.JSON.decode(
                    response.responseText,
                    // do not throw exception in case of error:
                    true
                );
                if (!success || !responseData || !responseData.success) {
                    var message = (responseData && responseData.message) || this.getViisonSnippet('error/unknown');
                    Shopware.Msg.createGrowlMessage(
                        this.getViisonSnippet('notification/generate_preview/error/title'),
                        this.getViisonSnippet('notification/generate_preview/error/message') + ' ' + message,
                        'Pickware'
                    );
                    progressModal.hide();
                } else {
                    var tempReportStore = Ext.create('Shopware.apps.ViisonPickwareERPStockValuation.store.Report');
                    tempReportStore.loadRawData([responseData.report]);
                    var updatedReport = tempReportStore.first();
                    progressModal.updateProgress(
                        updatedReport.get('progress'),
                        this.getTaskDescription(updatedReport.get('nextGenerationStep'))
                    );
                    if (updatedReport.get('generated')) {
                        setTimeout(function () {
                            progressModal.hide();
                            this.getReportsList().getSelectionModel().deselectAll();
                            this.getController('Report').showReport(report);
                        }.bind(this), 1000);
                    } else {
                        this.generateReport(report, progressModal);
                    }
                }
            },
        });
    },

    getTaskDescription: function (generationStep) {
        switch (generationStep) {
            case 'report-created':
                return this.getViisonSnippet('progress_modal/task/creating_report');
            case 'report-prepared':
                return this.getViisonSnippet('progress_modal/task/preparing_report');
            case 'stocks-calculated':
                return this.getViisonSnippet('progress_modal/task/calculating_stocks');
            case 'purchases-calculated':
                return this.getViisonSnippet('progress_modal/task/calculating_purchases');
            case 'average-purchase-price-calculated':
                return this.getViisonSnippet('progress_modal/task/calculating_average_purchase_prices');
            case 'stock-rated':
                return this.getViisonSnippet('progress_modal/task/rating_stock');
            case 'report-persisted':
                return this.getViisonSnippet('progress_modal/task/persisting_report');
            case null:
                return this.getViisonSnippet('progress_modal/task/finished');
            default:
                return null;
        }
    },
});
