// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPStockValuation.controller.Report', {

    extend: 'Ext.app.Controller',
    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_stock_valuation/main',

    refs: [
        {
            ref: 'reportView',
            selector: 'viison_pickware_erp_stock_valuation-report',
        },
        {
            ref: 'purchasesList',
            selector: 'viison_pickware_erp_stock_valuation-report-purchases_list',
        },
        {
            ref: 'excludeArticlesWithoutStockCheckbox',
            selector: 'viison_pickware_erp_stock_valuation-report-products_list checkbox[name=excludeArticlesWithoutStock]',
        },
        {
            ref: 'showAllPurchasesCheckbox',
            selector: 'viison_pickware_erp_stock_valuation-report-purchases_list checkbox[name=showAllPurchases]',
        },
    ],

    report: null,

    /**
     * @Override
     */
    init: function () {
        this.control({
            'viison_pickware_erp_stock_valuation-report-products_list': {
                select: this.onReportRowSelected,
                showArticleDetail: this.onShowArticleDetail,
                searchFieldChanged: this.onSearchFieldChanged,
            },
            'viison_pickware_erp_stock_valuation-report-header': {
                savePreview: this.onSavePreview,
                exportReport: this.onExportData,
            },
            'viison_pickware_erp_stock_valuation-report-products_list checkbox[name=excludeArticlesWithoutStock]': {
                change: this.onExcludeArticlesWithoutStockChanged,
            },
            'viison_pickware_erp_stock_valuation-report-purchases_list checkbox[name=showAllPurchases]': {
                change: this.onShowAllPurchasesChanged,
            },
        });

        this.callParent(arguments);
    },

    showReport: function (report) {
        this.getStore('ReportRow').filterByReport(report);
        this.getPurchasesList().store.removeAll();
        this.getPurchasesList().reconfigure(this.getPurchasesList().store, this.getPurchasesList().getColumns(report));

        if (report) {
            this.getReportView().showReport(report);
            this.loadTotalAmounts(report);
        } else {
            this.getReportView().hideReport();
        }

        this.report = report;
    },

    loadTotalAmounts: function (report) {
        var url = ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPStockValuation/getReportAggregation');
        Ext.Ajax.request({
            method: 'GET',
            url: url + '?reportId=' + report.get('id'),
            scope: this,
            callback: function (options, success, response) {
                var responseData = JSON.parse(response.responseText);
                this.getReportView().updateTotalAmounts(
                    responseData.data.totalValuationNet,
                    responseData.data.totalValuationGross
                );
            },
        });
    },

    onReportRowSelected: function (_, reportRow) {
        var store = reportRow.getPurchases();
        if (!this.getShowAllPurchasesCheckbox().getValue()) {
            store.filterByValuedQuantity();
        } else {
            store.clearFilter(true);
        }
        store.sort([
            {
                property: 'date',
                direction: 'ASC',
            },
        ]);
        this.getPurchasesList().reconfigure(store, this.getPurchasesList().getColumns(this.report));
    },

    /**
     * Launches a new article sub application showing the details of the
     * given article record.
     *
     * @param { Shopware.apps.ViisonPickwareERPStockValuation.model.Article } record
     */
    onShowArticleDetail: function (record) {
        Shopware.app.Application.addSubApplication({
            name: 'Shopware.apps.Article',
            action: 'detail',
            params: {
                articleId: record.getArticleDetail().get('articleId'),
            },
        });
    },

    /**
     * @param { string } query
     */
    onSearchFieldChanged: function (query) {
        this.getStore('ReportRow').filterBySearchQuery(query);
    },

    onExcludeArticlesWithoutStockChanged: function (checkbox) {
        this.getStore('ReportRow').filterByArticlesWithStock(checkbox.getValue());
    },

    onShowAllPurchasesChanged: function (checkbox) {
        var store = this.getPurchasesList().getStore();
        if (!checkbox.getValue()) {
            store.filterByValuedQuantity();
        } else {
            store.clearFilter();
        }
    },

    onExportData: function () {
        var excludeArticlesWithoutStock = this.getExcludeArticlesWithoutStockCheckbox().getValue();
        var url = ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPStockValuation/exportReportAsCsv', {
            reportId: this.report.get('id'),
            excludeArticlesWithoutStock: excludeArticlesWithoutStock,
        });
        window.open(url, '_blank');
    },

    onSavePreview: function () {
        var now = Ext.Date.format(new Date(), 'Y-m-d');
        var reportingDay = Ext.Date.format(this.report.get('reportingDay'), 'Y-m-d');

        if (reportingDay < now) {
            Ext.Msg.show({
                title: this.getViisonSnippet('confirm/save_preview/title'),
                msg: this.getViisonSnippet('confirm/save_preview/message'),
                prompt: true,
                value: this.report.get('comment') || '',
                buttons: Ext.Msg.OKCANCEL,
                icon: Ext.window.MessageBox.INFO,
                multiline: true,
                fn: function (buttonId, text) {
                    if (buttonId !== 'ok') {
                        return;
                    }
                    this.getController('Main').mainWindow.setLoading({ msg: this.getViisonSnippet('loading/save_report') });
                    var reportId = this.report.get('id');
                    Ext.Ajax.request({
                        method: 'POST',
                        url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPStockValuation/persistReport'),
                        params: {
                            reportId: reportId,
                            comment: text,
                        },
                        scope: this,
                        callback: function () {
                            this.showReport(null);
                            var reportStore = this.getStore('Report');
                            reportStore.reload({
                                callback: function () {
                                    var savedReport = reportStore.getById(reportId);
                                    if (savedReport) {
                                        this.getController('Main').getReportsList().getSelectionModel().select(savedReport);
                                    }
                                },
                                scope: this,
                            });
                            this.getController('Main').mainWindow.setLoading(false);
                        },
                    });
                },
                scope: this,
            });
        } else {
            Ext.Msg.alert({
                title: this.getViisonSnippet('alert/save_preview/title'),
                msg: this.getViisonSnippet('alert/save_preview/message'),
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.WARNING,
            });
        }
    },

});
