// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPSupplierCommon.view.AssignmentPanel', {

    extend: 'Ext.panel.Panel',
    alias: 'widget.viison_pickware_erp_supplier_common-assignment_panel',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_supplier_common/main',

    // Define snippets here to allow easy overriding in subclasses
    leftTitle: ViisonCommonApp.getSnippet(
        'view/assignment_panel/title/left',
        'backend/viison_pickware_erp_supplier_common/main'
    ),
    rightTitle: ViisonCommonApp.getSnippet(
        'view/assignment_panel/title/right',
        'backend/viison_pickware_erp_supplier_common/main'
    ),

    leftStore: null,
    rightStore: null,
    layout: {
        type: 'hbox',
        align: 'stretch',
    },
    bodyStyle: {
        border: 'none',
    },
    defaults: {
        border: 0,
    },

    /**
     * @Override
     */
    initComponent: function () {
        // Make sure both stores are created
        if (!this.leftStore) {
            this.leftStore = this.createLeftStore();
        }
        if (!this.rightStore) {
            this.rightStore = this.createRightStore();
        }

        // Create both grid panels
        this.leftPanel = this.createLeftPanel();
        this.rightPanel = this.createRightPanel();

        // Add both grid panels as well as a panel containing the assignment buttons
        this.items = [
            this.createLeftContainer(),
            this.createMiddleContainer(),
            this.createRightContainer(),
        ];

        this.callParent(arguments);
    },

    /**
     * Creates the store that is used for the left (available) grid panel.
     *
     * @return Ext.data.Store
     */
    createLeftStore: function () {
        // Method must be overridden
    },

    /**
     * Creates the store that is used for the right (assigned) grid panel.
     *
     * @return Ext.data.Store
     */
    createRightStore: function () {
        // Method must be overridden
    },

    /**
     * Creates a container panel containing the left grid panel.
     *
     * @return Ext.grid.Panel
     */
    createLeftContainer: function () {
        return {
            xtype: 'panel',
            flex: 1,
            layout: 'fit',
            border: false,
            items: [
                this.leftPanel,
            ],
        };
    },

    /**
     * Creates the grid panel that is used for the left (available) side.
     *
     * @return Ext.grid.Panel
     */
    createLeftPanel: function () {
        return Ext.create('Ext.grid.Panel', this.getLeftPanelConfig());
    },

    /**
     * Creates and returns an object containing the configuration for the left (available) grid panel.
     *
     * @return Object
     */
    getLeftPanelConfig: function () {
        return {
            cls: 'viison-common--grid has--vertical-lines has--no-borders',
            stripeRows: true,
            multiSelect: true,
            hideHeaders: false,
            header: false,
            viewConfig: {
                plugins: {
                    ptype: 'gridviewdragdrop',
                    dragGroup: ('leftGridDDGroup-' + this.id),
                    dropGroup: ('rightGridDDGroup-' + this.id),
                },
                listeners: {
                    drop: function (node, data) {
                        this.clearSelections();
                        this.fireEvent('moveItemsFromRightToLeftViaDragDrop', this, data.records);
                    },
                    scope: this,
                },
            },
            columns: this.createLeftColumns(),
            store: this.leftStore,
            dockedItems: [
                this.createSearchToolbar(this.leftStore, 'leftPanelSearchFieldChanged', this.leftTitle),
                this.createPagingToolbar(this.leftStore),
            ],
        };
    },

    /**
     * @return Ext.grid.column.Column[]
     */
    createLeftColumns: function () {
        return [];
    },

    /**
     * Creates a container panel containing the right grid panel.
     *
     * @return Ext.grid.Panel
     */
    createRightContainer: function () {
        return {
            xtype: 'panel',
            flex: 1,
            layout: 'fit',
            border: false,
            items: [
                this.rightPanel,
            ],
        };
    },

    /**
     * Updates the warehouse name for the target warehouse label.
     *
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.model.Warehouse warehouse
     */
    setTargetWarehouse: function (warehouse) {
        this.targetWarehouse = warehouse;
    },

    /**
     * Creates the grid panel that is used for the right (assigned) side.
     *
     * @return Ext.grid.Panel
     */
    createRightPanel: function () {
        return Ext.create('Ext.grid.Panel', this.getRightPanelConfig());
    },

    /**
     * Creates and returns an object containing the configuration for the right (assigned) grid panel.
     *
     * @return Object
     */
    getRightPanelConfig: function () {
        return {
            cls: 'viison-common--grid has--vertical-lines has--no-borders',
            layout: 'fit',
            flex: 1,
            stripeRows: true,
            multiSelect: true,
            hideHeaders: false,
            header: false,
            viewConfig: {
                plugins: {
                    ptype: 'gridviewdragdrop',
                    dragGroup: ('rightGridDDGroup-' + this.id),
                    dropGroup: ('leftGridDDGroup-' + this.id),
                },
                listeners: {
                    drop: function (node, data) {
                        this.clearSelections();
                        this.fireEvent('moveItemsFromLeftToRightViaDragDrop', this, data.records);
                    },
                    scope: this,
                },
            },
            listeners: {
                selectionchange: function (panel, selection) {
                    this.fireEvent('rightPanelSelectionChange', panel, selection);
                },
                scope: this,
            },
            columns: this.createRightColumns(),
            store: this.rightStore,
            dockedItems: this.addSearchToolbar(),
        };
    },

    /**
     * Adds SearchToolbar only to supplier managment
     *
     * @return Ext.toolbar.Toolbar
     */
    addSearchToolbar: function () {
        if (this.getXType() === 'viison_pickware_erp_supplier_management-edit-article_assignment') {
            return [this.createSearchToolbar(this.rightStore, 'rightPanelSearchFieldChanged', this.rightTitle)];
        }

        return [];
    },

    /**
     * @return Ext.grid.column.Column[]
     */
    createRightColumns: function () {
        return [];
    },

    /**
     * Creates and returns the two left/right assignment buttons.
     *
     * @return Ext.button.Button[]
     */
    createMiddleContainer: function () {
        return {
            xtype: 'panel',
            width: 32,
            border: 1,
            layout: {
                type: 'vbox',
                pack: 'center',
            },
            bodyStyle: {
                backgroundColor: '#ebedef',
                borderTop: 'none',
                borderBottom: 'none',
            },
            items: this.createMiddleButtons(),
        };
    },

    /**
     * Creates and returns the two left/right assignment buttons.
     *
     * @return Ext.button.Button[]
     */
    createMiddleButtons: function () {
        return [{
            xtype: 'button',
            cls: Ext.baseCSSPrefix + 'form-itemselector-btn',
            iconCls: Ext.baseCSSPrefix + 'form-itemselector-add',
            tooltip: '',
            navBtn: true,
            margin: '4 4 2 4',
            handler: function () {
                var selectedItems = this.swapSelectedItems(this.leftPanel, this.rightPanel);
                if (selectedItems.length > 0) {
                    this.fireEvent('moveItemsFromLeftToRight', this, selectedItems);
                }
            },
            scope: this,
        }, {
            xtype: 'button',
            cls: Ext.baseCSSPrefix + 'form-itemselector-btn',
            iconCls: Ext.baseCSSPrefix + 'form-itemselector-remove',
            tooltip: '',
            navBtn: true,
            margin: '2 4 4 4',
            handler: function () {
                var selectedItems = this.swapSelectedItems(this.rightPanel, this.leftPanel);
                if (selectedItems.length > 0) {
                    this.fireEvent('moveItemsFromRightToLeft', this, selectedItems);
                }
            },
            scope: this,
        }];
    },

    /**
     * Creates and returns a new paging toolbar and adds a page size selection to it.
     *
     * @param Ext.data.Store store
     * @return Shopware.apps.ViisonCommonPaginationToolbar
     */
    createPagingToolbar: function (store) {
        return {
            xtype: 'viison_common_pagination_toolbar-toolbar',
            store: store,
        };
    },

    /**
     * Creates and returns a new toolbar that contains a title on the left and a search
     * field on the right.
     *
     * @param Ext.data.Store store
     * @param string changeEventName
     * @param string title
     * @return Ext.toolbar.Toolbar
     */
    createSearchToolbar: function (store, changeEventName, title) {
        return {
            xtype: 'toolbar',
            dock: 'top',
            ui: 'shopware-ui',
            padding: '10 10 10 0',
            items: [
                {
                    xtype: 'container',
                    html: (title || ''),
                    padding: '0 0 0 10',
                    style: {
                        color: '#475c6a',
                        fontWeight: 'normal',
                        fontSize: '110%',
                        textShadow: '1px 1px white',
                    },
                },
                '->',
                {
                    xtype: 'textfield',
                    name: 'searchfield',
                    cls: 'searchfield',
                    width: 170,
                    enableKeyEvents: true,
                    emptyText: this.getViisonSnippet('view/assignment_panel/search_field/placeholder'),
                    listeners: {
                        buffer: 500,
                        change: function (field) {
                            this.fireEvent(changeEventName, field.value, store);
                        },
                        scope: this,
                    },
                },
            ],
        };
    },

    /**
     * Moves all rows that are selected in the 'fromPanel' to the 'toPanel',
     * by removing the respective records from one store and adding them to the other.
     *
     * @param Ext.grid.Panel fromPanel
     * @param Ext.grid.Panel toPanel
     * @param Ext.data.Model[]
     */
    swapSelectedItems: function (fromPanel, toPanel) {
        // Get the selected items and clear both selection models
        var selectedItems = fromPanel.getSelectionModel().getSelection();
        this.clearSelections();

        // Move the selected items from one store to the other
        fromPanel.getStore().remove(selectedItems);
        toPanel.getStore().add(selectedItems);

        return selectedItems;
    },

    /**
     * Deselects all items in both grid panels.
     */
    clearSelections: function () {
        this.leftPanel.getSelectionModel().deselectAll();
        this.rightPanel.getSelectionModel().deselectAll();
    },

});
