// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPSupplierManagement.controller.Edit', {

    extend: 'Ext.app.Controller',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_supplier_management/main',

    /**
     * @Override
     */
    init: function () {
        this.control({
            'viison_pickware_erp_supplier_management-edit-base button[action=cancel]': {
                click: this.onCancel,
            },
            'viison_pickware_erp_supplier_management-edit-base button[action=save]': {
                click: this.onSaveBase,
            },
            'viison_pickware_erp_supplier_management-edit-article_assignment': {
                moveItemsFromLeftToRight: this.onMoveItemsFromLeftToRight,
                moveItemsFromLeftToRightViaDragDrop: this.onMoveItemsFromLeftToRight,
                moveItemsFromRightToLeft: this.onMoveItemsFromRightToLeft,
                moveItemsFromRightToLeftViaDragDrop: this.onMoveItemsFromRightToLeft,
                leftPanelSearchFieldChanged: this.onSearchFieldChanged,
                rightPanelSearchFieldChanged: this.onSearchFieldChanged,
                articleRowEdited: this.onArticleRowEdited,
                showArticleDetail: this.onShowArticleDetail,
            },
            'viison_pickware_erp_supplier_management-edit-fabricator_assignment': {
                moveItemsFromLeftToRight: this.onMoveItemsFromLeftToRight,
                moveItemsFromLeftToRightViaDragDrop: this.onMoveItemsFromLeftToRight,
                moveItemsFromRightToLeft: this.onMoveItemsFromRightToLeft,
                moveItemsFromRightToLeftViaDragDrop: this.onMoveItemsFromRightToLeft,
                leftPanelSearchFieldChanged: this.onSearchFieldChanged,
                rightPanelSearchFieldChanged: this.onSearchFieldChanged,
            },
            'viison_pickware_erp_supplier_management-edit-fabricator_assignment button[action=assignAllFabricatorArticles]': {
                click: this.onAssignAllFabricatorArticles,
            },
        });

        this.callParent(arguments);
    },

    /**
     * Creates a new edit window for the given supplier record.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierManagement.model.Supplier record
     * @return Shopware.apps.ViisonPickwareERPSupplierManagement.view.Edit
     */
    createEditWindow: function (record) {
        // Create the two stores for the article assignment
        var leftArticleStore = Ext.create('Shopware.apps.ViisonPickwareERPSupplierManagement.store.ArticleDetail', {});
        if (record.get('id')) {
            leftArticleStore.filters.add('supplierBlacklist', Ext.create('Ext.util.Filter', {
                property: 'supplierIdBlacklist',
                expression: '!=',
                value: record.get('id'),
            }));
            leftArticleStore.getProxy().extraParams.currencyId = record.get('currencyId');
        }
        var rightArticleStore = Ext.create('Shopware.apps.ViisonPickwareERPSupplierManagement.store.AssignedArticleDetail', {});
        if (record.get('id')) {
            rightArticleStore.filters.add('supplierId', Ext.create('Ext.util.Filter', {
                property: 'supplierMapping.supplierId',
                value: record.get('id'),
            }));
        }

        // Create the two stores for the fabricator assignment
        var leftFabricatorStore = Ext.create('Shopware.apps.ViisonPickwareERPSupplierManagement.store.Fabricator', {});
        if (record.get('id')) {
            leftFabricatorStore.filters.add('supplierBlacklist', Ext.create('Ext.util.Filter', {
                property: 'supplierIdBlacklist',
                expression: '!=',
                value: record.get('id'),
            }));
        }
        var rightFabricatorStore = Ext.create('Shopware.apps.ViisonPickwareERPSupplierManagement.store.AssignedFabricator', {});
        if (record.get('id')) {
            rightFabricatorStore.filters.add('supplierId', Ext.create('Ext.util.Filter', {
                property: 'manufacturerMapping.supplierId',
                value: record.get('id'),
            }));
        }

        var documentLocalizationSubShopStore = Ext.create('Shopware.apps.ViisonPickwareERPSupplierManagement.store.SupplierDocumentLocalizationSubShop', {});
        var currencyStore = Ext.create('Shopware.apps.Base.store.Currency', {});

        // Create the edit window
        var title;
        if (record.get('name').length > 0) {
            title = this.getViisonSnippet('edit/window/title/edit') + ' - ' + record.get('name');
        } else {
            title = this.getViisonSnippet('edit/window/title/add');
        }

        var editWindow = this.getView('Edit').create({
            title: title,
            record: record,
            documentLocalizationSubShopStore: documentLocalizationSubShopStore.load(),
            leftArticleStore: leftArticleStore.load(),
            rightArticleStore: rightArticleStore.load(),
            leftFabricatorStore: leftFabricatorStore.load(),
            rightFabricatorStore: rightFabricatorStore.load(),
            currencyStore: currencyStore.load(),
        });

        return editWindow;
    },

    /**
     * Closes the edit window the given button is part of.
     *
     * @param Ext.button.Button button
     */
    onCancel: function (button) {
        button.up('viison_pickware_erp_supplier_management-edit').close();
    },

    /**
     * Saves the base data in the supplier record associated with the window,
     * the given button is part of.
     *
     * @param Ext.button.Button button
     */
    onSaveBase: function (button) {
        var baseForm = button.up('viison_pickware_erp_supplier_management-edit-base');
        if (!baseForm.getForm().isValid()) {
            return;
        }

        // If this is a new supplier, we need to add it to the store so that it gets persisted
        if (baseForm.record.phantom) {
            this.getController('Main').supplierStore.add(baseForm.record);
        }

        // Update the edited record with the form date
        baseForm.getForm().updateRecord(baseForm.record);

        // If this is a new supplier, no article detail mappings exist and we can simply continue with the save.
        if (baseForm.record.get('id') === 0) {
            this.syncSuppliers(button);

            return;
        }

        // Check if the supplier currency was changed. If so, ask the user to convert the purchase prices of any article
        // detail mappings of this supplier. Continue with no prompt if the currency was not changed or no mappings
        // exist.
        Ext.Ajax.request({
            url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPSupplierManagement/checkForCurrencyChange'),
            params: {
                id: baseForm.record.get('id'),
                currencyId: baseForm.record.get('currencyId'),
            },
            success: function (response) {
                var jsonResponseText = Ext.JSON.decode(response.responseText, true);
                if (!jsonResponseText || !jsonResponseText.success) {
                    Shopware.Notification.createGrowlMessage(
                        this.getViisonSnippet('notification/error/title'),
                        this.getViisonSnippet('notification/error/message/check_for_currency_change'),
                        'ViisonPickwareERPSupplierManagement'
                    );

                    return;
                }

                if (!jsonResponseText.isCurrencyChange || jsonResponseText.numberOfArticleDetailMappings === 0) {
                    this.syncSuppliers(button);

                    return;
                }

                this.promptConfirmationToConvertPurchasePrices(
                    button,
                    baseForm.record,
                    jsonResponseText.numberOfArticleDetailMappings
                );
            },
            scope: this,
        });
    },

    /**
     * Prompts a confirmation window to migrate the purchase prices of any article detail mapping of the current
     * supplier.
     *
     * @param {Ext.button.Button} button
     * @param {Shopware.apps.ViisonPickwareERPSupplierManagement.model.Supplier} record
     * @param {int} numberOfArticleDetailMappings
     */
    promptConfirmationToConvertPurchasePrices: function (button, record, numberOfArticleDetailMappings) {
        var message = this.getViisonSnippet('notification/message/migrate_purchase_prices_confirmation');
        message = message.replace('%d', numberOfArticleDetailMappings);
        Ext.Msg.confirm(
            this.getViisonSnippet('notification/title/migrate_purchase_prices_title'),
            message,
            function (buttonId) {
                if (buttonId === 'yes') {
                    Ext.Ajax.request({
                        url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPSupplierManagement/convertPurchasePrices'),
                        params: {
                            id: record.get('id'),
                            currencyId: record.get('currencyId'),
                        },
                        success: function (response) {
                            var jsonResponseText = Ext.JSON.decode(response.responseText, true);
                            if (!jsonResponseText || !jsonResponseText.success) {
                                Shopware.Notification.createGrowlMessage(
                                    this.getViisonSnippet('notification/error/title'),
                                    this.getViisonSnippet('notification/error/message/migrate_purchase_prices'),
                                    'ViisonPickwareERPSupplierManagement'
                                );

                                return;
                            }

                            Shopware.Notification.createGrowlMessage(
                                this.getViisonSnippet('notification/success/title'),
                                this.getViisonSnippet('notification/success/message/migrate_purchase_prices'),
                                'ViisonPickwareERPSupplierManagement'
                            );
                            this.syncSuppliers(button);
                        },
                        scope: this,
                    });
                } else {
                    this.syncSuppliers(button);
                }
            },
            this
        );
    },

    /**
     * Sync the changes of the current supplier to the server.
     *
     * @param {Ext.button.Button} button
     */
    syncSuppliers: function (button) {
        this.getController('Main').syncSuppliers(
            this.getViisonSnippet('notification/error/message/save'),
            function (success, jsonResponse) {
                if (success) {
                    button.up('viison_pickware_erp_supplier_management-edit').close();
                }
                if (!success && jsonResponse.message) {
                    Shopware.Notification.createGrowlMessage(
                        this.getViisonSnippet('notification/error/title'),
                        jsonResponse.message,
                        'ViisonPickwareERPSupplierManagement'
                    );
                }
            }.bind(this)
        );
    },

    /**
     * Marks the updated record as dirty and and syncs the changes.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierManagement.view.edit.ArticleAssignment panel
     * @param Shopware.apps.ViisonPickwareERPSupplierManagement.model.ArticleDetail record
     */
    onArticleRowEdited: function (panel, record) {
        // Save the changes
        record.setDirty();
        this.syncAssignmentStore(panel);
    },

    /**
     * Launches a new article sub application showing the details of the
     * given article record.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierCommon.model.ArticleDetail record
     */
    onShowArticleDetail: function (record) {
        Shopware.app.Application.addSubApplication({
            name: 'Shopware.apps.Article',
            action: 'detail',
            params: {
                articleId: record.get('articleId'),
            },
        });
    },

    /**
     * Sets the supplier ID of the supplier that is being edited to each of
     * the added newly assigned records syncs the changes.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierCommon.view.AssignmentPanel panel
     * @param Ext.data.Model[] records
     */
    onMoveItemsFromLeftToRight: function (panel, records) {
        // Update assignments
        Ext.each(records, function (assignedRecord) {
            assignedRecord.set('supplierId', panel.record.get('id'));
            assignedRecord.commit();
            assignedRecord.phantom = true;
        });

        // Save the changes
        this.syncAssignmentStore(panel, function () {
            panel.leftStore.loadPage(panel.leftStore.currentPage);
            panel.rightStore.loadPage(panel.rightStore.currentPage);
        });
    },

    /**
     * Syncs the changes made to the right (assigned) store before reloading both
     * the left (available) and right stores.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierCommon.view.AssignmentPanel panel
     */
    onMoveItemsFromRightToLeft: function (panel) {
        // Save the changes
        this.syncAssignmentStore(panel, function () {
            panel.leftStore.loadPage(panel.leftStore.currentPage);
            panel.rightStore.loadPage(panel.rightStore.currentPage);
        });
    },

    /**
     * Sets the given query as an extra parameter to the given store's proxy
     * and loads the first page of the store.
     *
     * @param string query
     * @param Ext.data.Store store
     */
    onSearchFieldChanged: function (query, store) {
        store.getProxy().extraParams.query = query;
        store.loadPage(1);
    },

    /**
     * Assigns the articles of all assigned fabricators to the given supplier. If the update succeeds,
     * both article assignment stored are reloaded to reflect the updated assignments.
     *
     * @param Ext.button.Button button
     */
    onAssignAllFabricatorArticles: function (button) {
        var record = button.up('viison_pickware_erp_supplier_management-edit-fabricator_assignment').record;
        var fabricatorAssignmentPanel = button.up('viison_pickware_erp_supplier_management-edit').down('viison_pickware_erp_supplier_management-edit-fabricator_assignment');
        var articleAssignmentPanel = button.up('viison_pickware_erp_supplier_management-edit').down('viison_pickware_erp_supplier_management-edit-article_assignment');
        fabricatorAssignmentPanel.setLoading(true);
        Ext.Ajax.request({
            url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPSupplierManagement/assignFabricatorArticles'),
            params: {
                id: record.get('id'),
            },
            success: function () {
                fabricatorAssignmentPanel.setLoading(false);
                // Reload both stores of the article assignment to get the update assignments
                articleAssignmentPanel.leftStore.loadPage(1);
                articleAssignmentPanel.rightStore.loadPage(1);
            },
            failure: function () {
                fabricatorAssignmentPanel.setLoading(false);
                Shopware.Notification.createGrowlMessage(
                    this.getViisonSnippet('notification/error/title'),
                    this.getViisonSnippet('notification/error/message/save_all_fabricator_article_mappings'),
                    'ViisonPickwareERPSupplierManagement'
                );
            },
            scope: this,
        });
    },

    /**
     * Syncs the right (assigned) store of the given panel.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierCommon.view.AssignmentPanel panel
     * @param function callback (optional)
     */
    syncAssignmentStore: function (panel, callback) {
        // Check the store for any changes
        var store = panel.rightStore;
        var numChanges = store.getNewRecords().length + store.getModifiedRecords().length + store.getRemovedRecords().length;
        if (numChanges === 0) {
            if (Ext.isFunction(callback)) {
                callback(true);
            }

            return;
        }

        panel.rightPanel.setLoading(true);
        store.sync({
            scope: this,
            success: function () {
                panel.rightPanel.setLoading(false);
                if (Ext.isFunction(callback)) {
                    callback(true);
                }
            },
            failure: function () {
                panel.rightPanel.setLoading(false);
                // Revert any changes like added, updated or removed records
                store.rejectChanges();
                // Display an error notification
                var message = '';
                if (store.model.modelName === 'Shopware.apps.ViisonPickwareERPSupplierManagement.model.ArticleDetail') {
                    message = this.getViisonSnippet('notification/error/message/save_article_mappings');
                } else if (store.model.modelName === 'Shopware.apps.ViisonPickwareERPSupplierManagement.model.Fabricator') {
                    message = this.getViisonSnippet('notification/error/message/save_fabricator_mappings');
                }
                Shopware.Notification.createGrowlMessage(
                    this.getViisonSnippet('notification/error/title'),
                    message,
                    'ViisonPickwareERPSupplierManagement'
                );
                if (Ext.isFunction(callback)) {
                    callback(false);
                }
            },
        });
    },

});
