// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPSupplierManagement.controller.Main', {

    extend: 'Ext.app.Controller',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_supplier_management/main',

    mainWindow: null,
    refs: [
        { ref: 'listView', selector: 'viison_pickware_erp_supplier_management-main-list' },
        { ref: 'detailView', selector: 'viison_pickware_erp_supplier_management-main-details' },
    ],

    /**
     * @Override
     */
    init: function () {
        this.control({
            'viison_pickware_erp_supplier_management-main-list button[action=addSupplier]': {
                click: this.onAddSupplier,
            },
            'viison_pickware_erp_supplier_management-main-list': {
                editSupplier: this.onEditSupplier,
                deleteSupplier: this.onDeleteSupplier,
                select: this.onSupplierSelected,
                searchFieldChanged: this.onSearchFieldChanged,
            },
        });

        // Create the main supplier store
        this.supplierStore = Ext.create('Shopware.apps.ViisonPickwareERPSupplierManagement.store.Supplier', {});
        this.supplierStore.on('load', this.clearDetailView, this);
        this.supplierStore.load();

        // Create the main window
        this.mainView = this.getView('Main').create({
            supplierStore: this.supplierStore,
        });

        this.callParent(arguments);
    },

    /**
     * Creates an empty supplier model instance and passes it to a new edit window.
     */
    onAddSupplier: function () {
        Ext.Ajax.request({
            url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPSupplierManagement/getDefaultSupplierData'),
            success: function (response) {
                var jsonResponseText = Ext.JSON.decode(response.responseText, true);
                if (!jsonResponseText || !jsonResponseText.success) {
                    Shopware.Notification.createGrowlMessage(
                        this.getViisonSnippet('notification/error/title'),
                        this.getViisonSnippet('notification/error/message/default_supplier_data'),
                        'ViisonPickwareERPSupplierManagement'
                    );

                    return;
                }
                this.openSupplierEditWindow(
                    jsonResponseText.unusedSupplierNumber,
                    jsonResponseText.defaultCurrencyId
                );
            },
            scope: this,
        });
    },

    /**
     * Opens the "edit" window for a new supplier with the given supplierNumber and currency id
     *
     * @param {string} supplierNumber
     * @param {int} currencyId
     */
    openSupplierEditWindow: function (supplierNumber, currencyId) {
        var newSupplierRecord = Ext.create('Shopware.apps.ViisonPickwareERPSupplierManagement.model.Supplier', {
            number: supplierNumber,
            currencyId: currencyId,
        });
        this.getController('Edit').createEditWindow(newSupplierRecord);
    },

    /**
     * Opens a new edit window using the given supplier record.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierManagement.model.Supplier record
     */
    onEditSupplier: function (record) {
        this.getController('Edit').createEditWindow(record);
    },

    /**
     * Ask the user for confirmation of the deletion and, if confirmed, deletes
     * the given record from the store and syncs the store afterwards.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierManagement.model.Supplier record
     */
    onDeleteSupplier: function (record) {
        // Present a confirm alert before deleting the supplier
        Ext.Msg.confirm(
            this.getViisonSnippet('alert/title'),
            this.getViisonSnippet('alert/message/delete'),
            function (buttonId) {
                if (buttonId !== 'yes') {
                    return;
                }

                // Remove the given record from the store and sync the changes
                this.supplierStore.remove(record);
                this.syncSuppliers(this.getViisonSnippet('notification/error/message/delete'));
            },
            this
        );
    },

    /**
     * Updates the content of the detail panel using the data of the given record
     * and expands the detail panel.
     *
     * @param Ext.selection.DataViewModel selection
     * @param Shopware.apps.ViisonPickwareERPSupplierManagement.model.Supplier record
     */
    onSupplierSelected: function (selection, record) {
        this.getDetailView().dataView.update(record.data);
        this.getDetailView().expand();
    },

    /**
     * Sets the given query as an extra parameter to the supplier store's proxy
     * and reloads the store.
     *
     * @param string query
     */
    onSearchFieldChanged: function (query) {
        this.supplierStore.getProxy().extraParams.query = query;
        this.supplierStore.loadPage(1);
    },

    /**
     * Synchronizes the suppliers store and, in case of failure, shows a growl notification.
     *
     * @param string errorMessage
     * @param function callback (optional)
     */
    syncSuppliers: function (errorMessage, callback) {
        // Check the store for any changes
        var numChanges = (
            this.supplierStore.getNewRecords().length
            + this.supplierStore.getModifiedRecords().length
            + this.supplierStore.getRemovedRecords().length
        );
        if (numChanges === 0) {
            if (Ext.isFunction(callback)) {
                callback(true);
            }

            return;
        }

        // Sync the store
        this.selectedRow = null;
        this.mainView.setLoading(true);
        this.supplierStore.sync({
            scope: this,
            success: function (response) {
                this.mainView.setLoading(false);
                // We need to load the store after sync because the sync does not map the currency association
                // correctly. Reloading the store calls the controller which fills each entity correctly.
                this.supplierStore.load();
                this.clearDetailView();
                if (Ext.isFunction(callback)) {
                    callback(true, response.proxy.getReader().jsonData);
                }
            },
            failure: function (response) {
                this.mainView.setLoading(false);
                // Revert any changes like added, updated or removed records
                this.supplierStore.rejectChanges();
                // Display an error notification
                var message = errorMessage || '';
                Shopware.Notification.createGrowlMessage(
                    this.getViisonSnippet('notification/error/title'),
                    message,
                    'ViisonPickwareERPSupplierManagement'
                );
                if (Ext.isFunction(callback)) {
                    callback(false, response.proxy.getReader().jsonData);
                }
            },
        });
    },

    /**
     * Clears the detail view.
     */
    clearDetailView: function () {
        this.getDetailView().dataView.update({});
    },

});
