// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

/**
 * Overrides the default readRecords implementation to flatten the data,
 * because the actual article data is nested in an 'articleDetail' object,
 * while the supplier article detail assignment data is toplevel.
 */
Ext.define('Shopware.apps.ViisonPickwareERPSupplierManagement.reader.AssignedArticleDetailJson', {

    extend: 'Ext.data.reader.Json',
    type: 'json',
    root: 'data',
    totalProperty: 'total',

    /**
     * @Override
     */
    readRecords: function (data) {
        if (data && data.data) {
            var newData = [];
            Ext.each(data.data, function (item) {
                var newItem = item.articleDetail;
                newItem.supplierId = item.supplierId;
                newItem.purchasePrice = item.purchasePrice;
                newItem.packingUnit = item.packingUnit;
                newItem.minimumOrderAmount = item.minimumOrderAmount;
                newItem.supplierArticleNumber = item.supplierArticleNumber;
                newData.push(newItem);
            });
            data.data = newData;
        }

        return this.callParent([data]);
    },

});

/**
 * Overrides the default getRecordData implementation to only send information about
 * the supplier article details assignment, since all article information is static.
 */
Ext.define('Shopware.apps.ViisonPickwareERPSupplierManagement.writer.AssignedArticleDetailJson', {

    extend: 'Ext.data.writer.Json',
    type: 'json',
    root: 'data',
    allowSingle: false,

    /**
     * @Override
     */
    getRecordData: function (record) {
        return {
            supplierId: record.get('supplierId'),
            articleDetailId: record.get('id'),
            purchasePrice: record.get('purchasePrice'),
            packingUnit: record.get('packingUnit'),
            minimumOrderAmount: record.get('minimumOrderAmount'),
            supplierArticleNumber: record.get('supplierArticleNumber'),
        };
    },

});

/**
 * Although this store extends the customised version of the common ArticleDetail store,
 * it communicates with the ViisonPickwareERPSupplierManagement and reads/writes supplier article
 * detail assignments. However, these assignments are locally handled as ArticleDetail
 * model instances. This allows us to use drag'n'drop from an ArticleDetail grid to one
 * that actually holds supplier article detail assignments. The communication with the
 * different API is possible by using the AssignedArticleDetailJson readers and writers,
 * which 'translate' between the data format of supplier article detail assignments and
 * (customised) article details.
 */
Ext.define('Shopware.apps.ViisonPickwareERPSupplierManagement.store.AssignedArticleDetail', {

    extend: 'Shopware.apps.ViisonPickwareERPSupplierManagement.store.ArticleDetail',
    sorters: [
        { property: 'name' },
    ],
    constructor: function () {
        this.setProxy({
            type: 'ajax',
            batchActions: true,
            api: {
                read: ViisonCommonApp.assembleBackendUrl(
                    'ViisonPickwareERPSupplierManagement/getSupplierArticleDetailList'
                ),
                create: ViisonCommonApp.assembleBackendUrl(
                    'ViisonPickwareERPSupplierManagement/createSupplierArticleDetails'
                ),
                update: ViisonCommonApp.assembleBackendUrl(
                    'ViisonPickwareERPSupplierManagement/updateSupplierArticleDetails'
                ),
                destroy: ViisonCommonApp.assembleBackendUrl(
                    'ViisonPickwareERPSupplierManagement/deleteSupplierArticleDetails'
                ),
            },
            reader: Ext.create('Shopware.apps.ViisonPickwareERPSupplierManagement.reader.AssignedArticleDetailJson', {}),
            writer: Ext.create('Shopware.apps.ViisonPickwareERPSupplierManagement.writer.AssignedArticleDetailJson', {}),
        });

        this.callParent(arguments);
    },

});
