// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

/**
 * Overrides the default readRecords implementation to flatten the data data,
 * because the actual fabricator data is nested in an 'fabricator' object,
 * while the supplier fabricator assignment data is toplevel.
 */
Ext.define('Shopware.apps.ViisonPickwareERPSupplierManagement.reader.AssignedFabricatorJson', {

    extend: 'Ext.data.reader.Json',
    type: 'json',
    root: 'data',
    totalProperty: 'total',

    /**
     * @Override
     */
    readRecords: function (data) {
        if (data && data.data) {
            var newData = [];
            Ext.each(data.data, function (item) {
                var newItem = item.manufacturer;
                newItem.supplierId = item.supplierId;
                newData.push(newItem);
            });
            data.data = newData;
        }

        return this.callParent([data]);
    },

});

/**
 * Overrides the default getRecordData implementation to only send information about
 * the supplier fabricator assignment, since all fabricator information is static.
 */
Ext.define('Shopware.apps.ViisonPickwareERPSupplierManagement.writer.AssignedFabricatorJson', {

    extend: 'Ext.data.writer.Json',
    type: 'json',
    root: 'data',
    allowSingle: false,

    /**
     * @Override
     */
    getRecordData: function (record) {
        return {
            supplierId: record.get('supplierId'),
            manufacturerId: record.get('id'),
        };
    },

});

/**
 * Although this store extends the customised version of the common Fabricator store,
 * it communicates with the ViisonPickwareERPSupplierManagement and reads/writes supplier fabricator
 * assignments. However, these assignments are locally handled as Fabricator model instances.
 * This allows us to use drag'n'drop from a Fabricator grid to one that actually holds
 * supplier fabricator assignments. The communication with the different API is possible
 * by using the AssignedFabricatorJson readers and writers, which 'translate' between the
 * data format of supplier fabricator assignments and (customised) fabricator.
 */
Ext.define('Shopware.apps.ViisonPickwareERPSupplierManagement.store.AssignedFabricator', {

    extend: 'Shopware.apps.ViisonPickwareERPSupplierManagement.store.Fabricator',
    proxy: {
        type: 'ajax',
        batchActions: true,
        api: {
            read: ViisonCommonApp.assembleBackendUrl(
                'ViisonPickwareERPSupplierManagement/getSupplierFabricatorList'
            ),
            create: ViisonCommonApp.assembleBackendUrl(
                'ViisonPickwareERPSupplierManagement/createSupplierFabricators'
            ),
            update: ViisonCommonApp.assembleBackendUrl(
                'ViisonPickwareERPSupplierManagement/updateSupplierFabricators'
            ),
            destroy: ViisonCommonApp.assembleBackendUrl(
                'ViisonPickwareERPSupplierManagement/deleteSupplierFabricators'
            ),
        },
        reader: Ext.create('Shopware.apps.ViisonPickwareERPSupplierManagement.reader.AssignedFabricatorJson', {}),
        writer: Ext.create('Shopware.apps.ViisonPickwareERPSupplierManagement.writer.AssignedFabricatorJson', {}),
    },

});
