// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPSupplierOrders.controller.Detail', {

    extend: 'Ext.app.Controller',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_supplier_orders/main',

    /**
     * @Override
     */
    init: function () {
        this.control({
            'viison_pickware_erp_supplier_orders-detail-configuration': {
                configurationPanelReady: this.onUpdateArticleList,
                sourceWarehouseSelectionChanged: this.onUpdateArticleList,
                considerOpenOrdersValueChanged: this.onUpdateArticleList,
                useTotalIncomingStockValueChanged: this.onUpdateArticleList,
            },
            'viison_pickware_erp_supplier_orders-detail-article_assignment': {
                moveItemsFromLeftToRight: this.onMoveArticleDetailsFromLeftToRight,
                moveItemsFromLeftToRightViaDragDrop: this.onMoveArticleDetailsFromLeftToRight,
                moveItemsFromRightToLeft: this.onMoveArticleDetailsFromRightToLeft,
                moveItemsFromRightToLeftViaDragDrop: this.onMoveArticleDetailsFromRightToLeft,
                supplierFilterChanged: this.onSupplierFilterChanged,
                leftPanelSearchFieldChanged: this.onLeftPanelSearchFieldChanged,
                rightPanelSearchFieldChanged: this.onRightPanelSearchFieldChanged,
                showArticleDetail: this.onShowArticleDetail,
                removeArticleDetail: this.onRemoveArticleDetail,
                saveArticleDetailForBarcodeLabelPrinting: this.onSaveArticleDetailForBarcodeLabelPrinting,
                assignReorderArticles: this.onAssignReorderArticles,
                assignUrgentReorderArticles: this.onAssignUrgentReorderArticles,
                createSingleOrder: this.onCreateSingleOrder,
                createAllOrders: this.onCreateAllOrders,
                rightPanelRemoveArticles: this.onRightPanelRemoveArticles,
                rightPanelAddArticleLabels: this.onRightPanelAddArticleLabels,
                rightPanelAssignSupplier: this.onRightPanelAssignSupplier,
                rightPanelSelectionChange: this.onRightPanelSelectionChange,
                articleRowEdited: this.onArticleRowEdited,
                cancelEditing: this.onCancelEditing,
                copyOrder: this.onCopyOrder,
                saveOrder: this.onSaveOrder,
            },
            'viison_pickware_erp_supplier_orders-detail-attachments': {
                downloadAttachment: this.onDownloadAttachment,
                deleteAttachment: this.onDeleteAttachment,
                uploadAttachment: this.onUploadAttachment,
                attachmentUploaded: this.onAttachmentUploaded,
            },
        });

        // Create and load a shared supplier store
        this.supplierStore = Ext.create('Shopware.apps.ViisonPickwareERPSupplierManagement.store.Supplier', {
            pageSize: 1000,
        });
        this.supplierStore.load();

        // Create and load a shared item status store
        this.itemStatusStore = this.getStore('order.ArticleStatus');
        this.itemStatusStore.load();

        this.callParent(arguments);
    },

    /**
     * Creates a new edit window.
     *
     * @param {Shopware.apps.ViisonPickwareERPSupplierOrders.model.Order} record (optional)
     * @param {function} callback
     * @return {Shopware.apps.ViisonPickwareERPSupplierOrders.view.Edit}
     */
    createEditWindow: function (record, callback) {
        // Create the left (available) article store
        var leftArticleStore = Ext.create('Shopware.apps.ViisonPickwareERPSupplierOrders.store.ArticleDetail', {
            sorters: [
                { property: 'reorder', direction: 'DESC' },
            ],
        });
        if (record) {
            // Filter out all articles, that are assigned to the order
            leftArticleStore.filters.add('orderBlacklist', Ext.create('Ext.util.Filter', {
                property: 'orderIdBlacklist',
                value: record.get('id'),
            }));
        } else {
            // Filter out all articles, that are assigned, but not part of an order
            leftArticleStore.filters.add('notPreAssignedAsOrderArticle', Ext.create('Ext.util.Filter', {
                property: 'preAssignedOrderArticlesBlacklist',
            }));
        }

        // Create and load the right (assigned) article store
        var rightArticleStore = Ext.create('Shopware.apps.ViisonPickwareERPSupplierOrders.store.AssignedArticleDetail', {});
        if (record) {
            // Filter by order ID
            rightArticleStore.filters.add('preAssignedOrderArticles', Ext.create('Ext.util.Filter', {
                property: 'orderArticle.supplierOrderId',
                value: record.get('id'),
            }));
            rightArticleStore.on('load', function (store) {
                // Group the results
                store.group('supplierId');
                store.sort();
            });
        } else {
            // Filter out all articles, that are part of an order
            rightArticleStore.filters.add('preAssignedOrderArticles', Ext.create('Ext.util.Filter', {
                property: 'orderArticle.supplierOrderId',
                expression: 'IS NULL',
            }));
            rightArticleStore.on('load', function (store) {
                // Group the results
                store.group('supplierId');
                store.sort();
            });
        }

        // Create the edit window
        var title;
        if (record) {
            title = this.getViisonSnippet('edit/window/title/edit') + ' - #' + record.get('orderNumber');
        } else {
            title = this.getViisonSnippet('edit/window/title/add');
        }
        this.supplierStore.load(function () {
            this.getView((record) ? 'DetailEdit' : 'DetailCreate').create({
                title: title,
                record: record,
                supplierStore: this.supplierStore,
                itemStatusStore: this.itemStatusStore,
                leftArticleStore: leftArticleStore,
                rightArticleStore: rightArticleStore.load(),
            });
            if (typeof callback === 'function') {
                callback();
            }
        }.bind(this));
    },

    /**
     * Replaces the just added article details records with a customised instance
     * that contains more data fields. Furthermore, if the given panel has an order
     * record, one new order item is added to the order for each moved article detail.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierOrders.view.detail.ArticleAssignment panel
     * @param Shopware.apps.ViisonPickwareERPSupplierCommon.model.ArticleDetail[] records
     */
    onMoveArticleDetailsFromLeftToRight: function (panel, records) {
        if (panel.record) {
            Ext.each(records, function (assignedRecord) {
                // Try to apply the supplier association data
                var supplierId = panel.record.getSupplier().get('id');
                assignedRecord.set('supplierId', supplierId);
                assignedRecord.applySupplierWithId(supplierId);
                assignedRecord.commit();

                // Add the article to the order. Check if the same article is already part of the store. If so, replace
                // it with this new entity. We had concurrency issues when an order is edited simultaneously. With this
                // check no article detail can be added to the same order multiple times.
                var existingOrderArticle = panel.record.items().findRecord('articleDetailId', assignedRecord.get('id'));
                if (existingOrderArticle) {
                    panel.record.items().remove(existingOrderArticle);
                }
                var orderArticle = this.createOrderArticle(assignedRecord);
                panel.record.items().add(orderArticle);
                panel.record.setDirty(true);
                orderArticle.phantom = true;
            }, this);

            // Reload available articles
            this.updateBlacklistFilter(panel.leftStore, panel.rightStore, 'id', 'articleDetail.id');
            panel.leftStore.loadPage(panel.leftStore.currentPage);
        } else {
            Ext.each(records, function (assignedRecord) {
                assignedRecord.set('orderedQuantity', Math.max(1, assignedRecord.get('suggestedReorderQuantity')));

                // Use the currently selected supplier or the default supplier if none is selected
                var supplierCombobox = panel.leftPanel.down('#supplierSelection');
                if (supplierCombobox && supplierCombobox.getValue()) {
                    assignedRecord.applySupplierWithId(supplierCombobox.getValue());
                } else {
                    assignedRecord.applyDefaultSupplier();
                }

                assignedRecord.commit();
                assignedRecord.phantom = true;
            }, this);

            // Sync changes
            this.syncAssignmentStore(
                panel,
                this.getViisonSnippet('notification/error/message/assign_articles'),
                function () {
                    panel.rightStore.group('supplierId'); // XXX This grouper has already been added by the rightStore.onLoad() handler...?
                    panel.rightStore.sort();
                    panel.leftStore.loadPage(panel.leftStore.currentPage);
                }
            );
        }
    },

    onUpdateArticleList: function (configurationPanel) {
        var articleAssignmentPanel = configurationPanel.up('window').down('viison_pickware_erp_supplier_orders-detail-article_assignment');
        var articleListStore = articleAssignmentPanel.leftStore;

        this.applyConfigurationToArticleStore(articleListStore, configurationPanel);
        articleListStore.loadPage(1);
    },

    /**
     * Updates the filter on the available article store (left) and reloads it.
     * If the given panel has an order record, the order items corresponding
     * to the moved article details are removed from the order.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierOrders.view.detail.ArticleAssignment panel
     * @param Shopware.apps.ViisonPickwareERPSupplierCommon.model.ArticleDetail[] records
     */
    onMoveArticleDetailsFromRightToLeft: function (panel, records) {
        if (panel.record) {
            // Remove the articles from the order
            var orderArticles = panel.record.items();
            Ext.each(records, function (articleRecord) {
                var orderArticleIndex = orderArticles.findExact('articleDetailId', articleRecord.get('id'));
                if (orderArticleIndex > -1) {
                    orderArticles.removeAt(orderArticleIndex);
                }
            });
            panel.record.setDirty(true);

            // Reload available articles
            this.updateBlacklistFilter(panel.leftStore, panel.rightStore, 'id', 'articleDetail.id');
            panel.leftStore.loadPage(panel.leftStore.currentPage);
        } else {
            // Sync changes
            this.syncAssignmentStore(
                panel,
                this.getViisonSnippet('notification/error/message/unassign_articles'),
                function () {
                    panel.leftStore.loadPage(panel.leftStore.currentPage);
                }
            );
        }
    },

    /**
     * Sets the ID of the given record (or null) as an extra parameter to the given store's proxy
     * and reloads the store.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierManagement.model.Supplier selectedSupplierRecord
     * @param Ext.data.Store store
     */
    onSupplierFilterChanged: function (selectedSupplierRecord, store) {
        store.getProxy().extraParams.supplierId = (selectedSupplierRecord) ? selectedSupplierRecord.get('id') : null;
        store.loadPage(1);
    },

    /**
     * Sets the given query as an extra parameter to the given store's proxy
     * and reloads the store.
     *
     * @param string query
     * @param Ext.data.Store store
     */
    onLeftPanelSearchFieldChanged: function (query, store) {
        store.getProxy().extraParams.query = query;
        store.loadPage(1);
    },

    /**
     * @param {Shopware.apps.ViisonPickwareERPSupplierOrders.view.detail.ArticleAssignment} panel
     * @param {string} searchQuery
     */
    onRightPanelSearchFieldChanged: function (panel, searchQuery) {
        panel.rightStore.getProxy().extraParams.query = searchQuery;
        panel.rightStore.load();
    },

    /**
     * Launches a new article sub application showing the details of the
     * given article record.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierCommon.model.ArticleDetail record
     */
    onShowArticleDetail: function (record) {
        Shopware.app.Application.addSubApplication({
            name: 'Shopware.apps.Article',
            action: 'detail',
            params: {
                articleId: record.get('articleId'),
            },
        });
    },

    /**
     * Removes the given record from the assigned articles store (right) and
     * updates the filter on the available article store (left) and reloads it.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierOrders.view.detail.ArticleAssignment panel
     * @param Shopware.apps.ViisonPickwareERPSupplierCommon.model.ArticleDetail record
     */
    onRemoveArticleDetail: function (panel, record) {
        panel.rightStore.remove(record);
        this.onMoveArticleDetailsFromRightToLeft(panel, [record]);
    },

    /**
     * Pre-selects the selected record for the (article) barcode label printing.
     *
     * @param {Shopware.apps.ViisonPickwareERPSupplierOrders.view.detail.ArticleAssignment} panel
     * @param {Shopware.apps.ViisonPickwareERPSupplierOrders.model.ArticleDetail} record
     */
    onSaveArticleDetailForBarcodeLabelPrinting: function (panel, record) {
        this.addToBarcodeLabelPrintQueue(panel, [record]);
    },

    /**
     * Creates a new article store to load a list of up to 10,000 articles that need
     * new stock and that are not already assigned to the right list. Finally, the new
     * articles are added to the assigned list (right) and the left list is reloaded.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierOrders.view.detail.ArticleAssignment panel
     */
    onAssignReorderArticles: function (panel) {
        var configurationPanel = panel.up('window').down('viison_pickware_erp_supplier_orders-detail-configuration');

        // Load all articles that need new stock but are not already added to the assigned articles,
        // while respecting the supplier filter of the unassigned (left) panel
        panel.setLoading(true);
        var tmpStore = Ext.create('Shopware.apps.ViisonPickwareERPSupplierOrders.store.ArticleDetail', {
            pageSize: 10000,
        });
        tmpStore.filters.add('notPreAssignedAsOrderArticle', Ext.create('Ext.util.Filter', {
            property: 'preAssignedOrderArticlesBlacklist',
        }));

        tmpStore.getProxy().extraParams.filterReorder = 1;
        tmpStore.getProxy().extraParams.supplierId = panel.leftStore.getProxy().extraParams.supplierId;
        this.applyConfigurationToArticleStore(tmpStore, configurationPanel);
        tmpStore.load({
            callback: function (records) {
                panel.setLoading(false);
                // Add all responded articles to the assigned (right) store
                panel.rightStore.add(records);
                this.onMoveArticleDetailsFromLeftToRight(panel, records);
            },
            scope: this,
        });
    },

    onAssignUrgentReorderArticles: function (panel) {
        var configurationPanel = panel.up('window').down('viison_pickware_erp_supplier_orders-detail-configuration');

        // Load all articles that need new stock but are not already added to the assigned articles,
        // while respecting the supplier filter of the unassigned (left) panel
        panel.setLoading(true);
        var tmpStore = Ext.create('Shopware.apps.ViisonPickwareERPSupplierOrders.store.ArticleDetail', {
            pageSize: 10000,
        });
        tmpStore.filters.add('notPreAssignedAsOrderArticle', Ext.create('Ext.util.Filter', {
            property: 'preAssignedOrderArticlesBlacklist',
        }));
        tmpStore.getProxy().extraParams.filterReorder = 2;
        tmpStore.getProxy().extraParams.supplierId = panel.leftStore.getProxy().extraParams.supplierId;
        this.applyConfigurationToArticleStore(tmpStore, configurationPanel);
        tmpStore.load({
            callback: function (records) {
                panel.setLoading(false);
                // Add all responded articles to the assigned (right) store
                panel.rightStore.add(records);
                this.onMoveArticleDetailsFromLeftToRight(panel, records);
            },
            scope: this,
        });
    },

    /**
     * Creates a new order for the supplier with the given ID, containing the given article
     * records and syncs it to the server. Once the order is created, it is opened in a new edit
     * window.
     *
     * @param {Shopware.apps.ViisonPickwareERPSupplierOrders.view.detail.ArticleAssignment} panel
     * @param {Shopware.apps.ViisonPickwareERPSupplierOrders.model.ArticleDetail[]} articleDetails
     * @param {int} supplierId
     */
    onCreateSingleOrder: function (panel, articleDetails, supplierId) {
        // Check records for validity
        if (!this.validateArticleDetailRecords(panel, articleDetails)) {
            return;
        }
        this.createOrder(articleDetails, supplierId, this.getSelectedTargetWarehouseId(panel));

        panel.setLoading(true);
        this.saveOrders(articleDetails, function (success, newOrders) {
            this.openCreatedOrder(success, newOrders, this);
            // Remove the synced items from the assigned list
            panel.rightStore.remove(articleDetails);
            panel.rightStore.sort();

            // Reload the left list to add the products that ware just assigned to the created orders
            panel.leftStore.loadPage(panel.leftStore.currentPage);
            panel.setLoading(false);
        }.bind(this));
    },

    /**
     * Creates a new order for each of the existing supplier groups in the assigned article list
     * and syncs them to the server.
     *
     * @param {Shopware.apps.ViisonPickwareERPSupplierOrders.view.detail.ArticleAssignment} panel
     */
    onCreateAllOrders: function (panel) {
        // Get the records from all valid groups
        var allArticleDetails = [];
        Ext.each(panel.rightStore.getGroups(), function (group) {
            if (group.name) {
                allArticleDetails = allArticleDetails.concat(group.children);
            }
        });

        // Check records for validity
        if (!this.validateArticleDetailRecords(panel, allArticleDetails)) {
            return;
        }

        Ext.each(panel.rightStore.getGroups(), function (group) {
            if (group.name) {
                this.createOrder(group.children, group.name, this.getSelectedTargetWarehouseId(panel));
            }
        }, this);

        panel.setLoading(true);
        this.saveOrders(allArticleDetails, function () {
            // Remove the synced items from the assigned list
            panel.rightStore.remove(allArticleDetails);
            panel.rightStore.sort();

            // Reload the left list to add the products that ware just assigned to the created orders
            panel.leftStore.loadPage(panel.leftStore.currentPage);
            panel.setLoading(false);
        });
    },

    /**
     * Triggers saving of the order
     *
     * @param {Shopware.apps.ViisonPickwareERPSupplierOrders.model.ArticleDetail[]} articleDetails
     * @param {function} callback
     */
    saveOrders: function (articleDetails, callback) {
        this.checkForMappingChanges(articleDetails, function (hasChanges) {
            if (hasChanges) {
                Ext.Msg.confirm(
                    this.getViisonSnippet('alert/title'),
                    this.getViisonSnippet('alert/message/save_article_detail_supplier_mapping'),
                    function (buttonId) {
                        this.createOrders(articleDetails, callback);
                        if (buttonId !== 'yes') {
                            return;
                        }
                        this.saveArticleMapping(articleDetails);
                    },
                    this
                );
            } else {
                this.createOrders(articleDetails, callback);
            }
        }.bind(this));
    },

    /**
     *
     * @param {Shopware.apps.ViisonPickwareERPSupplierOrders.model.ArticleDetail[]} articleDetails
     * @param {function} callback
     */
    createOrders: function (articleDetails, callback) {
        this.syncNewOrders(
            articleDetails,
            this.getViisonSnippet('notification/error/message/create_order'),
            callback
        );
    },

    /**
     * Converts records to json data and send request to check if changes happend to the supplier or price
     * Create orders after the check or the confirm dialog
     *
     * @param {Shopware.apps.ViisonPickwareERPSupplierOrders.model.ArticleDetail[]} articleDetails
     * @param {function} callback
     */
    checkForMappingChanges: function (articleDetails, callback) {
        var articleDetailsFormatted = [];
        articleDetails.forEach(function (record, index) {
            articleDetailsFormatted[index] = record.data;
        });
        Ext.Ajax.request({
            url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPSupplierOrders/checkArticleDetailsForChangedArticleDetailSupplierMappings'),
            method: 'POST',
            jsonData: articleDetailsFormatted,
            success: function (response) {
                var responseData = Ext.JSON.decode(response.responseText, true);
                if (!responseData.success) {
                    Shopware.Notification.createGrowlMessage(
                        this.getViisonSnippet('notification/save_supplier_mappings/failure/title'),
                        responseData.message,
                        'ViisonPickwareERPSupplierOrders'
                    );
                }
                callback(responseData.hasNewArticleDetailSupplierMappings);
            },
            scope: this,

        });
    },

    /**
     * Sends request that saves the ArticleDetailSupplierMapping
     *
     * @param {Shopware.apps.ViisonPickwareERPSupplierOrders.model.ArticleDetail[]} articleDetails
     */
    saveArticleMapping: function (articleDetails) {
        var articleDetailsFormatted = [];
        articleDetails.forEach(function (record, index) {
            articleDetailsFormatted[index] = record.data;
        });
        Ext.Ajax.request({
            url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPSupplierOrders/saveArticleDetailSupplierMappingsForArticleDetails'),
            method: 'POST',
            jsonData: articleDetailsFormatted,
            success: function (savingResponse) {
                var responseData = Ext.JSON.decode(savingResponse.responseText, true);
                if (responseData.success) {
                    Shopware.Notification.createGrowlMessage(
                        this.getViisonSnippet('notification/save_supplier_mappings/success/title'),
                        this.getViisonSnippet('notification/save_supplier_mappings/success/message'),
                        'ViisonPickwareERPSupplierOrders'
                    );
                } else {
                    Shopware.Notification.createGrowlMessage(
                        this.getViisonSnippet('notification/save_supplier_mappings/failure/title'),
                        responseData.message,
                        'ViisonPickwareERPSupplierOrders'
                    );
                }
            },
            scope: this,
        });
    },

    /**
     * Removes any selected records from the right grid panel and reloads both grids with the.
     *
     * @param {Shopware.apps.ViisonPickwareERPSupplierOrders.view.detail.ArticleAssignment} panel
     * @param {Ext.selection.Model} selectionModel
     */
    onRightPanelRemoveArticles: function (panel, selectionModel) {
        // If less than 2 records are selected, handle this action as 'select all'.
        if (selectionModel.getSelection().length < 2) {
            selectionModel.selectAll(true);
        }
        var records = selectionModel.getSelection();
        panel.rightStore.remove(records);

        this.onMoveArticleDetailsFromRightToLeft(panel, records);
    },

    /**
     * Pre-selects any selected records from the right grid panel for the barcode label printing.
     *
     * @param {Shopware.apps.ViisonPickwareERPSupplierOrders.view.detail.ArticleAssignment} panel
     * @param {Ext.selection.Model} selectionModel
     */
    onRightPanelAddArticleLabels: function (panel, selectionModel) {
        // If less than 2 records are selected, handle this action as 'select all'.
        if (selectionModel.getSelection().length < 2) {
            selectionModel.selectAll(true);
        }
        var records = selectionModel.getSelection();

        this.addToBarcodeLabelPrintQueue(panel, records);
        selectionModel.deselectAll();
        this.onRightPanelSelectionChange(panel.rightPanel, []);
    },

    /**
     * Pre-selects any selected records from the right grid panel for the barcode label printing.
     *
     * @param {Shopware.apps.ViisonPickwareERPSupplierOrders.view.detail.ArticleAssignment} panel
     * @param {Ext.grid.Panel} rightPanel
     */
    onRightPanelAssignSupplier: function (panel, rightPanel) {
        var selectionModel = rightPanel.getSelectionModel();
        if (!selectionModel || !selectionModel.getSelection().length) {
            return;
        }

        var supplierAlert = Ext.create('Shopware.apps.ViisonPickwareERPSupplierOrders.view.detail.SupplierAssignMessageBox', {
            supplierStore: this.supplierStore,
        });

        supplierAlert.show({
            title: this.getViisonSnippet('messagebox/assign_supplier/title'),
            msg: this.getViisonSnippet('messagebox/assign_supplier/message'),
            buttons: Ext.Msg.OKCANCEL,
            width: 300,
            scope: {
                parentPanel: rightPanel,
                selectedItems: selectionModel.getSelection(),
                supplierAlert: supplierAlert,
            },
            callback: function (buttonId) {
                if (buttonId === 'cancel' || !this.selectedItems || !this.selectedItems.length) {
                    return;
                }

                var supplierId = this.supplierAlert.supplierAssing.getValue();
                this.selectedItems.forEach(function (item) {
                    if (item.data) {
                        item.set('supplierId', supplierId);
                    }
                });

                this.parentPanel.store.sync();
            },
        });
    },

    /**
     * Sends a backend request to pre-select the given records for the barcode label printing.
     *
     * Uses their currently selected purchase quantity. Shows a growl message the request was sent.
     *
     * @param {Shopware.apps.ViisonPickwareERPSupplierOrders.view.detail.ArticleAssignment} panel
     * @param {Shopware.apps.ViisonPickwareERPSupplierOrders.model.ArticleDetail[]} records
     */
    addToBarcodeLabelPrintQueue: function (panel, records) {
        var articleNumbers = [];
        var labelQuantities = [];
        Ext.each(records, function (record) {
            if (record.get('orderedQuantity') > 0) {
                articleNumbers.push(record.get('orderNumber'));
                labelQuantities.push(record.get('orderedQuantity'));
            }
        });

        if (articleNumbers.length === 0) {
            Shopware.Msg.createGrowlMessage(
                this.getViisonSnippet('notification/error/title'),
                this.getViisonSnippet('notification/error/message/barcode_label_printing/no_selection'),
                'ViisonPickwareERPSupplierOrders'
            );

            return;
        }

        var data = {
            type: 'article',
            identifiers: articleNumbers,
            quantities: labelQuantities,
        };

        panel.setLoading(true);
        Ext.Ajax.request({
            method: 'POST',
            url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPBarcodeLabelPrinting/enqueueBarcodeLabel'),
            jsonData: data,
            scope: this,
            callback: function (options, success, response) {
                if (success && response && response.responseText) {
                    try {
                        var responseData = JSON.parse(response.responseText);
                        success = responseData.success;
                    } catch (exception) {
                        success = false;
                    }
                } else {
                    success = false;
                }

                var title;
                var message;
                if (success) {
                    title = this.getViisonSnippet('notification/success/title');
                    message = this.getViisonSnippet('notification/error/message/barcode_label_printing/success');
                } else {
                    title = this.getViisonSnippet('notification/error/title');
                    message = this.getViisonSnippet('notification/error/message/barcode_label_printing/error');
                }

                // Display a notification
                Shopware.Msg.createGrowlMessage(title, message, 'ViisonPickwareERPSupplierOrders');
                panel.setLoading(false);
            },
        });
    },

    /**
     * Updates the action buttons captions depending on whether or not multiple rows are selected.
     *
     * @param {Shopware.apps.ViisonPickwareERPSupplierOrders.view.detail.ArticleAssignment} panel
     * @param {Shopware.apps.ViisonPickwareERPSupplierOrders.model.ArticleDetail[]} selection
     */
    onRightPanelSelectionChange: function (panel, selection) {
        var gridTopToolbar = panel.view.panel.dockedItems.get('topToolbar');
        var deleteButtonCaption;
        var labelButtonCaption;
        var assignSupplierButton = gridTopToolbar.items.get('assignSupplierButton');

        if (selection.length > 1) {
            // Since we are not using checkbox selection, but simple row-clicking a single selection is not easily
            // spotted. For the one-line use case exists a button in the action column. Hence, only switch to 'use
            // selected rows' when selecting 2+ rows.
            deleteButtonCaption = this.getViisonSnippet('list/toolbar/button/delete_articles/selected');
            labelButtonCaption = this.getViisonSnippet('list/toolbar/button/save_for_barcode_label_printing/selected');
        } else {
            deleteButtonCaption = this.getViisonSnippet('list/toolbar/button/delete_articles/all');
            labelButtonCaption = this.getViisonSnippet('list/toolbar/button/save_for_barcode_label_printing/all');
        }

        // Activate the supplier assign button if min one Article is selected
        if (assignSupplierButton) {
            assignSupplierButton.setDisabled(selection.length < 1);
        }

        gridTopToolbar.items.get('deleteButton').setText(deleteButtonCaption);
        gridTopToolbar.items.get('barcodeLabelButton').setText(labelButtonCaption);
    },

    /**
     * Updates the order article record that corresponds with the article record
     * that was just edited.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierOrders.view.detail.ArticleAssignment panel
     * @param Shopware.apps.ViisonPickwareERPSupplierOrders.model.ArticleDetail record
     */
    onArticleRowEdited: function (panel, record) {
        if (panel.record) {
            // Update the respective order article
            var orderArticleIndex = panel.record.items().findExact('articleDetailId', record.get('id'));
            if (orderArticleIndex > -1) {
                var orderArticle = panel.record.items().getAt(orderArticleIndex);
                orderArticle.set('orderedQuantity', record.get('orderedQuantity'));
                orderArticle.set('deliveredQuantity', record.get('deliveredQuantity'));
                orderArticle.set('price', record.get('price'));
                orderArticle.set('statusId', record.get('statusId'));
            }
        } else {
            // Sync changes
            record.setDirty(true);
            this.syncAssignmentStore(
                panel,
                this.getViisonSnippet('notification/error/message/edit_assigned_article')
            );
        }
    },

    /**
     * Rejects the changes made to the order record and its articles and closes the edit window
     * that contains the panel, which fired the event.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierOrders.view.detail.ArticleAssignment panel
     */
    onCancelEditing: function (panel) {
        panel.record.reject();
        panel.record.items().rejectChanges();
        panel.record.setDirty(false);
        panel.up('viison_pickware_erp_supplier_orders-edit').close();
    },

    /**
     * Creates a new order containing all articles that are also contained in the order
     * associated with the given panel. The new order articles are cloned, but the
     * 'deliveredQuantity', 'deliveryTime' and 'statusId' fields are reset. Furthermore
     * the comment and document comment of the order are copied too. Finally the new order
     * is synced to the server and opened in a new window.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierOrders.view.detail.ArticleAssignment panel
     */
    onCopyOrder: function (panel) {
        // Collect all articles
        var articleDetails = [];
        panel.rightStore.each(function (articleDetail) {
            articleDetails.push(articleDetail);
        });

        // Create a new order record using the information of the panel record
        var order = this.createOrder(
            articleDetails,
            panel.record.getSupplier().get('id'),
            this.getSelectedTargetWarehouseId(panel)
        );
        order.set('comment', panel.record.get('comment'));
        order.set('documentComment', panel.record.get('documentComment'));

        order.items().each(function (orderArticle) {
            orderArticle.set('deliveredQuantity', 0);
            orderArticle.set('deliveryTime', 0);
            orderArticle.set('statusId', 0);
        });

        // At this point, the user may have modified the original order before clicking "copy order". We need to reset
        // these modifications to prevent the changes to the original from being synced without the user having clicked
        // save on the original order.
        panel.record.reject(true);
        // Also close the original order so the user interaction switches to the saved copy.
        panel.up('viison_pickware_erp_supplier_orders-edit').close();

        // Sync the new order
        this.syncNewOrders(
            panel,
            [],
            this.getViisonSnippet('notification/error/message/create_order'),
            function (success, newOrders) {
                this.openCreatedOrder(success, newOrders, this);
            }.bind(this)
        );
    },

    /**
     * Updates the order record that is associated with the given panel using the
     * panel's document form and syncs the changes.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierOrders.view.detail.ArticleAssignment panel
     */
    onSaveOrder: function (panel) {
        // Check records for validity
        var articleRecords = [];
        panel.rightStore.each(function (articleRecord) {
            articleRecords.push(articleRecord);
        });
        if (!this.validateArticleDetailRecords(panel, articleRecords)) {
            return;
        }

        // Update destination warehouse
        panel.record.set('warehouseId', this.getSelectedTargetWarehouseId(panel));

        // Update the record with the edited comments
        var commentForm = panel.rightPanel.nextSibling('form');
        commentForm.getForm().updateRecord(panel.record);
        panel.record.setDirty(true);

        // Sync the changes
        var editWindow = panel.up('viison_pickware_erp_supplier_orders-edit');
        editWindow.setLoading(true);
        var mainController = this.getController('Main');
        mainController.syncOrders(
            this.getViisonSnippet('notification/error/message/save'),
            function (success) {
                editWindow.setLoading(false);
                if (success) {
                    mainController.orderStore.load();
                    editWindow.close();
                }
            }
        );
    },

    /**
     * Opens a new tab triggering a download of the selected attachment.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierOrders.model.order.Attachment record
     */
    onDownloadAttachment: function (record) {
        var url = ViisonCommonApp.assembleBackendUrl('mediaManager/download', {
            mediaId: record.get('mediaId'),
        });
        window.open(url, '_blank');
    },

    /**
     * Deletes the given attachment record from the order record associated with the given panel
     * and syncs the changes.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierOrders.view.edit.Attachments panel
     * @param Shopware.apps.ViisonPickwareERPSupplierOrders.model.order.Attachment record
     */
    onDeleteAttachment: function (panel, record) {
        // Remove the respective attachment from the order and save the changes
        panel.record.attachments().remove(record);
        panel.record.setDirty(true);
        panel.setLoading(true);
        this.getController('Main').syncOrders(
            this.getViisonSnippet('notification/error/message/delete_attachment'),
            function () {
                panel.setLoading(false);
            }
        );
    },

    /**
     * Fetches the files selected in the file field and and simulates
     * dropping them in the drag'n'drop upload area.
     *
     * @param Ext.form.field.File field
     */
    onUploadAttachment: function (field) {
        var dropZone = field.nextSibling('html5fileupload');
        var fileField = field.getEl().down('input[type=file]').dom;
        dropZone.iterateFiles(fileField.files);
    },

    /**
     * Checks the response of the upload operation for success and, in case of success,
     * adds the uploaded media item as a new attachment to the order, which is then
     * synced.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierOrders.view.edit.Attachments panel
     * @param XMLHttpRequest operation
     */
    onAttachmentUploaded: function (panel, operation) {
        var response = Ext.decode(operation.responseText);
        if (!response.success) {
            Shopware.Notification.createGrowlMessage(
                this.getViisonSnippet('notification/error/title'),
                this.getViisonSnippet('notification/error/message/upload_attachment'),
                'ViisonPickwareERPSupplierOrders'
            );

            return;
        }

        // Add the new attachment to the order
        panel.record.attachments().add(
            Ext.create('Shopware.apps.ViisonPickwareERPSupplierOrders.model.order.Attachment', {
                mediaId: response.data.id,
                filename: (response.data.name + '.' + response.data.extension),
                date: new Date(),
            })
        );
        panel.record.setDirty(true);

        // Save the changes
        panel.setLoading(true);
        this.getController('Main').syncOrders(
            this.getViisonSnippet('notification/error/message/upload_attachment'),
            function () {
                panel.setLoading(false);
            }
        );
    },

    /**
     * Checks all given records for validity. That is, whether the orderedQuantity is
     * greater than zero. If any of the records is invalid,
     * its respective row editor is enabled.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierOrders.view.detail.ArticleAssignment panel
     * @param Shopware.apps.ViisonPickwareERPSupplierOrders.model.ArticleDetail[]
     * @return boolean
     */
    validateArticleDetailRecords: function (panel, records) {
        var valid = true;
        Ext.each(records, function (articleDetail) {
            if (articleDetail.get('orderedQuantity') <= 0) {
                valid = false;
                // Start editing the respective row
                var index = panel.rightStore.indexOf(articleDetail);
                panel.rowEditor.startEdit(index, 0);

                return false;
            }

            return undefined;
        });

        return valid;
    },

    /**
     * Creates and returns a new order that is mapped to the supplier with the given ID
     * and contains all given article details as items.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierOrders.model.ArticleDetail articleDetails
     * @param int supplierId
     * @param int warehouseId
     * @return Shopware.apps.ViisonPickwareERPSupplierOrders.model.Order
     */
    createOrder: function (articleDetails, supplierId, warehouseId) {
        // Create a new order and add the supplier and all given article details as items
        var order = Ext.create('Shopware.apps.ViisonPickwareERPSupplierOrders.model.Order', {});
        order.setSupplier(this.supplierStore.getById(supplierId));
        order.set('warehouseId', warehouseId);
        Ext.each(articleDetails, function (articleRecord) {
            var orderArticle = this.createOrderArticle(articleRecord);
            order.items().add(orderArticle);
        }, this);
        this.getController('Main').orderStore.add(order);

        return order;
    },

    /**
     * Creates a new order article record using the data of the given article detail record.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierOrders.model.ArticleDetail article
     * @return Shopware.apps.ViisonPickwareERPSupplierOrders.model.order.Article
     */
    createOrderArticle: function (article) {
        var data = article.getData();
        data.articleDetailId = data.id;
        data.id = null;
        delete data.statusId;

        return Ext.create('Shopware.apps.ViisonPickwareERPSupplierOrders.model.order.Article', data);
    },

    /**
     * Sets the assignment panel loading and starts syncing the orders. Once the sync completed
     * the panel is set to not loading and all order items that have been synced are removed from
     * the assigned list (right).
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierOrders.view.detail.ArticleAssignment panel
     * @param Shopware.apps.ViisonPickwareERPSupplierOrders.model.ArticleDetail[] orderItems
     * @param string errorMessage
     * @param function callback (optional)
     */
    syncNewOrders: function (orderItems, errorMessage, callback) {
        // Save the new orders on the server
        var orderStore = this.getController('Main').orderStore;
        this.getController('Main').syncOrders(errorMessage, function (success, newOrders) {
            if (success) {
                // Reload the main store
                orderStore.loadPage(1, {
                    callback: function () {
                        if (Ext.isFunction(callback)) {
                            callback(true, newOrders);
                        }
                    },
                });
            } else if (Ext.isFunction(callback)) {
                callback(false);
            }
        });
    },

    /**
     * Opens a new edit window for the order that was created by the sync operation
     * that triggers this function.
     *
     * @param boolean success
     * @param Shopware.apps.ViisonPickwareERPSupplierOrders.model.Order[] createdOrders
     * @param Shopware.apps.ViisonPickwareERPSupplierOrders.controller.Edit editController
     */
    openCreatedOrder: function (success, createdOrders, editController) {
        if (success && createdOrders.length === 1) {
            // Open the new order in an edit window
            // NOTE: We can neither use the 'order' created above, nor the order contained in 'createdOrders'
            //       when editing. This is due the two problems: Firstly the created 'order' is not updated
            //       by the store, hence it does not have an ID. Secondly the order container in 'createdOrders'
            //       is somehow copied to the store and hence does not trigger any syncs in the store.
            //       The solution here is to use the ID of the order contained in 'createdOrders' to fetch
            //       'the same' record from the store.
            var newOrder = editController.getController('Main').orderStore.getById(createdOrders[0].get('id'));
            editController.createEditWindow(newOrder);
        }
    },

    /**
     * Syncs the right (assigned) store of the given panel.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierCommon.view.AssignmentPanel panel
     * @param string errorMessage (optional)
     * @param function callback (optional)
     */
    syncAssignmentStore: function (panel, errorMessage, callback) {
        // Check the store for any changes
        var store = panel.rightStore;
        var numChanges = store.getNewRecords().length + store.getModifiedRecords().length + store.getRemovedRecords().length;
        if (numChanges === 0) {
            if (Ext.isFunction(callback)) {
                callback(true);
            }

            return;
        }

        panel.rightPanel.setLoading(true);
        store.sync({
            scope: this,
            success: function () {
                panel.rightPanel.setLoading(false);
                if (Ext.isFunction(callback)) {
                    callback(true);
                }
            },
            failure: function () {
                panel.rightPanel.setLoading(false);
                // Revert any changes like added, updated or removed records
                store.rejectChanges();
                // Display an error notification
                var message = errorMessage || '';
                Shopware.Notification.createGrowlMessage(
                    this.getViisonSnippet('notification/error/title'),
                    message,
                    'ViisonPickwareERPSupplierOrders'
                );
                if (Ext.isFunction(callback)) {
                    callback(false);
                }
            },
        });
    },

    /**
     * Creates a blacklist of the IDs extracted from the given items using the 'idKey'
     * and applies it as a filter to the given store.
     *
     * @param Ext.data.Store store
     * @param Ext.data.Store items
     * @param string idKey
     * @param string remoteIdKey
     */
    updateBlacklistFilter: function (store, items, idKey, remoteIdKey) {
        if (items.count() > 0) {
            var blacklist = [];
            items.each(function (item) {
                blacklist.push(item.get(idKey));
            });
            // Set the 'blacklist' filter using the given blacklist
            store.filters.add('blacklist', Ext.create('Ext.util.Filter', {
                property: remoteIdKey,
                expression: 'NOT IN',
                value: blacklist,
            }));
        } else {
            // Remove the blacklist filter because Doctrine has problems evaluating
            // a 'NOT IN' filter on an empty array
            store.filters.removeAtKey('blacklist');
        }
    },

    applyConfigurationToArticleStore: function (store, configurationPanel) {
        var configurationValues = configurationPanel.getConfigurationValues();

        if (configurationValues.sourceWarehouses.length > 0) {
            store.filters.add('selectedWarehouseIds', Ext.create('Ext.util.Filter', {
                property: 'articleDetailStockCounts.warehouseId',
                value: configurationValues.sourceWarehouses,
            }));
        } else {
            store.filters.removeAtKey('selectedWarehouseIds');
        }

        store.getProxy().extraParams.considerOpenOrders = configurationValues.considerOpenOrders ? 1 : 0;
        store.getProxy().extraParams.useTotalIncomingStock = configurationValues.useTotalIncomingStock ? 1 : 0;
    },

    getSelectedTargetWarehouseId: function (panel) {
        var configurationPanel = panel.up('window').down('viison_pickware_erp_supplier_orders-detail-configuration');

        return configurationPanel.getSelectedTargetWarehouseId();
    },

});
