// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPSupplierOrders.controller.Main', {

    extend: 'Ext.app.Controller',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_supplier_orders/main',

    mainWindow: null,
    refs: [
        { ref: 'listView', selector: 'viison_pickware_erp_supplier_orders-main-list' },
    ],

    /**
     * @Override
     */
    init: function () {
        this.control({
            'viison_pickware_erp_supplier_orders-main-list button[action=addOrder]': {
                click: this.onAddOrder,
            },
            'viison_pickware_erp_supplier_orders-main-list': {
                saveOrder: this.onSaveOrder,
                editOrder: this.onEditOrder,
                deleteOrder: this.onDeleteOrder,
                sendOrderMail: this.onSendOrderMail,
                downloadOrderPDF: this.onDownloadOrderPDF,
                downloadOrderCSV: this.onDownloadOrderCSV,
                searchFieldChanged: this.onSearchFieldChanged,
            },
            'viison_common_mail_composer-window viison_common_mail_composer-main': {
                sendMail: this.onSendMail,
            },
            'viison_pickware_erp_supplier_orders-main-sidebar': {
                applyFilters: this.onApplyFilters,
                resetFilters: this.onResetFilters,
            },
        });

        // Create the main order store
        this.orderStore = Ext.create('Shopware.apps.ViisonPickwareERPSupplierOrders.store.Order', {});

        // Create and load a shared warehouse store
        this.warehouseStore = Ext.create('Shopware.apps.ViisonPickwareERPWarehouseManagement.store.Warehouse', {});
        this.warehouseStore.load();

        // Create and load a shared supplier store
        this.supplierStore = Ext.create('Shopware.apps.ViisonPickwareERPSupplierManagement.store.Supplier', {});
        this.supplierStore.load();

        // Check for a passed order ID
        if (this.subApplication && this.subApplication.params && Ext.isNumeric(this.subApplication.params.orderId)) {
            // Load only the order whose ID was passed to this app
            this.orderStore.filters.add('singleOrder', Ext.create('Ext.util.Filter', {
                property: 'supplierOrder.id',
                value: this.subApplication.params.orderId,
            }));
            this.orderStore.load({
                scope: this,
                callback: function (records) {
                    // Try to open the requested order in an edit window
                    this.orderStore.filters.removeAtKey('singleOrder');
                    if (records.length === 1) {
                        this.onEditOrder(records[0]);
                    } else {
                        Shopware.Notification.createGrowlMessage(
                            this.getViisonSnippet('notification/error/title'),
                            Ext.String.format(
                                this.getViisonSnippet('notification/error/message/order_not_found'),
                                this.subApplication.params.orderId
                            ),
                            'ViisonPickwareERPSupplierOrders'
                        );
                    }
                },
            });
        } else {
            // Just load the store and create the main window
            this.mainView = this.getView('Main').create({
                orderStore: this.orderStore,
                warehouseStore: this.warehouseStore,
                orderStatusStore: this.getStore('order.Status'),
                paymentStatusStore: this.getStore('order.PaymentStatus'),
                supplierStore: this.supplierStore,
            });

            this.orderStore.load();
        }

        // Listen on load events of the order status store
        this.getStore('order.Status').on('load', this.onOrderStatusStoreLoad, this);

        this.callParent(arguments);
    },

    /**
     * Opens an empty article assignment window.
     */
    onAddOrder: function () {
        this.mainView.setLoading(true);
        this.getController('Detail').createEditWindow(undefined, function () {
            this.mainView.setLoading(false);
        }.bind(this));
    },

    /**
     * Saves the changes in the given supplier order by syncing them to the server.
     * If the order status was changed to 'completely received', its delivery date
     * is set to today's date and the user is asked whether they want to mark the
     * order items as 'completely received', too.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierOrders.model.Order record
     * @param object originalValue
     */
    onSaveOrder: function (record, originalValues) {
        if (originalValues.statusId !== 4 && record.get('statusId') === 4) {
            // Set the delivery date to today
            record.set('deliveryDate', new Date());
            // Ask user about updating the order items
            Ext.Msg.confirm(
                this.getViisonSnippet('alert/title'),
                this.getViisonSnippet('alert/message/mark_items_as_completely_received'),
                function (buttonId) {
                    if (buttonId === 'yes') {
                        // Update the delivered quantity and status of all order items,
                        // whose status is open or 'partly received'
                        record.items().each(function (orderArticle) {
                            if (orderArticle.get('statusId') < 4) {
                                orderArticle.set('statusId', 4);
                                // Don't set the deliveredQuantity to the orderedQuantity for positions where the
                                // delivered quantity is higher than the ordered quantity to prevent negative bookings
                                if (orderArticle.get('deliveredQuantity') < orderArticle.get('orderedQuantity')) {
                                    orderArticle.set('deliveredQuantity', orderArticle.get('orderedQuantity'));
                                }
                            }
                        });
                    }

                    this.syncOrders(this.getViisonSnippet('notification/error/message/save'));
                },
                this
            );
        } else {
            this.syncOrders(this.getViisonSnippet('notification/error/message/save'));
        }
    },

    /**
     * Opens a new edit window using the given supplier order record.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierOrders.model.Order record
     */
    onEditOrder: function (record) {
        this.getController('Detail').createEditWindow(record);
    },

    /**
     * Ask the user for confirmation of the deletion and, if confirmed, deletes
     * the given record from the store and syncs the store afterwards.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierOrders.model.Order record
     */
    onDeleteOrder: function (record) {
        // Present a confirm alert before deleting the order
        Ext.Msg.confirm(
            this.getViisonSnippet('alert/title'),
            this.getViisonSnippet('alert/message/delete'),
            function (buttonId) {
                if (buttonId !== 'yes') {
                    return;
                }

                // Remove the given record from the store and sync the changes
                this.orderStore.remove(record);
                this.syncOrders(this.getViisonSnippet('notification/error/message/delete'));
            },
            this
        );
    },

    /**
     * Opens a new email editor window passing the pre-complied message and recipient info.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierOrders.model.Order record
     */
    onSendOrderMail: function (record) {
        // Load the email data
        this.setMainViewLoading(true);
        Ext.Ajax.request({
            url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPSupplierOrders/getOrderMail'),
            params: {
                orderId: record.get('id'),
            },
            success: function (response) {
                this.setMainViewLoading(false);
                var json = Ext.JSON.decode(response.responseText, true);
                if (!json || !json.data) {
                    Shopware.Notification.createGrowlMessage(
                        this.getViisonSnippet('notification/error/title'),
                        this.getViisonSnippet('notification/error/message/create_mail'),
                        'ViisonPickwareERPSupplierOrders'
                    );

                    return;
                }

                // Open the email composer window
                Ext.create('Shopware.apps.ViisonCommonMailComposer.view.Window', {
                    subApp: this.subApplication,
                    mail: json.data,
                    orderController: this,
                    orderRecord: record,
                }).show();
            },
            failure: function () {
                this.setMainViewLoading(false);
                Shopware.Notification.createGrowlMessage(
                    this.getViisonSnippet('notification/error/title'),
                    this.getViisonSnippet('notification/error/message/create_mail'),
                    'ViisonPickwareERPSupplierOrders'
                );
            },
            scope: this,
        });
    },

    /**
     * Opens a new tab showing the order PDF document.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierOrders.model.Order record
     */
    onDownloadOrderPDF: function (record) {
        var url = ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPSupplierOrders/downloadOrderPDF', {
            orderId: record.get('id'),
        });
        window.open(url, '_blank');
    },

    /**
     * Opens a new tab which downloads the order exported as CSV.
     *
     * @param Shopware.apps.ViisonPickwareERPSupplierOrders.model.Order record
     */
    onDownloadOrderCSV: function (record) {
        var url = ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPSupplierOrders/downloadOrderCSV', {
            orderId: record.get('id'),
        });
        window.open(url, '_blank');
    },

    /**
     * Sets the given query as an extra parameter to the order store's proxy
     * and reloads the store.
     *
     * @param string query
     */
    onSearchFieldChanged: function (query) {
        this.orderStore.getProxy().extraParams.query = query;
        this.orderStore.loadPage(1);
    },

    /**
     * Reconfigures the main order list to re-render it.
     */
    onOrderStatusStoreLoad: function () {
        if (this.getListView()) {
            this.getListView().reconfigure(this.orderStore);
        }
    },

    /**
     * Checks the given form for validity and, id valid, sends a new order mail
     * using the form data.
     *
     * @param Shopware.apps.ViisonCommonMailComposer.view.Main mailForm
     */
    onSendMail: function (mailForm) {
        if (!mailForm.getForm().isValid()) {
            return;
        }

        // Send the email using the mail form data
        var params = mailForm.getValues();
        var mailWindow = mailForm.up('viison_common_mail_composer-window');
        var orderRecord = mailWindow.orderRecord;
        params.orderId = orderRecord.get('id');
        params.isHtml = params.isHtml ? 1 : 0;
        mailWindow.setLoading(true);
        Ext.Ajax.request({
            url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPSupplierOrders/sendOrderMail'),
            params: params,
            scope: this,
            success: function (response) {
                mailWindow.setLoading(false);
                mailWindow.close();
                // Update the status of the order
                var json = Ext.JSON.decode(response.responseText, true);
                if (json && json.success === true && json.data && json.data.statusId) {
                    orderRecord.set('statusId', json.data.statusId);
                    orderRecord.commit();
                }
            },
            failure: function () {
                mailWindow.setLoading(false);
                Shopware.Notification.createGrowlMessage(
                    this.getViisonSnippet('notification/error/title'),
                    this.getViisonSnippet('notification/error/message/send_mail'),
                    'ViisonPickwareERPSupplierOrders'
                );
            },
        });
    },

    /**
     * Applies the filter options to the orderStore
     *
     * @param {Object} filterOptions
     */
    onApplyFilters: function (filterOptions) {
        this.orderStore.clearFilter(true);
        var filters = this.createFilters(filterOptions);
        this.orderStore.filter(filters);
    },

    /**
     * Resets the filter form and removes all filters from the orderStore
     *
     * @param {Ext.form.Panel} form
     */
    onResetFilters: function (form) {
        if (!form) {
            return;
        }
        form.getForm().reset();
        this.orderStore.clearFilter(true);
        this.orderStore.load();
    },

    /**
     * Creates a filter for each set filter option
     *
     * @param {Object} filterOptions
     * @returns {[Ext.util.Filter]}
     */
    createFilters: function (filterOptions) {
        var filter = [];
        if (filterOptions.createdFrom) {
            filter.push({
                property: 'supplierOrder.created',
                expression: '>=',
                value: Ext.Date.parse(filterOptions.createdFrom, 'd.m.Y'),
            });
        }
        if (filterOptions.createdTo) {
            var to = Ext.Date.add(Ext.Date.parse(filterOptions.createdTo, 'd.m.Y'), Ext.Date.DAY, 1);
            filter.push({
                property: 'supplierOrder.created',
                expression: '<',
                value: to,
            });
        }
        if (filterOptions.deliveryDateFrom) {
            filter.push({
                property: 'supplierOrder.deliveryDate',
                expression: '>=',
                value: Ext.Date.parse(filterOptions.deliveryDateFrom, 'd.m.Y'),
            });
        }
        if (filterOptions.deliveryDateTo) {
            var deliveryDateTo = Ext.Date.add(Ext.Date.parse(filterOptions.deliveryDateTo, 'd.m.Y'), Ext.Date.DAY, 1);
            filter.push({
                property: 'supplierOrder.deliveryDate',
                expression: '<',
                value: deliveryDateTo,
            });
        }
        if (filterOptions.paymentDueDateFrom) {
            filter.push({
                property: 'supplierOrder.paymentDueDate',
                expression: '>=',
                value: Ext.Date.parse(filterOptions.paymentDueDateFrom, 'd.m.Y'),
            });
        }
        if (filterOptions.paymentDueDateTo) {
            var paymentDueDateTo = Ext.Date.add(Ext.Date.parse(filterOptions.paymentDueDateTo, 'd.m.Y'), Ext.Date.DAY, 1);
            filter.push({
                property: 'supplierOrder.paymentDueDate',
                expression: '<',
                value: paymentDueDateTo,
            });
        }
        if (filterOptions.warehouse !== null && filterOptions.warehouse !== undefined) {
            filter.push({
                property: 'supplierOrder.warehouseId',
                value: filterOptions.warehouse,
            });
        }
        if (filterOptions.supplier !== null && filterOptions.supplier !== undefined) {
            filter.push({
                property: 'supplierOrder.supplierId',
                value: filterOptions.supplier,
            });
        }
        if (filterOptions.orderStatus !== null && filterOptions.orderStatus !== undefined) {
            filter.push({
                property: 'supplierOrder.statusId',
                value: filterOptions.orderStatus,
            });
        }
        if (filterOptions.paymentStatus !== null && filterOptions.paymentStatus !== undefined) {
            filter.push({
                property: 'supplierOrder.paymentStatus',
                value: filterOptions.paymentStatus,
            });
        }

        return filter;
    },

    /**
     * Synchronizes the order store and, in case of failure, shows a growl notification.
     *
     * @param string errorMessage
     * @param function callback (optional)
     */
    syncOrders: function (errorMessage, callback) {
        // Check the store for any changes
        var numberOfNewRecords = this.orderStore.getNewRecords().length;
        var numberOfModifiedRecords = this.orderStore.getModifiedRecords().length;
        var numberOfRemovedRecords = this.orderStore.getRemovedRecords().length;
        var numChanges = numberOfNewRecords + numberOfModifiedRecords + numberOfRemovedRecords;
        if (numChanges === 0) {
            if (Ext.isFunction(callback)) {
                callback(true);
            }

            return;
        }

        // Sync the store
        this.setMainViewLoading(true);
        this.orderStore.sync({
            scope: this,
            success: function (batch) {
                // this.orderStore.sync() does not set the dirty flags of the order-items in the store to false. Therefore future calls
                // of rejectChanges() would reset changes that were already saved on the server.
                // Therefore the order-store is reloaded to set the dirty-flags of the order-items to false.
                this.reloadOrderStore(function () {
                    this.setMainViewLoading(false);

                    if (!Ext.isFunction(callback)) {
                        return;
                    }
                    if (!batch.operations) {
                        callback(true, []);
                    }

                    // Find the create operation to retrieve the created orders
                    var createOperation;
                    Ext.each(batch.operations, function (operation) {
                        if (operation.action === 'create') {
                            createOperation = operation;

                            return false;
                        }

                        return undefined;
                    });

                    // Return all created orders
                    var createdOrders = [];
                    if (createOperation && createOperation.resultSet && createOperation.resultSet.records) {
                        createdOrders = createOperation.resultSet.records;
                    }
                    callback(true, createdOrders);
                });
            },
            failure: function () {
                this.setMainViewLoading(false);
                // Revert any changes like added, updated or removed records
                this.orderStore.rejectChanges();
                // Display an error notification
                var message = errorMessage || '';
                Shopware.Notification.createGrowlMessage(
                    this.getViisonSnippet('notification/error/title'),
                    message,
                    'ViisonPickwareERPSupplierOrders'
                );
                if (Ext.isFunction(callback)) {
                    callback(false);
                }
            },
        });
    },

    reloadOrderStore: function (callback) {
        this.orderStore.load({
            scope: this,
            callback: callback,
        });
    },

    /**
     * Sets the main view, in case it exists, (not) loading depending on the given argument.
     *
     * @param boolean loading
     */
    setMainViewLoading: function (loading) {
        if (this.mainView) {
            this.mainView.setLoading(loading);
        }
    },

});
